/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.type;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.group.Group;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.group.GroupManager;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.core.group.GroupHelper;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.SubscriptionException;
import org.ametys.plugins.pagesubscription.SubscriptionException.Type;
import org.ametys.plugins.pagesubscription.SubscriptionFactory;
import org.ametys.plugins.pagesubscription.context.SubscriptionContext;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.site.Site;

/**
 * Abstract class for a type of subscription
 * @param <C> the context for the subscription type
 * @param <T> the targeted element of the subscription
 */
public abstract class AbstractSubscriptionType<C extends SubscriptionContext, T> extends AbstractLogEnabled implements SubscriptionType<C, T>, Configurable, Serviceable
{
    /** The user preferences context for notifications */
    private static final String __NOTIFICATION_USER_PREF_CONTEXT_PREFIX = "/page-subscription/notifications/";
    
    /** The id of user preferences for the last update of notifications */
    private static final String __NOTIFICATION_USER_PREF_LAST_UPDATE = "lastUpdate";
    
    /** Name of the subscription root node */
    private static final String __ROOT_NODE_NAME = "ametys:subscriptions";
    
    /** Name of the plugin holding the subscription */
    private static final String __PLUGIN_NODE_NAME = "subscriptions";
    
    /** Prefix used for naming the subscription node */
    private static final String __SUBSCRIPTION_NAME_PREFIX = "ametys-subscription_";
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The group manager */
    protected GroupManager _groupManager;

    /** The user helper */
    protected UserHelper _userHelper;
    
    /** The user manager */
    private UserManager _userManager;
    
    /** The group helper */
    protected GroupHelper _groupHelper;

    /** The user preferences manager */
    protected UserPreferencesManager _userPrefManager;
    
    /** The json utils */
    protected JSONUtils _jsonUtils;
    
    /** The id of the subscription type */
    private String _id;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _groupManager = (GroupManager) manager.lookup(GroupManager.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _groupHelper = (GroupHelper) manager.lookup(GroupHelper.ROLE);
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
        _userPrefManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _id = configuration.getAttribute("id");
    }
    
    public String getId()
    {
        return _id;
    }
    
    public Subscription subscribe(Site site, UserIdentity user, Frequency frequency, List<BroadcastChannel> broadcastChannels, C context) throws Exception
    {
        boolean alreadySubscribe = getUserSubscriptions(site, user, false, context).stream() // Get all subscription concerning the user for the given context
            .filter(s -> s.getSubscribersGroup().isEmpty() || s.isForced()) // then remove not forced group subscription
            .findAny()
            .isPresent();
        if (alreadySubscribe)
        {
            throw new SubscriptionException("The user " + user + " try to subscribe whereas the subscription already exists", Type.ALREADY_EXIST);
        }
        
        Subscription subscription = _addSubscription(site, this, user, frequency, broadcastChannels, ZonedDateTime.now());
        setAdditionalData(subscription, context);
        subscription.saveChanges();
        return subscription;
    }
    
    public Subscription forceSubscription(Site site, UserIdentity user, Frequency frequency, FrequencyTiming forcedTiming, List<BroadcastChannel> broadcastChannels, C context) throws Exception
    {
        boolean alreadySubscribe = getUserSubscriptions(site, user, true, context).stream() // Get all forced subscription concerning the user for the given context
            .filter(s -> s.getSubscribersGroup().isEmpty()) // then remove group subscription
            .findAny()
            .isPresent();
        if (alreadySubscribe)
        {
            throw new SubscriptionException("The user " + user + " try to subscribe whereas the subscription already exists", Type.ALREADY_EXIST);
        }
        
        Subscription subscription = _forceSubscription(site, this, user, frequency, forcedTiming, broadcastChannels, ZonedDateTime.now());
        setAdditionalData(subscription, context);
        subscription.saveChanges();
        return subscription;
    }
    
    public Subscription subscribe(Site site, GroupIdentity group, Frequency frequency, List<BroadcastChannel> broadcastChannels, C context) throws Exception
    {
        boolean alreadySubscribe = !getGroupsSubscriptions(site, Set.of(group), false, context).isEmpty(); // Get all group subscriptions for the given context
        if (alreadySubscribe)
        {
            throw new SubscriptionException("The group " + group + " try to subscribe whereas the subscription already exists", Type.ALREADY_EXIST);
        }
        
        Subscription subscription = _addSubscription(site, this, group, frequency, broadcastChannels, ZonedDateTime.now());
        setAdditionalData(subscription, context);
        subscription.saveChanges();
        return subscription;
    }
    
    public Subscription forceSubscription(Site site, GroupIdentity group, Frequency frequency, FrequencyTiming forcedTiming, List<BroadcastChannel> broadcastChannels, C context) throws Exception
    {
        boolean alreadySubscribe = !getGroupsSubscriptions(site, Set.of(group), true, context).isEmpty(); // Get all forced group subscriptions for the given context
        if (alreadySubscribe)
        {
            throw new SubscriptionException("The group " + group + " try to subscribe whereas the subscription already exists", Type.ALREADY_EXIST);
        }
        
        Subscription subscription = _forceSubscription(site, this, group, frequency, forcedTiming, broadcastChannels, ZonedDateTime.now());
        setAdditionalData(subscription, context);
        subscription.saveChanges();
        return subscription;
    }
    
    /**
     * Set the additional data to the subscription
     * @param subscription the subscription
     * @param context the context
     */
    protected void setAdditionalData(Subscription subscription, C context)
    {
        //Do nothing by default
    }
    
    public Map<String, Object> subscriptionToJSON(Subscription subscription)
    {
        Map<String, Object> result = new HashMap<>();
     
        result.put("id", subscription.getId());
        result.put(SubscriptionFactory.FREQUENCY, _frequencyToJson(subscription));
        result.put(SubscriptionFactory.BROADCAST_CHANNEL, _channelsToJson(subscription));
        
        if (subscription.isForced())
        {
            result.put(SubscriptionFactory.IS_FORCED, subscription.isForced());
            FrequencyTiming timing = subscription.getForceFrequencyTiming();
            result.put(SubscriptionFactory.FORCED_DAY, timing.day());
            result.put(SubscriptionFactory.FORCED_HOUR, timing.time());
        }
        
        Optional<UserIdentity> subscriber = subscription.getSubscriber();
        if (subscriber.isPresent())
        {
            result.put(SubscriptionFactory.SUBSCRIBER, _userHelper.user2json(subscriber.get()));
        }
        
        Optional<GroupIdentity> subscribersGroup = subscription.getSubscribersGroup();
        if (subscribersGroup.isPresent())
        {
            result.put(SubscriptionFactory.SUBSCRIBERS_GROUP, _groupHelper.group2JSON(subscribersGroup.get(), false));
        }
        
        return result;
    }
    
    /**
     * Get broadcast channels as json
     * @param subscription the subscription
     * @return the broadcast channels as json
     */
    protected List<Map<String, Object>> _channelsToJson(Subscription subscription)
    {
        return subscription.getBroadcastChannels()
                .stream()
                .map(c -> Map.of(
                        "name", c.name(),
                        "label", BroadcastChannelHelper.getLabel(c)
                    )
                )
                .toList();
    }
    
    /**
     * Get frequency as json
     * @param subscription the subscription
     * @return the frequency as json
     */
    protected Map<String, Object> _frequencyToJson(Subscription subscription)
    {
        Frequency frequency = subscription.getFrequency();
        
        Map<String, Object> frequency2json = new HashMap<>();
        
        frequency2json.put("name", frequency.name());
        frequency2json.put("label", FrequencyHelper.getLabel(frequency));
        
        FrequencyTiming timing = subscription.getForceFrequencyTiming();
        frequency2json.put("smartLabel", FrequencyHelper.getSmartLabel(frequency, timing));
        frequency2json.put("fullLabel", FrequencyHelper.getFullLabel(frequency, subscription.getBroadcastChannels(), timing));
        
        return frequency2json;
    }
    
    public void saxSubscription(ContentHandler contentHandler, Subscription subscription) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", subscription.getId());
        if (subscription.isForced())
        {
            attrs.addCDATAAttribute(SubscriptionFactory.IS_FORCED, String.valueOf(subscription.isForced()));
            FrequencyTiming timing = subscription.getForceFrequencyTiming();
            attrs.addCDATAAttribute(SubscriptionFactory.FORCED_DAY, String.valueOf(timing.day()));
            attrs.addCDATAAttribute(SubscriptionFactory.FORCED_HOUR, timing.time());
        }
        
        XMLUtils.startElement(contentHandler, "subscription", attrs);
        
        _saxFrequency(contentHandler, subscription);
        _saxBroadcastChannel(contentHandler, subscription.getBroadcastChannels());
        _saxSubscriber(contentHandler, subscription);
        _saxSubscribersGroup(contentHandler, subscription);
        
        _saxAdditionalData(contentHandler, subscription);
        
        XMLUtils.endElement(contentHandler, "subscription");
    }
    
    /**
     * SAX additional data
     * @param contentHandler the content handler to sax into
     * @param subscription the subscription
     * @throws SAXException if a error occured while saxing
     */
    protected void _saxAdditionalData(ContentHandler contentHandler, Subscription subscription)  throws SAXException
    {
        // Nothing
    }
    
    /**
     * SAX the frequency of a subscription
     * @param contentHandler the content handler to sax into
     * @param subscription the subscription
     * @throws SAXException if a error occured while saxing
     */
    protected void _saxFrequency(ContentHandler contentHandler, Subscription subscription) throws SAXException
    {
        Frequency frequency = subscription.getFrequency();

        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("name", frequency.name());
        
        XMLUtils.startElement(contentHandler, "frequency", attrs);
        
        FrequencyTiming timing = subscription.getForceFrequencyTiming();
        FrequencyHelper.getLabel(frequency).toSAX(contentHandler, "label");
        FrequencyHelper.getSmartLabel(frequency, timing).toSAX(contentHandler, "smartLabel");
        FrequencyHelper.getFullLabel(frequency, subscription.getBroadcastChannels(), timing).toSAX(contentHandler, "fullLabel");
        
        XMLUtils.endElement(contentHandler, "frequency");
    }
    
    /**
     * SAX the channels of a subscription
     * @param contentHandler the content handler to sax into
     * @param channels the channels
     * @throws SAXException if a error occured while saxing
     */
    protected void _saxBroadcastChannel(ContentHandler contentHandler, List<BroadcastChannel> channels) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "channels");
        
        for (BroadcastChannel channel : channels)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("name", channel.name());
            
            XMLUtils.startElement(contentHandler, "channel", attrs);
            BroadcastChannelHelper.getLabel(channel).toSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "channel");
        }
        
        XMLUtils.endElement(contentHandler, "channels");
    }
    
    /**
     * SAX the subscriber of a subscription
     * @param contentHandler the content handler to sax into
     * @param subscription the subscription
     * @throws SAXException if a error occured while saxing
     */
    protected void _saxSubscriber(ContentHandler contentHandler, Subscription subscription) throws SAXException
    {
        Optional<UserIdentity> subscriber = subscription.getSubscriber();
        if (subscriber.isPresent())
        {
            _userHelper.saxUserIdentity(subscriber.get(), contentHandler, SubscriptionFactory.SUBSCRIBER);
        }
    }
    
    /**
     * SAX the subscribers group of a subscription
     * @param contentHandler the content handler to sax into
     * @param subscription the subscription
     * @throws SAXException if a error occured while saxing
     */
    protected void _saxSubscribersGroup(ContentHandler contentHandler, Subscription subscription) throws SAXException
    {
        Optional<GroupIdentity> subscribersGroup = subscription.getSubscribersGroup();
        if (subscribersGroup.isPresent())
        {
            _groupHelper.saxGroupIdentity(subscribersGroup.get(), contentHandler, SubscriptionFactory.SUBSCRIBERS_GROUP);
        }
    }
    
    public void editSubscription(Subscription subscription, Frequency frequency, List<BroadcastChannel> broadcastChannels, C context) throws Exception
    {
        _editSubscription(subscription, frequency, broadcastChannels);
        subscription.saveChanges();
    }
    
    public void editForceSubscription(Subscription subscription, Frequency frequency, List<BroadcastChannel> broadcastChannels, FrequencyTiming forcedTiming, C context) throws Exception
    {
        _editSubscription(subscription, frequency, broadcastChannels);
        subscription.setValue(SubscriptionFactory.IS_FORCED, true);
        subscription.setValue(SubscriptionFactory.FORCED_DAY, forcedTiming.day(), ModelItemTypeConstants.LONG_TYPE_ID);
        subscription.setValue(SubscriptionFactory.FORCED_HOUR, forcedTiming.time(), ModelItemTypeConstants.STRING_TYPE_ID);
        subscription.saveChanges();
    }
    
    private void _editSubscription(Subscription subscription, Frequency frequency, List<BroadcastChannel> broadcastChannels)
    {
        subscription.setValue(SubscriptionFactory.FREQUENCY, frequency.name());
        String[] broadcastChannelNames = broadcastChannels.stream()
                .map(BroadcastChannel::name)
                .toArray(String[]::new);
        subscription.setValue(SubscriptionFactory.BROADCAST_CHANNEL, broadcastChannelNames);
    }
    
    public void unsubscribe(Site site, UserIdentity user, C context) throws Exception
    {
        for (Subscription subscription : getUserSubscriptions(site, user, false, context))
        {
            _removeUsersPreference(subscription);
            subscription.remove();
        }
        site.saveChanges();
    }
    
    public void unsubscribe(Site site, GroupIdentity group, C context) throws Exception
    {
        for (Subscription subscription : getGroupsSubscriptions(site, Set.of(group), false, context))
        {
            unsubscribe(subscription);
        }
    }
    
    public void unsubscribe(Subscription subscription) throws Exception
    {
        Site site = subscription.getSite();
        _removeUsersPreference(subscription);
        subscription.remove();
        site.saveChanges();
    }
    
    public Set<UserIdentity> getSubscribers(Site site, Frequency frequency, BroadcastChannel broadcastChannel, C context)
    {
        return getSubscriptions(site, frequency, broadcastChannel, context)
                .stream()
                .map(this::getSubscribers)
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
    }
    
    public Set<UserIdentity> getSubscribers(Subscription subscription)
    {
        Set<UserIdentity> subscribers = new HashSet<>();
        
        Optional<UserIdentity> subscriber = subscription.getSubscriber();
        Optional<GroupIdentity> subscribersGroup = subscription.getSubscribersGroup();
        if (subscriber.isPresent())
        {
            User user = _userManager.getUser(subscriber.get());
            if (user != null) // Check if the user exist. It can be an old subcription with deleted user
            {
                subscribers.add(subscriber.get());
            }
        }
        else if (subscribersGroup.isPresent())
        {
            Group group = _groupManager.getGroup(subscribersGroup.get());
            if (group != null) // Check if the group exist. It can be an old subcription with deleted group
            {
                subscribers.addAll(group.getUsers());
            }
        }
        
        return subscribers;
    }
    
    /**
     * Get all subscriptions for given parameters
     * @param site the site holding the subscriptions
     * @param frequency the frequency. Can be null.
     * @param broadcastChannel the broadcast channel. Can be null.
     * @param context the context. Can be null.
     * @return the list of subscriptions
     */
    public List<Subscription> getSubscriptions(Site site, Frequency frequency, BroadcastChannel broadcastChannel, C context)
    {
        Expression filterExpr = getSubscriptionsExpressions(frequency, broadcastChannel, null, null, false, false, context);
        String xpathQuery = _getSubscriptionXPathQuery(site.getName(), filterExpr);
        
        AmetysObjectIterable<Subscription> subscriptions = _resolver.query(xpathQuery);
        return subscriptions.stream()
            .filter(this::isSubscriptionValid)
            .toList();
    }
    
    /**
     * Get user subscriptions for given parameters
     * @param site the site holding the subscriptions
     * @param user the user. Can be null.
     * @param onlyForced <code>true</code> to get only forced subscription
     * @param context the context. Can be null.
     * @return the list of subscriptions
     */
    public List<Subscription> getUserSubscriptions(Site site, UserIdentity user, boolean onlyForced, C context)
    {
        return getUserSubscriptions(site, null, null, user, onlyForced, context);
    }
    
    /**
     * Get user subscriptions (including subscriptions from group) matching the given parameters
     * @param site the site holding the subscriptions
     * @param frequency the frequency. Can be null to not filter on frequency.
     * @param broadcastChannel the broadcast channel. Can be null to not filter on channel..
     * @param user the user. Can not be null.
     * @param onlyForced <code>true</code> to get only forced subscription
     * @param context the context. Can be null.
     * @return the list of matching subscriptions
     */
    public List<Subscription> getUserSubscriptions(Site site, Frequency frequency, BroadcastChannel broadcastChannel, UserIdentity user, boolean onlyForced, C context)
    {
        // get user' groups
        Set<GroupIdentity> userGroups = _groupManager.getUserGroups(user);

        // DO NOT FILTER by frequency and channel at this point !
        // Indeed, if for example, we search for all user subscriptions for SITE channel and WEEKLY frequency
        // USER has subscribed to TARGET on SITE channel with WEEKLY frequency
        // but USER is part of GROUP subscribed (forced) to TARGET on MAIL only with DAILY frequency
        // Group subscription overrides the user subscription => user subscription should be ignored
        Expression filterExpr = getSubscriptionsExpressions(null, null, user, userGroups, false, onlyForced, context);
        String xpathQuery = _getSubscriptionXPathQuery(site.getName(), filterExpr);
        
        AmetysObjectIterable<Subscription> userSubscriptions = _resolver.query(xpathQuery);
        
        Map<Object, List<Subscription>> subscriptionsByTarget = new HashMap<>();
        
        // First get subscriptions by group (subscription by group are always forced)
        userSubscriptions.stream()
            .filter(s -> s.getSubscribersGroup().isPresent())
            .filter(this::isSubscriptionValid)
            .forEach(s -> {
                List<Subscription> list = subscriptionsByTarget.getOrDefault(s.getSubscriptionType().getTarget(s), new ArrayList<>());
                list.add(s);
                subscriptionsByTarget.put(s.getSubscriptionType().getTarget(s), list);
            });
        
        // Then add user subscription only if a forced subscription by group does not already exist for the same target or user subscription is forced
        userSubscriptions.stream()
            .filter(s -> s.getSubscriber().isPresent())
            .filter(this::isSubscriptionValid)
            .filter(s -> !subscriptionsByTarget.containsKey(s.getSubscriptionType().getTarget(s)) || s.isForced())
            .forEach(s -> {
                List<Subscription> list = subscriptionsByTarget.getOrDefault(s.getSubscriptionType().getTarget(s), new ArrayList<>());
                list.add(s);
                subscriptionsByTarget.put(s.getSubscriptionType().getTarget(s), list);
            });
        
        // Finally, filter by frequency and channel
        return subscriptionsByTarget.values().stream()
                .flatMap(x -> x.stream())
                .filter(s -> frequency == null || s.getFrequency() == frequency)
                .filter(s -> broadcastChannel == null || s.getBroadcastChannels().contains(broadcastChannel))
                .sorted(Comparator.comparing(Subscription::getDate))
                .toList();
    }
    
    /**
     * Get groups subscriptions for given parameters
     * @param site the site holding the subscriptions
     * @param groups the groups. Can be null.
     * @param onlyForced <code>true</code> to get only forced subscription
     * @param context the context. Can be null.
     * @return the list of subscriptions
     */
    public List<Subscription> getGroupsSubscriptions(Site site, Set<GroupIdentity> groups, boolean onlyForced, C context)
    {
        return getGroupsSubscriptions(site, null, null, groups, onlyForced, context);
    }
    
    /**
     * Get groups subscriptions for given parameters
     * @param site the site holding the subscriptions
     * @param frequency the frequency. Can be null.
     * @param broadcastChannel the broadcast channel. Can be null.
     * @param groups the groups. Can be null.
     * @param onlyForced <code>true</code> to get only forced subscription
     * @param context the context. Can be null.
     * @return the list of subscriptions
     */
    public List<Subscription> getGroupsSubscriptions(Site site, Frequency frequency, BroadcastChannel broadcastChannel, Set<GroupIdentity> groups, boolean onlyForced, C context)
    {
        Expression filterExpr = getSubscriptionsExpressions(frequency, broadcastChannel, null, groups, true, onlyForced, context);
        String xpathQuery = _getSubscriptionXPathQuery(site.getName(), filterExpr);
        
        return _resolver.query(xpathQuery)
                .stream()
                .filter(Subscription.class::isInstance)
                .map(Subscription.class::cast)
                .filter(this::_isGroupExist)
                .filter(this::isSubscriptionValid)
                .toList();
    }
    
    private boolean _isGroupExist(Subscription subscription)
    {
        return subscription.getSubscribersGroup()
            .map(g -> _groupManager.getGroup(g) != null)
            .orElse(false);
    }
    
    /**
     * <code>true</code> if the subscription is valid
     * @param subscription the subscription
     * @return <code>true</code> if the subscription is valid
     */
    protected abstract boolean isSubscriptionValid(Subscription subscription);
    
    /** 
     * Get the filter expressions to search for subscriptions
     * @param frequency the frequency filter. Can be null.
     * @param broadcastChannel the bradcast channel filter. Can be null.
     * @param user the user filter. Can be null.
     * @param userGroups the groups filter. Can be null or empty
     * @param onlyGroup true to retreive only group subscriptions
     * @param onlyForced true to retreive only forced subscriptions
     * @param context the context
     * @return the list of expressions to match subscriptions
     */
    protected Expression getSubscriptionsExpressions(Frequency frequency, BroadcastChannel broadcastChannel, UserIdentity user, Set<GroupIdentity> userGroups, boolean onlyGroup, boolean onlyForced, C context)
    {
        List<Expression> filterExprs = new ArrayList<>();
        
        filterExprs.add(new StringExpression(SubscriptionFactory.SUBSCRIPTION_TYPE_ID, Operator.EQ, _id));
        
        if (frequency != null)
        {
            filterExprs.add(new StringExpression(SubscriptionFactory.FREQUENCY, Operator.EQ, frequency.name()));
        }
        
        if (broadcastChannel != null)
        {
            filterExprs.add(new StringExpression(SubscriptionFactory.BROADCAST_CHANNEL, Operator.EQ, broadcastChannel.name()));
        }
        
        if (onlyGroup)
        {
            filterExprs.add(new MetadataExpression(SubscriptionFactory.SUBSCRIBERS_GROUP));
            
            if (userGroups != null && !userGroups.isEmpty())
            {
                StringExpression[] groupsExpr = userGroups.stream()
                    .map(GroupIdentity::groupIdentityToString)
                    .map(id -> (new StringExpression(SubscriptionFactory.SUBSCRIBERS_GROUP, Operator.EQ, id)))
                    .toArray(StringExpression[]::new);
                
                filterExprs.add(new OrExpression(groupsExpr));
            }
        }
        else if (user != null || userGroups != null && !userGroups.isEmpty())
        {
            List<Expression> userExprs = new ArrayList<>();
            if (user != null)
            {
                userExprs.add(new UserExpression(SubscriptionFactory.SUBSCRIBER, Operator.EQ, user));
            }
            
            if (userGroups != null && !userGroups.isEmpty())
            {
               userGroups.stream()
                    .map(GroupIdentity::groupIdentityToString)
                    .forEach(id -> userExprs.add(new StringExpression(SubscriptionFactory.SUBSCRIBERS_GROUP, Operator.EQ, id)));
            }
            
            filterExprs.add(new OrExpression(userExprs));
        }
        
        if (onlyGroup)
        {
            filterExprs.add(new MetadataExpression(SubscriptionFactory.SUBSCRIBERS_GROUP));
        }
        
        if (onlyForced)
        {
            filterExprs.add(new BooleanExpression(SubscriptionFactory.IS_FORCED, true));
        }
        
        if (context != null)
        {
            Expression additionalExpr = getAdditionalFilterExpression(context);
            if (additionalExpr != null)
            {
                filterExprs.add(additionalExpr);
            }
        }
        
        return new AndExpression(filterExprs.toArray(new Expression[filterExprs.size()]));
    }
    
    /**
     * Creates the XPath query corresponding to specified {@link Expression}.
     * The query will include to sort the result by subscription date.
     * If no expression is provided, all the subscription nodes will be returned
     * @param filterExpression the query predicates. Can be null.
     * @return the created XPath query.
     */
    private String _getSubscriptionXPathQuery(String siteName, Expression filterExpression)
    {
        String predicats = null;
        
        if (filterExpression != null)
        {
            predicats = StringUtils.trimToNull(filterExpression.build());
        }
        
        StringBuilder sb = new StringBuilder();
        sb.append("//element(" + siteName + ", ametys:site)");
        sb.append("//element(*, " + SubscriptionFactory.NODE_TYPE + ")");
        if (predicats != null)
        {
            sb.append("[").append(predicats).append("]");
        }
        SortCriteria sort = new SortCriteria();
        sort.addCriterion(SubscriptionFactory.DATE, false, false);
        
        sb.append(" ").append(sort.build());
        return sb.toString();
    }
    
    /**
     * Get additional query filter from context to search for subscriptions
     * @param context the context
     * @return the additional query filter
     */
    protected abstract Expression getAdditionalFilterExpression(C context);
    
    public ZonedDateTime getLastReadDate(Subscription subscription, UserIdentity user)
    {
        try
        {
            String context = __NOTIFICATION_USER_PREF_CONTEXT_PREFIX + subscription.getSite().getName();
            Map<String, Object> userPreferences = _getUserPreferences(context, user);
            
            String userPreferenceContextId = getUserPreferenceContextId(subscription);
            if (userPreferences.containsKey(userPreferenceContextId))
            {
                return DateUtils.parseZonedDateTime((String) userPreferences.get(userPreferenceContextId));
            }
        }
        catch (UserPreferencesException e)
        {
            getLogger().warn("Unable to get last unread notifications date from user preferences", e);
        }
        return null;
    }
    
    public void markAsRead(Subscription subscription, UserIdentity user)
    {
        try
        {
            String context = __NOTIFICATION_USER_PREF_CONTEXT_PREFIX + subscription.getSite().getName();
            Map<String, Object> userPreferences = _getUserPreferences(context, user);
            userPreferences.put(getUserPreferenceContextId(subscription), DateUtils.zonedDateTimeToString(ZonedDateTime.now()));
            
            _userPrefManager.setUserPreferences(user, context, Map.of(), Map.of(__NOTIFICATION_USER_PREF_LAST_UPDATE, _jsonUtils.convertObjectToJson(userPreferences)));
        }
        catch (UserPreferencesException e)
        {
            getLogger().warn("Unable to set last unread notifications date from user preferences", e);
        }
    }
    
    private void _removeUsersPreference(Subscription subscription)
    {
        for (UserIdentity subscriber : getSubscribers(subscription))
        {
            try
            {
                String context = __NOTIFICATION_USER_PREF_CONTEXT_PREFIX + subscription.getSite().getName();
                Map<String, Object> userPreferences = _getUserPreferences(context, subscriber);
                userPreferences.remove(getUserPreferenceContextId(subscription));
                
                _userPrefManager.setUserPreferences(subscriber, context, Map.of(), Map.of(__NOTIFICATION_USER_PREF_LAST_UPDATE, _jsonUtils.convertObjectToJson(userPreferences)));
            }
            catch (UserPreferencesException e)
            {
                getLogger().warn("Unable to remove user preferences", e);
            }
        }
    }
    
    private Map<String, Object> _getUserPreferences(String context, UserIdentity user) throws UserPreferencesException
    {
        String userPreferencesAsString = _userPrefManager.getUserPreferenceAsString(user, context, Map.of(), __NOTIFICATION_USER_PREF_LAST_UPDATE);
        return StringUtils.isNotBlank(userPreferencesAsString)
                ? _jsonUtils.convertJsonToMap(userPreferencesAsString)
                : new HashMap<>();
    }
    /**
     * Get the user preference context id
     * @param subscription the subscription
     * @return the context id
     */
    protected abstract String getUserPreferenceContextId(Subscription subscription);
    
    /**
     * Get the root plugin storage object.
     * @param site the site
     * @return the root plugin storage object.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    protected ModifiableTraversableAmetysObject _getSiteRootNode(Site site) throws AmetysRepositoryException
    {
        try
        {
            return _getOrCreateRootNode(site);
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the susbcriptions root node", e);
        }
    }
    
    private ModifiableTraversableAmetysObject _getOrCreateRootNode(Site site) throws AmetysRepositoryException
    {
        ModifiableTraversableAmetysObject pluginsNode = site.getRootPlugins();
        
        ModifiableTraversableAmetysObject pluginNode = (ModifiableTraversableAmetysObject) _getOrCreateNode(pluginsNode, __PLUGIN_NODE_NAME, "ametys:unstructured");
        
        return (ModifiableTraversableAmetysObject) _getOrCreateNode(pluginNode, __ROOT_NODE_NAME, "ametys:collection");
    }
    
    private AmetysObject _getOrCreateNode(ModifiableTraversableAmetysObject parentNode, String nodeName, String nodeType) throws AmetysRepositoryException
    {
        AmetysObject definitionsNode;
        if (parentNode.hasChild(nodeName))
        {
            definitionsNode = parentNode.getChild(nodeName);
        }
        else
        {
            definitionsNode = parentNode.createChild(nodeName, nodeType);
            parentNode.saveChanges();
        }
        return definitionsNode;
    }
    
    /**
     * Add a user subscription to the given site
     * @param site the given site
     * @param type the type of subscription
     * @param user the user who subscribe
     * @param frequency the frequency of the subscription
     * @param broadcastChannels the list of broadcast channel of the subscription
     * @param date the subscription date
     * @return the created subscription
     */
    protected Subscription _addSubscription(Site site, SubscriptionType type, UserIdentity user, Frequency frequency, List<BroadcastChannel> broadcastChannels, ZonedDateTime date)
    {
        ModifiableTraversableAmetysObject siteRootNode = _getSiteRootNode(site);

        Subscription subscription = _addSubscription(siteRootNode, type, frequency, broadcastChannels, date);
        subscription.setValue(SubscriptionFactory.SUBSCRIBER, user, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID);
        siteRootNode.saveChanges();
        
        return subscription;
    }
    
    /**
     * Force a user subscription to the given site
     * @param site the given site
     * @param type the type of subscription
     * @param user the user who subscribe
     * @param frequency the frequency of the subscription
     * @param forcedTiming the force frequency timing
     * @param broadcastChannels the list of broadcast channel of the subscription
     * @param date the subscription date
     * @return the created subscription
     */
    protected Subscription _forceSubscription(Site site, SubscriptionType type, UserIdentity user, Frequency frequency, FrequencyTiming forcedTiming, List<BroadcastChannel> broadcastChannels, ZonedDateTime date)
    {
        Subscription subscription = _addSubscription(site, type, user, frequency, broadcastChannels, date);
        _forceSubscription(subscription, forcedTiming);
        
        return subscription;
    }
    
    /**
     * Add a group subscription to the given site
     * @param site the given site
     * @param type the type of subscription
     * @param group the group who subscribe
     * @param frequency the frequency of the subscription
     * @param broadcastChannels the list of broadcast channel of the subscription
     * @param date the subscription date
     * @return the created subscription
     */
    protected Subscription _addSubscription(Site site, SubscriptionType type, GroupIdentity group, Frequency frequency, List<BroadcastChannel> broadcastChannels, ZonedDateTime date)
    {
        ModifiableTraversableAmetysObject siteRootNode = _getSiteRootNode(site);

        Subscription subscription = _addSubscription(siteRootNode, type, frequency, broadcastChannels, date);
        subscription.setValue(SubscriptionFactory.SUBSCRIBERS_GROUP, GroupIdentity.groupIdentityToString(group), ModelItemTypeConstants.STRING_TYPE_ID);
        siteRootNode.saveChanges();
        
        return subscription;
    }
    
    /**
     * Add a group subscription to the given site
     * @param site the given site
     * @param type the type of subscription
     * @param group the group who subscribe
     * @param frequency the frequency of the subscription
     * @param forcedTiming the force frequency timing
     * @param broadcastChannels the list of broadcast channel of the subscription
     * @param date the subscription date
     * @return the created subscription
     */
    protected Subscription _forceSubscription(Site site, SubscriptionType type, GroupIdentity group, Frequency frequency, FrequencyTiming forcedTiming, List<BroadcastChannel> broadcastChannels, ZonedDateTime date)
    {
        Subscription subscription = _addSubscription(site, type, group, frequency, broadcastChannels, date);
        _forceSubscription(subscription, forcedTiming);
        
        return subscription;
    }
    
    private Subscription _addSubscription(ModifiableTraversableAmetysObject siteRootNode, SubscriptionType type, Frequency frequency, List<BroadcastChannel> broadcastChannels, ZonedDateTime date)
    {
        String subscriptionName = __SUBSCRIPTION_NAME_PREFIX + UUID.randomUUID().toString();
        Subscription subscription = siteRootNode.createChild(subscriptionName, SubscriptionFactory.NODE_TYPE);
        
        subscription.setValue(SubscriptionFactory.SUBSCRIPTION_TYPE_ID, type.getId(), ModelItemTypeConstants.STRING_TYPE_ID);
        subscription.setValue(SubscriptionFactory.DATE, date, ModelItemTypeConstants.DATETIME_TYPE_ID);
        subscription.setValue(SubscriptionFactory.FREQUENCY, frequency.name(), ModelItemTypeConstants.STRING_TYPE_ID);
        String[] broadcastChannelNames = broadcastChannels.stream()
                .map(BroadcastChannel::name)
                .toArray(String[]::new);
        subscription.setValue(SubscriptionFactory.BROADCAST_CHANNEL, broadcastChannelNames, ModelItemTypeConstants.STRING_TYPE_ID);
        
        return subscription;
    }
    
    private void _forceSubscription(Subscription subscription, FrequencyTiming forcedTiming)
    {
        subscription.setValue(SubscriptionFactory.IS_FORCED, true, ModelItemTypeConstants.BOOLEAN_TYPE_ID);
        subscription.setValue(SubscriptionFactory.FORCED_DAY, forcedTiming.day(), ModelItemTypeConstants.LONG_TYPE_ID);
        subscription.setValue(SubscriptionFactory.FORCED_HOUR, forcedTiming.time(), ModelItemTypeConstants.STRING_TYPE_ID);
        subscription.saveChanges();
    }
}
