/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.schedulable.tag;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.quartz.JobKey;
import org.quartz.SchedulerException;

import org.ametys.cms.tag.CMSTag;
import org.ametys.core.group.Group;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.group.GroupManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.pagesubscription.type.TagSubscriptionType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.site.SiteManager;

/**
 * Helper to create runnable to send notification mail link to forced tag subscriptions
 */
public class ForcedSubscriptionSchedulerHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The avalon role */
    public static final String ROLE = ForcedSubscriptionSchedulerHelper.class.getName();
    
    /** The scheduler */
    protected Scheduler _scheduler;
    
    /** The subscription type EP */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The tag subscription type */
    protected TagSubscriptionType _tagSubscriptionType;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The group manager */
    protected GroupManager _groupManager;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _scheduler = (Scheduler) smanager.lookup(Scheduler.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) smanager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _tagSubscriptionType = (TagSubscriptionType) _subscriptionTypeEP.getExtension(TagSubscriptionType.ID);
        _groupManager = (GroupManager) smanager.lookup(GroupManager.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    /**
     * Schedule a new runnable if not exist for the forced group tag subscription
     * @param subscription the group tag subscription
     * @throws SchedulerException if an error occcurred
     */
    public void scheduleForcedSubscription(Subscription subscription) throws SchedulerException
    {
        String id = _getRunnableId(subscription);

        JobKey jobKey = new JobKey(id, Scheduler.JOB_GROUP);
        if (!_scheduler.getScheduler().checkExists(jobKey))
        {
            I18nizableText label = _getLabel(subscription);
            SendGroupTagNotificationSummaryRunnable subscriptionRunable = new SendGroupTagNotificationSummaryRunnable(
                subscription,    
                id,
                label, 
                label
            );
            _scheduler.scheduleJob(subscriptionRunable);
        }
    }
    
    private I18nizableText _getLabel(Subscription subscription)
    {
        CMSTag tag = _tagSubscriptionType.getTarget(subscription);
        
        GroupIdentity groupIdentity = subscription.getSubscribersGroup().get(); // Currently, we create runnable only for group subscriptions
        Group group = _groupManager.getGroup(groupIdentity);
        
        Map<String, I18nizableTextParameter> params = new HashMap<>();
        params.put("tag", tag.getTitle());
        params.put("group", new I18nizableText(group.getLabel()));
        params.put("groupId", new I18nizableText(group.getLabel()));
        
        List<String> parameters = new ArrayList<>();
        parameters.add(_i18nUtils.translate(tag.getTitle())); // Translate because the map of i18nizable parameters doesn't work for runnable renderer ...
        parameters.add(group.getLabel());
        parameters.add(GroupIdentity.groupIdentityToString(groupIdentity));
        return new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_TAG_" + subscription.getFrequency().name() + "_NOTIFICATIONS_SCHEDULER_FORCED_GROUP_LABEL", parameters);
    }
    
    private String _getRunnableId(Subscription subscription)
    {
        return "runnable-" + subscription.getId();
    }

    /**
     * Unschedule the runnable for the forced group tag subscription
     * @param subscription the group tag subscription
     * @throws SchedulerException if an error occurred
     */
    public void unscheduleForcedSubscription(Subscription subscription) throws SchedulerException
    {
        String id = _getRunnableId(subscription);
        
        JobKey jobKey = new JobKey(id, Scheduler.JOB_GROUP);
        if (_scheduler.getScheduler().checkExists(jobKey))
        {
            _scheduler.getScheduler().deleteJob(jobKey);
        }
    }
}
