/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.pagesubscription.schedulable.tag;

import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.quartz.SchedulerException;

import org.ametys.core.engine.BackgroundEngineHelper;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.pagesubscription.type.TagSubscriptionType;
import org.ametys.runtime.plugin.Init;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Init the scheduler for the forced group tag subscriptions
 */
public class InitSchedulerForForcedSubscription implements Init, Serviceable, Contextualizable, LogEnabled
{
    /** The subscription type EP */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The tag subscription type */
    protected TagSubscriptionType _tagSubscriptionType;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The group tag runnable helper */
    protected ForcedSubscriptionSchedulerHelper _groupTagRunnableHelper;

    /** The manager */
    protected ServiceManager _manager;
    
    /** The context */
    protected Context _context;
    
    /** The cocoon context */
    protected org.apache.cocoon.environment.Context _cocoonContext;
    
    /** The logger */
    protected Logger _logger;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) manager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _tagSubscriptionType = (TagSubscriptionType) _subscriptionTypeEP.getExtension(TagSubscriptionType.ID);
        _groupTagRunnableHelper = (ForcedSubscriptionSchedulerHelper) manager.lookup(ForcedSubscriptionSchedulerHelper.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (org.apache.cocoon.environment.Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    public void enableLogging(Logger logger)
    {
        _logger = logger;
    }
    
    @Override
    public void init() throws Exception
    {
        // Create false request because some methods need it as getTag or getGroup
        Map<String, Object> environment = BackgroundEngineHelper.createAndEnterEngineEnvironment(_manager, _cocoonContext, _logger);
        try
        {
            for (Site site : _siteManager.getSites())
            {
                Request request = ContextHelper.getRequest(_context);
                request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, site.getName());
                
                _tagSubscriptionType.getGroupsSubscriptions(site, null, BroadcastChannel.MAIL, Set.of(), true, null).stream()
                    .filter(s -> s.getFrequency() != Frequency.INSTANT)
                    .forEach(s -> {
                        try
                        {
                            _groupTagRunnableHelper.scheduleForcedSubscription(s);
                        }
                        catch (SchedulerException e)
                        {
                            _logger.error("An error occurred creating runnable for subscription with id '" +  s.getId() + "'", e);
                        }
                    });
            }
        }
        finally 
        {
            BackgroundEngineHelper.leaveEngineEnvironment(environment);
        }
        
    }
}
