/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.proxiedcontent;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Session;
import org.apache.cocoon.environment.SourceResolver;
import org.jasig.cas.client.proxy.Cas20ProxyRetriever;
import org.jasig.cas.client.util.AbstractCasFilter;
import org.jasig.cas.client.validation.Assertion;

import org.ametys.core.authentication.CredentialProvider;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.extrausermgt.authentication.cas.CASCredentialProvider;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Get the page url to integrate
 */
public class GetUrlAction extends ServiceableAction
{
    /** The DAO for user populations */
    protected UserPopulationDAO _userPopulationDAO;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(manager);
        _userPopulationDAO = (UserPopulationDAO) smanager.lookup(UserPopulationDAO.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String url = request.getParameter("url");
        String server = request.getParameter("server");
        String isForm = request.getParameter("isForm");
        String baseUrl = Utils.normalizeUrl(source);
        String baseHost = Utils.getRemoteHostFromUrl(baseUrl);
        String remoteHost;
        String completeUrl = "";
        
        if (url == null)
        {
            url = source;
        }
        if (server == null)
        {
            server = source;
        }
        
        url = Utils.normalizeUrl(url);
        server = Utils.normalizeUrl(server);
        
        remoteHost = Utils.getRemoteHostFromUrl(url);
        
        if (baseHost.equals(remoteHost))
        {
            if (isForm != null)
            {
                result.put("queryString", "?" + request.getQueryString());
            }
            
            completeUrl = url.substring(0, url.lastIndexOf("/") + 1);
            
            result.put("url", url);
            result.put("server", server);
            result.put("remote-server", remoteHost);
            result.put("complete-url", completeUrl);
            ZoneItem zoneItem = (ZoneItem) request.getAttribute(ZoneItem.class.getName());
            result.put("zoneitemid", zoneItem.getId());
            
            if (zoneItem.getServiceParameters().getValue("cas", false, false))
            {
                _addCasProxyTicketInUrl(request, url, result);
            }
            
            return result;
        }
        else
        {
            getLogger().error("The specified page '" + url + "' is not on the same host than the base page, and therefore could not be proxified.");
            return null;
        }
    }
    
    private void _addCasProxyTicketInUrl(Request request, String url, Map<String, String> result)
    {
        String proxyTicket = null;
        
        // Case BO: try to get a proxy ticket from the assertion in the current session
        Session session = request.getSession(false);
        if (session != null)
        {
            Assertion assertion = (Assertion) session.getAttribute(AbstractCasFilter.CONST_CAS_ASSERTION);
            if (assertion != null)
            {
                proxyTicket = assertion.getPrincipal().getProxyTicketFor(url);
            }
        }
        
        // Case FO request
        if ("true".equals(request.getHeader("X-Ametys-FO")) && request.getHeader("X-Ametys-FO-PGT") != null)
        {
            String populationId = request.getHeader("X-Ametys-FO-Population");
            UserPopulation population = _userPopulationDAO.getUserPopulation(populationId);
           
            if (population != null)
            {
                String cpId = request.getHeader("X-Ametys-FO-Credential-Provider");
                CredentialProvider credentialProvider = population.getCredentialProvider(cpId);
                if (credentialProvider instanceof CASCredentialProvider)
                {
                    String casUrl = (String) credentialProvider.getParameterValues().get(CASCredentialProvider.PARAM_SERVER_URL);
                    String proxyGrantingTicket = request.getHeader("X-Ametys-FO-PGT");
                    proxyTicket = new Cas20ProxyRetriever(casUrl, "UTF-8", null).getProxyTicketIdFor(proxyGrantingTicket, url);
                }
            }
        }
        
        // If proxy ticket was found, change url and append the ticket
        if (proxyTicket != null)
        {
            StringBuilder urlWithTicket = new StringBuilder(url);
            urlWithTicket.append(!url.contains("?") ? "?" : "&")
                .append("ticket=")
                .append(URIUtils.encodeParameter(proxyTicket));
            result.put("url", urlWithTicket.toString());
        }
        else
        {
            getLogger().warn(String.format("The application was unable to retrieve a proxy ticket from CAS for target service '%s'", url));
        }
    }
}
