/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The queries DAO
 * @private
 */
Ext.define('Ametys.plugins.queriesdirectory.QueriesDAO', {
	singleton: true,
    
    constructor: function(config)
    {
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method getQueriesProperties 
         * Gets queries information.
         * This calls the method 'getQueriesProperties' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String[]} parameters.ids The id of queries to retrieve
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object[]} callback.queries The queries' properties. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object[]} callback.notAllowedQueries The unauthorized queries
         * @param {String[]} callback.unknownQueries The unknown queries
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "getQueriesProperties",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n DAOS_QUERY_REQUEST_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method getIdsOfPath 
         * Gets the ids of the path elements of a query or query container, i.e. the parent ids.
         * For instance, if the query path is 'a/b/c', then the result list will be ["id-of-a", "id-of-b", "id-of-c"]
         * This calls the method 'getIdsOfPath' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.queryId The id of the query
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {String[]} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "getIdsOfPath",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n DAOS_QUERY_REQUEST_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method getQueryContainerProperties 
         * Get the query container properties
         * This calls the method 'getQueryContainerProperties' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The query container id
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {String[]} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "getQueryContainerProperties",
                callback: {
                    handler: Ext.emptyFn
                },
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n DAOS_QUERY_REQUEST_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method createQuery 
         * Creates a new query
         * This calls the method 'createQuery' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.title The title of the query
         * @param {String} parameters.desc The description of the query
         * @param {String} parameters.documentation The query documentation link
         * @param {String} parameters.type The type of the query
         * @param {String} parameters.content The content of the query
         * @param {String} parameters.parentId The id of the parent of the query
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "createQuery",
                callback: {
                    handler: this._createQueryCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_QUERIESDIRECTORY_CREATION_FAILED}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method createQueryContainer 
         * Creates a new query container
         * This calls the method 'createQueryContainer' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.parentId The id of the parent
         * @param {String} parameters.name The desired name for the new query container
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "createQueryContainer",
                callback: {
                    handler: this._createQueryContainerCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_QUERIESDIRECTORY_CONTAINER_CREATION_FAILED}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method updateQuery 
         * Edits a query
         * This calls the method 'updateQuery' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the query
         * @param {String} parameters.title The title of the query
         * @param {String} parameters.desc The description of the query
         * @param {String} parameters.documentation The query documentation link
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "updateQuery",
                callback: {
                    handler: this._updateQueryCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_QUERIESDIRECTORY_MODIFICATION_FAILED}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method renameQueryContainer 
         * Renames a query container
         * This calls the method 'renameQueryContainer' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the query container
         * @param {String} parameters.newName The new name of the container
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "renameQueryContainer",
                callback: {
                    handler: this._renameQueryContainerCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_QUERIESDIRECTORY_RENAMING_FAILED}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method moveQuery 
         * Moves a query
         * This calls the method 'moveQuery' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the query
         * @param {String} parameters.newParentId The id of the new parent container of the query
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "moveQuery",
                callback: {
                    handler: this._moveQueryCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_QUERIESDIRECTORY_MOVE_FAILED}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method moveQueryContainer 
         * Moves a query container
         * This calls the method 'moveQueryContainer' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the query container
         * @param {String} parameters.newParentId The id of the new parent container of the query container
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "moveQueryContainer",
                callback: {
                    handler: this._moveQueryContainerCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_QUERIESDIRECTORY_MOVE_FAILED}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method saveQuery 
         * Saves a query
         * This calls the method 'saveQuery' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the query
         * @param {String} parameters.type The type of the query
         * @param {String} parameters.content The content of the query
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "saveQuery",
                callback: {
                    handler: this._saveQueryCb
                },
                errorMessage: {
                    msg: "{{i18n DAOS_QUERY_SAVEAS_ERROR}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method deleteQuery 
         * Deletes queries
         * This calls the method 'deleteQuery' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.ids The ids of the queries to delete
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "deleteQuery",
                callback: {
                    handler: this._deleteQueryCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_QUERIESDIRECTORY_DELETE_QUERY_ERROR}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method mustWarnBeforeDeletion 
         * Determines if application must warn before deleting the given query containers
         * This calls the method 'mustWarnBeforeDeletion' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.ids The ids of the query containers
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Boolean} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "mustWarnBeforeDeletion",
                callback: {
                    handler: Ext.emptyFn
                }
            }
        );
                
        /**
         * @callable
         * @member Ametys.plugins.queriesdirectory.QueriesDAO
         * @method deleteQueryContainer 
         * Deletes query containers
         * This calls the method 'deleteQueryContainer' of the server DAO 'QueryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.ids The ids of the query containers to delete
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.queriesdirectory.QueryDAO",
                methodName: "deleteQueryContainer",
                callback: {
                    handler: this._deleteContainerCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_QUERIESDIRECTORY_DELETE_CONTAINER_ERROR}}",
                    category:Ext.getClassName(this)
                }
            }
        );
    },
    
    /**
     * Callback function called after {@link #createQuery} has been processed.
     * @param {Object} response The server response
     * @param {String} response.id The query id
     * @private
     */
    _createQueryCb: function(response)
    {
        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {  
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.CREATED,
                targets: {
                    id: Ametys.message.MessageTarget.QUERY,
                    parameters: {
                        id: response.id
                    }
                }
            });
        }
    },
    
    /**
     * Callback function called after {@link #createQueryContainer} has been processed.
     * @param {Object} response The server response
     * @param {String} response.id The query container id
     * @param {String} response.name The query container name
     * @private
     */
    _createQueryContainerCb: function(response)
    {
        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        { 
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.CREATED,
                targets: {
                    id: Ametys.message.MessageTarget.QUERY_CONTAINER,
                    parameters: {
                        id: response.id,
                        canRead: response.canRead,
                        canRename: response.canRename,
                        canWrite: response.canWrite,
                        canDelete: response.canDelete,
                        canAssignRights: response.canAssignRights,
                        fullPath: response.fullPath,
                        name: response.name
                    }
                }
            });
        }
    },
	
    /**
     * Callback function called after {@link #updateQuery} has been processed.
     * @param {Object} response The server response
     * @param {Object} args The callback arguments
     * @param {Array} params The server call parameters
     * @private
     */
    _updateQueryCb: function(response, args, params)
    {
        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {   
            var queryId = response.id;
            this.localUpdate(queryId, {
                title: params[1],
                description: params[2],
                documentation: params[3]
            });
                    
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MODIFIED,
                targets: {
                    id: Ametys.message.MessageTarget.QUERY,
                    parameters: {
                        id: queryId
                    }
                }
            });
        }
    },
    
    /**
     * Callback function called after {@link #renameQueryContainer} has been processed.
     * @param {Object} response The server response
     * @private
     */
    _renameQueryContainerCb: function(response)
    {
        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else if (response.message == 'cannot-rename')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_CANNOT_RENAME_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_CANNOT_RENAME_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.WARNING
            });
        }
        else
        {                       
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MODIFIED,
                targets: {
                    id: Ametys.message.MessageTarget.QUERY_CONTAINER,
                    parameters: {
                        id: response.id,
                        canRead: response.canRead,
                        canRename: response.canRename,
                        canWrite: response.canWrite,
                        canDelete: response.canDelete,
                        canAssignRights: response.canAssignRights,
                        fullPath: response.fullPath,
                        name: response.name
                    }
                }
            });
        }
    },

    /**
     * Callback function called after {@link #moveQuery} has been processed.
     * @param {Object} response The server response
     * @private
     */
    _moveQueryCb: function(response)
    {
        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else if (response.message == 'cannot-move')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_CANNOT_MOVE_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_CANNOT_MOVE_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.WARNING
            });
        }
        else
        {   
            var queryId = response.id;

            // Remove this query in the cache
            this.localRemove(response.id);
            
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MOVED,
                targets: {
                    id: Ametys.message.MessageTarget.QUERY,
                    parameters: {
                        ids: [queryId]
                    }
                }
            });
        }
    },

    /**
     * Callback function called after {@link #moveQueryContainer} has been processed.
     * @param {Object} response The server response
     * @private
     */
    _moveQueryContainerCb: function(response)
    {

        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_NOT_ALLOWED_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else if (response.message == 'cannot-move')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_CANNOT_MOVE_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_CANNOT_MOVE_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.WARNING
            });
        }
        else
        {   
            var queryContainerId = response.id,
                queryContainerName = response.name;
                    
            this.invalidateCache();
            
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MOVED,
                targets: {
                    id: Ametys.message.MessageTarget.QUERY_CONTAINER,
                    parameters: {
                        id: response.id,
                        canWrite: response.canWrite,
                        canRename: response.canRename,
                        canDelete: response.canDelete,
                        canAssignRights: response.canAssignRights,
                        fullPath: response.fullPath,
                        name: response.name
                    }
                }
            });
        }
    },
    
    /**
     * Callback function called after {@link #saveQuery} has been processed.
     * @param {Object} response The server response
     * @private
     */
    _saveQueryCb: function(response)
    {
        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_UPDATE_QUERY_NOT_ALLOWED_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_UPDATE_QUERY_NOT_ALLOWED_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {
            var id = response.id;
            
            // Remove this query in the cache
            this.localRemove(id);
            
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MODIFIED,
                targets: {
                    id: Ametys.message.MessageTarget.QUERY,
                    parameters: { 
                        id: id
                    }
                }
            });  
        }
    },
    
    /**
     * Callback function called after {@link #deleteQuery} has been processed.
     * @param {Object} response The server response
     * @private
     */
    _deleteQueryCb: function(response)
    {

        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_DELETE_CONTAINER_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_DELETE_CONTAINER_NOT_ALLOWED}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.DELETED,
                targets: {
                    id: Ametys.message.MessageTarget.QUERY,
                    parameters: {
                        ids: response.deletedQueries
                    }
                }
            });
            
            var notallowedQueries = response.notallowedQueries;
            if (notallowedQueries.length > 0)
            {
                Ametys.Msg.show({
                    title: "{{i18n PLUGINS_QUERIESDIRECTORY_DELETE_QUERY_TITLE}}",
                    msg: "{{i18n PLUGINS_QUERIESDIRECTORY_DELETE_QUERY_NOT_ALLOWED}}" + notallowedQueries.join(','),
                    buttons: Ext.Msg.OK,
                    icon: Ext.Msg.ERROR
                });
            }
        }
    },
	
    /**
     * Callback function called after {@link #deleteQueryContainer} has been processed.
     * @param {Object} response The server response
     * @private
     */
    _deleteContainerCb: function(response)
    {
        var deletedContainers = response.deletedQueryContainers;
        if (deletedContainers.length > 0)
        {
            var targets = deletedContainers.map(function(queryCt) {
                    return {
                        id: Ametys.message.MessageTarget.QUERY_CONTAINER,
                        parameters: queryCt
                    };
                });
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.DELETED,
                targets: targets
            });
        }
            
        var notAllowedContainers = response.notallowedQueryContainers;
        if (notAllowedContainers.length > 0)
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_QUERIESDIRECTORY_DELETE_CONTAINER_TITLE}}",
                msg: "{{i18n PLUGINS_QUERIESDIRECTORY_DELETE_CONTAINER_NOT_ALLOWED}}" + notAllowedContainers.join(','),
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
    },
	    
    /**
     * Get a query by its id
     * @param {String} id The id of the query to retrieve
     * @param {Function} callback The callback function :
     * @param {Ametys.plugins.queriesdirectory.Query} callback.query The query or null if not found
     */
    getQuery: function (id, callback)
    {
        if (Ext.isEmpty(id))
        {
            callback(null);
            return;
        }
        this._getQueries ([id], Ext.bind(this._getQueryCb, this, [callback], 1));
    },
    
    /**
     * @private
     * Callback function after getting a query properties
     * @param {Ametys.plugins.queriesdirectory.Query[]} queries The queries
     * @param {Function} callback The callback function
     */
    _getQueryCb: function (queries, callback)
    {
        callback((queries.length == 0) ? null : queries[0]);
    },
    
    /**
     * Get queries by their ids
     * @param {String[]} ids The id of the queries to retrieve
     * @param {Function} callback The callback function :
     * @param {Ametys.plugins.queriesdirectory.Query} callback.queries The queries or empty if not found
     */
    getQueries: function (ids, callback)
    {
        if (Ext.isEmpty(ids))
        {
            callback([]);
            return;
        }
        this._getQueries (ids, Ext.bind(this._getQueriesCb, this, [callback], 1));
    },
    
    /**
     * @private
     * Callback function after getting queries' properties
     * @param {Ametys.plugins.queriesdirectory.Query[]} queries The queries
     * @param {Function} callback The callback function
     */
    _getQueriesCb: function (queries, callback)
    {
        callback(queries);
    },
    
	/**
	 * Retrieve queries from the cache, or by doing a server request.
	 * @param {String/String[]} queryIds Query identifiers
	 * @param {Function} callback The callback function called after retrieving the queries. Has the following parameters: 
	 * @param {Object[]} callback.queryData An array of retrieved query data. Can be empty.
	 * @param {String[]} callback.unknownQueries Array of unknown query ids among the requested list.
	 * @param {Object} [scope=window] The callback scope, default to window. Can be null.
	 * @param {Boolean} [displayErrors=false] Set to true to display error dialog box if unknown queries are reported.
	 * @private
	 */
	_getQueries: function(queryIds, callback, scope, displayErrors)
	{
		this._initCache();
		
		queryIds = Ext.Array.from(queryIds);
		
		var resolved = {}, unresolved = [], query;
		
		Ext.Array.each(queryIds, function(id) {
			query = this._getFromCache(id);
			if (query)
			{
				resolved[id] = query;
			}
			else
			{
				unresolved.push(id);
                
                if (!Ext.Array.contains(this._resolving, id))
                {
    				this._resolving.push(id);
                }
			}
		}, this);
		
		if (unresolved.length > 0)
		{
            this.getQueriesProperties([queryIds], this._sendGetQueryRequestCb, {scope: this, arguments: [callback, scope, displayErrors || false]});
        }
		else
		{
			callback.call(scope || window, Ext.Object.getValues(resolved), []);
		}
	},
	
	/**
	 * Callback function called after #_sendGetQueryRequest is processed
	 * @param {Object} response The response object.
     * @param {Object[]} args The transmitted arguments
     * @param {Function} [args.0] A callback
     * @param {String[]} [args.0.queryies] The known queries identifiers
     * @param {String[]} [args.0.unknownQueries] The unknown queries identifiers
     * @param {Object} [args.1] The callback scope
     * @param {Boolean} [args.2=false] Should display errors
	 * @param {Object} params The callback arguments (see {@link #_sendGetQueryRequest documentation)
	 * @private
	 */
	_sendGetQueryRequestCb: function (response, args, params)
	{
		var callback = args[0] || Ext.emptyFn;
        var scope = args[1] || window;
        var displayErrors = args[2] || false;
		
		var result = Ext.JSON.decode(Ext.dom.Query.selectValue('', response));
		
        var queries = [];
        var unknownQueries = response.unknownQueries || [];
        
        // Populate cache.
        Ext.Array.forEach(response.queries || [], function(data) {
            var query = Ext.create ('Ametys.plugins.queriesdirectory.Query', data);
            queries.push(query);
            this._addToCache(data.id, query);
        }, this);
		
		if (displayErrors && unknownQueries.length > 0)
		{
			Ametys.log.ErrorDialog.display({
				title: "{{i18n DAOS_QUERY_NOT_FOUND_ERROR_TITLE}}",
				text: "{{i18n DAOS_QUERY_NOT_FOUND_ERROR_MSG}}", 
				details: unknownQueries.join(", "),
				category: this.self.getName()
			});
		}
		
		// Execute callback with the retrieved queries
		callback.call(scope, queries, unknownQueries);
	},
	
	/**
	 * @property {Object} _cache The DAO cache. Keys are query id, Values are an object representing the query data.
	 * @private
	 */
	_cache: null,
	/**
	 * @property {Object} _cacheExpirations The object holding the expiration times for each entry in the cache
	 * @private
	 */
	_cacheExpirations : null,
	/**
	 * @property {String[]} _resolving Array of the requested query that are currently being resolved on the server.
	 * Used to avoid making several requests for the same query in a short period of time.
	 * @private
	 */
	_resolving: null,

	/**
	 * Update a query in the cache. Nothing will be done if the entry is not present in the cache.
	 * This must be done when some data of a query are modified.
	 * @param {String} queryId the query id
	 * @param {Object} dataToUpdate The properties to update for this query.
	 */
	localUpdate: function(queryId, dataToUpdate)
	{
		this._initCache();
		
		var query = this._getFromCache(queryId);
		if (query)
		{
			query.setProperties(dataToUpdate);
			this._addToCache(queryId, query);
		}
	},
	
	/**
	 * Remove a query in the cache.
	 * This must be used to invalidate an entry in the cache.
	 * Typically it must be done when a query have just been removed on the server.
	 * @param {String} queryId the query id
	 */
	localRemove: function(queryId)
	{
		this._initCache();
			
		delete this._cache[queryId];
		delete this._cacheExpirations[queryId];
		delete this._resolving[queryId];
		
	},
    
    /**
     * Cache initialization
     * @private
     */
    _initCache: function()
    {
        if (this._cache == null)
        {
            this._cache = {};
            this._cacheExpirations = {};
            this._resolving = [];
        }
    },
    
    /**
     * Cache invalidation
     */
    invalidateCache: function()
    {
        this._cache = null;
		this._initCache();
    },
	
	/**
	 * Internal method to put an entry in the cache.
     * @param {String} queryId The id of query
     * @param {Ametys.plugins.queriesdirectory.Query} query the query
	 * @private
	 */
	_addToCache: function(queryId, query)
	{
		this._cache[queryId] = query;
		this._cacheExpirations[queryId] = Ext.Date.add(new Date(), Ext.Date.MINUTE, 15);
		
		Ext.Array.remove(this._resolving, queryId);
	},
	
	/**
	 * Internal method to retrieve an entry in the cache.
     * @return {Ametys.plugins.queriesdirectory.Query} the query or null if not found
	 * @private
	 */
	_getFromCache: function(queryId)
	{
		if (this._cacheExpirations[queryId] && new Date() <= this._cacheExpirations[queryId])
		{
			return this._cache[queryId];
		}
		else
		{
			this.localRemove(queryId);
			return null;
		}
	}
});
