/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class provides a widget to select a query directory in the tree.
 * 
 * This widget is registered for fields of type Ametys.form.WidgetManager#TYPE_STRING.
 */
Ext.define('Ametys.plugins.queriesdirectory.widget.QueryContainer', {
    extend : 'Ametys.form.AbstractField',
    
    /**
     * @cfg {String} chooseContainerWindowTitle Title of the dialog box to choose containers.
     */
    chooseContainerWindowTitle: "{{i18n PLUGINS_QUERIESDIRECTORY_WIDGET_CONTAINER_DIALOG_TITLE}}",
    /**
     * @cfg {String} chooseContainerWindowIconCls The icon class for the dialog box to choose containers. 
     */
    chooseContainerWindowIconCls: 'ametysicon-file98',
    /**
     * @cfg {String} chooseContainerHelpMessage The help message to display on top of dialog box to choose containers. 
     */
    chooseContainerHelpMessage: "{{i18n PLUGINS_QUERIESDIRECTORY_WIDGET_CONTAINER_DIALOG_HINT}}",
    
    /**
     * @cfg {String} chooseButtonIconCls The choose button icon class
     */
    chooseButtonIconCls: 'ametysicon-file98',
    /**
     * @cfg {String} chooseButtonTooltipText The choose button tooltip text
     */ 
    chooseButtonTooltipText : "{{i18n PLUGINS_QUERIESDIRECTORY_WIDGET_CONTAINER_CHOOSE_BUTTON}}",
    /**
     * @cfg {String} deleteButtonIconCls The delete button icon class
     */
    deleteButtonIconCls: 'ametysicon-delete30', 
    /**
     * @cfg {String} deleteButtonTooltipText The delete button tooltip text
     */ 
    deleteButtonTooltipText : "{{i18n PLUGINS_QUERIESDIRECTORY_WIDGET_CONTAINER_DELETE_BUTTON}}",
    /**
     * @cfg {String} emptyText The text for empty field
     */
    emptyText: "{{i18n PLUGINS_QUERIESDIRECTORY_WIDGET_CONTAINER_NO_DIRECTORY_SELECTED}}",
    
    xtype: 'edition.query-container',
    
    initComponent: function()
    {
        this._text = Ext.create('Ext.Component', {
            cls: Ametys.form.AbstractField.READABLE_TEXT_CLS,
            html: this.emptyText,
            flex: 1
        });
        this._chooseButton = Ext.create('Ext.button.Button', {
            iconCls: this.chooseButtonIconCls,
            tooltip: this.chooseButtonTooltipText,
            handler: this._showChoosePopup,
            scope: this
        });
        this._deleteButton = Ext.create('Ext.button.Button', {
            iconCls: this.deleteButtonIconCls,
            tooltip: this.deleteButtonTooltipText,
            handler: this._deleteValue,
            scope: this,
            hidden: true
        });
        
        this.items = [this._text, this._chooseButton, this._deleteButton];

        this.layout = 'hbox';
        this.cls = this.emptyCls;
        
        this.callParent(arguments);
    },
    
    /**
     * Deletes the current widget value
     * @private
     */
     _deleteValue: function()
     {
         this.setValue(null);
         this._updateUI();
     },
    
    afterRender: function()
    {
        this.callParent(arguments);
        this._updateUI();
    },
    
    /**
     * Updates UI
     * @private
     */
    _updateUI: function()
    {   
        var value = this.value;
        
        if (!this.rendered)
        {
            return;
        }
        
        if (!value) 
        {
            this._deleteButton.hide();
            this._chooseButton.show();
        }
        else
        {
            this._deleteButton.show();
            this._chooseButton.hide();
        }
        
        this._updateDisplayField();
    },    

    /**
     * Updates the display field as an understanding value for the end user
     * @private
     */
    _updateDisplayField: function()
    {
        if (!this.rendered)
        {
            return;
        }
        
        if (this.value)
        {
            Ametys.plugins.queriesdirectory.QueriesDAO.getQueryContainerProperties([this.value], this._updateDisplayFieldCb, {scope: this});
        }
        else
        {
            this._updateDisplayFieldCb(null);
        }
    },
    
    /**
     * Transform the widget value in a human readable string
     * @return {String} a human readable string
     * @private
     */
    _getReadableValue: function ()
    {
        if (this.value)
        {
            return this._readableValue || this.value;
        }
        else
        {
            return this.emptyText;
        }
    },
    
    setValue: function(value) 
    {   
        this.callParent(arguments);
        this._updateUI();
    },
    
    /**
     * Opens a popup to choose value for current widget
     * @private
     */
    _showChoosePopup: function()
    {
        var me = this;
        Ametys.plugins.queriesdirectory.helper.ChooseQueryContainer.open({
            title: this.chooseContainerWindowTitle,
            iconCls: this.chooseContainerWindowIconCls,
            helpmessage: this.chooseContainerHelpMessage,
            callback: function(id) {
                me.setValue(id);
            }
        });
    },
    
    /**
     * Sets the readable value from the selected container
     * @param {Object} response The server response
     * @private
     */
    _updateDisplayFieldCb: function(response)
    {
        if (response)
        {
            if (this.value == 'root')
            {
                this._readableValue = "{{i18n PLUGINS_QUERIESDIRECTORY_UITOOL_QUERIES_ROOT_NAME}}";
            }
            else
            {
                this._readableValue = response.title;
            }
        }
        
        this._text.update(this._getReadableValue());
    }
});
