/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Combobox showing all catalogs (to be used with a CatalogEnumerator).
 * The value can be set with the "CURRENT" string to use the catalog defined as default catalog.
 */
Ext.define('Ametys.plugins.queriesdirectory.widget.SelectQuery', {
    extend: 'Ext.form.field.ComboBox',
    
    /**
     * @cfg {String} [profile=read_access] The access profile: 'read_access' to get queries in read access, 'write_access' to get queries in write access
     */
    
    /**
     * @cfg {String[]|String} [acceptedQueryTypes] The accepted query types. Can be an array or a coma-separated string. Can be empty or null to not filtered by query types.
     */
    
    constructor: function (config)
    {
        config.acceptedQueryTypes = config.acceptedQueryTypes || [];
        config.acceptedQueryTypes = Ext.isArray(config.acceptedQueryTypes) ? config.acceptedQueryTypes : config.acceptedQueryTypes.split(',');
        config.profile = config.profile || 'read_access';
        
        config.displayField = 'fullPath';
        config.valueField = 'id';
        
        config.store = {
            model: 'Ametys.plugins.queriesdirectory.tree.QueriesTree.QueryEntry',
            sortOnLoad: true,
            sorters: [{property: 'fullPath', direction:'ASC'}],
            
            proxy: {
                type: 'ametys',
                plugin: 'queries-directory',
                url: 'queries/list.json',
                extraParams: {
                    profile: config.profile,
                    acceptedQueryTypes: config.acceptedQueryTypes || [],
                    onlyQueries: true,
                    allDescendants: true
                },
                reader: {
                    type: 'json',
                    rootProperty: 'queries'
                }
            },
            autoLoad: true
        }
        
        this.callParent(arguments);
    },
    
    /**
     * Set the profile access
     * @param {String} profile the profile access: read_access or write_access
     */
    setProfileAccess: function(profile)
    {
        this.getStore().getProxy().setExtraParam('profile', profile);
    },
    
    /**
     * Set the accepted query types
     * @param {String|String[]} queryTypes The accepted query types. Can be empty or null to not filtered by query types.
     */
    setQueryTypes: function(queryTypes)
    {
        this.getStore().getProxy().setExtraParam('acceptedQueryTypes', Ext.Array.from(queryTypes));
    }
});
