/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.queriesdirectory;

import java.time.ZonedDateTime;

import javax.jcr.Node;

import org.ametys.cms.data.ametysobject.ModifiableModelAwareDataAwareAmetysObject;
import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.cms.indexing.solr.SolrAclCacheUninfluentialObject;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.MovableAmetysObject;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;

/**
 * Class representing a query, backed by a JCR node.<br>
 */
@SolrAclCacheUninfluentialObject
public class Query extends DefaultAmetysObject<QueryFactory> implements ModifiableModelAwareDataAwareAmetysObject, MovableAmetysObject
{

    /** Property name for query title */
    public static final String TITLE = "label";
    /** Property name for query type */
    public static final String TYPE = "type";
    /** Property name for query description */
    public static final String DESCRIPTION = "description";
    /** Property name for query documentation */
    public static final String DOCUMENTATION = "documentation";
    /** Property name for query content */
    public static final String CONTENT = "content";
    /** Property name for query author */
    public static final String AUTHOR = "author";
    /** Property name for query last contributor */
    public static final String CONTRIBUTOR = "contributor";
    /** Property name for query creation date */
    public static final String CREATIONDATE = "creationDate";
    /** Property name for query last modification date */
    public static final String LASTMODIFICATIONDATE = "lastModificationDate";

    /**
     * Type of a Query
     */
    public enum Type
    {
        /** Simple type. */
        SIMPLE,
        /** Default type. */
        ADVANCED,
        /** Script type. */
        SCRIPT;
        
        @Override
        public String toString()
        {
            return name().toLowerCase();
        }
    }
    
    /**
     * Rights profiles
     */
    public enum QueryProfile
    {
        /** Read access */
        READ_ACCESS,
        /** Write access */
        WRITE_ACCESS,
        /** Right access */
        RIGHT_ACCESS;
        
        @Override
        public String toString()
        {
            return name().toLowerCase();
        }
        
    }

    /**
     * Creates an {@link Query}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public Query(Node node, String parentPath, QueryFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    /**
     * Set the title of this query.
     * @param title the description
     */
    public void setTitle (String title)
    {
        this.setValue(TITLE, title);
    }
    
    /**
     * Set the type of this query.
     * @param type the description
     */
    public void setType (String type)
    {
        this.setValue(TYPE, type);
    }
    
    /**
     * Set the description of this query.
     * @param description the description
     */
    public void setDescription (String description)
    {
        this.setValue(DESCRIPTION, description);
    }
    
    /**
     * Set the documentation of this query.
     * @param documentation the documentation
     */
    public void setDocumentation (String documentation)
    {
        this.setValue(DOCUMENTATION, documentation);
    }
    
    /**
     * Set the content of this query.
     * @param content the content
     */
    public void setContent (String content)
    {
        this.setValue(CONTENT, content);
    }

    /**
     * Set the author of this query.
     * @param author the author
     */
    public void setAuthor(UserIdentity author)
    {
        this.setValue(AUTHOR, author);
    }

    /**
     * Set the last contributor of this query.
     * @param contributor the last contributor
     */
    public void setContributor(UserIdentity contributor)
    {
        this.setValue(CONTRIBUTOR, contributor);
    }
    
    /**
     * Set the date of the creation.
     * @param creationDate the last modification date to set.
     */
    public void setCreationDate(ZonedDateTime creationDate)
    {
        this.setValue(CREATIONDATE, creationDate);
    }
    
    /**
     * Set the date of the last modification.
     * @param lastModificationDate the last modification date to set.
     */
    public void setLastModificationDate(ZonedDateTime lastModificationDate)
    {
        this.setValue(LASTMODIFICATIONDATE, lastModificationDate);
    }
    
    /**
     * Get the title of the query
     * @return The title
     */
    public String getTitle()
    {        
        return this.getValue(TITLE);
    }
    
    /**
     * Get the type of the query
     * @return The type
     */
    public String getType()
    {        
        return this.getValue(TYPE);
    }

    
    /**
     * Get the description of the query
     * @return The description
     */
    public String getDescription()
    {
        return this.getValue(DESCRIPTION);
    }
    
    /**
     * Get the documentation of the query
     * @return The documentation
     */
    public String getDocumentation()
    {
        return this.getValue(DOCUMENTATION);
    }
    
    /**
     * Get the content of the query
     * @return The content
     */
    public String getContent()
    {
        return this.getValue(CONTENT);
    }
    

    /**
     * Get the author of the query
     * @return The author
     */
    public UserIdentity getAuthor ()
    {
        return this.getValue(AUTHOR);
    }
    
    /**
     * Get the last contributor of the query
     * @return The contributor
     */
    public UserIdentity getContributor()
    {
        return this.getValue(CONTRIBUTOR);
    }
    
    /**
     * Get the date of the last modification of the query
     * @return The date
     */
    public ZonedDateTime getCreationDate()
    {
        return this.getValue(CREATIONDATE);
    }
    
    /**
     * Get the date of the last modification of the query
     * @return The date
     */
    public ZonedDateTime getLastModificationDate()
    {
        return this.getValue(LASTMODIFICATIONDATE);
    }
    
    @Override
    public boolean canMoveTo(AmetysObject newParent) throws AmetysRepositoryException
    {
        return QueryAndContainerCommonMethods.canMoveTo(newParent, this, _getFactory().getQueryDAO());
    }
    
    @Override
    public void moveTo(AmetysObject newParent, boolean renameIfExist) throws AmetysRepositoryException, RepositoryIntegrityViolationException
    {
        QueryAndContainerCommonMethods.moveTo(newParent, renameIfExist, this);
    }
    
    @Override
    public void orderBefore(AmetysObject siblingNode) throws AmetysRepositoryException
    {
        QueryAndContainerCommonMethods.orderBefore(siblingNode, this);
    }

    public ModifiableIndexableDataHolder getDataHolder()
    {
        JCRRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, this._getFactory().getModel());
    }

}
