/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.queriesdirectory;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.util.Text;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.queriesdirectory.observation.ObservationConstants;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.MovableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableMap;

/**
 * DAO for manipulating queries
 */
public class QueryDAO extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = QueryDAO.class.getName();
    
    /** The right id to handle query */
    public static final String QUERY_HANDLE_RIGHT_ID = "QueriesDirectory_Rights_Admin";
    
    /** The right id to handle query container */
    public static final String QUERY_CONTAINER_HANDLE_RIGHT_ID = "QueriesDirectory_Rights_Containers";
    
    /** The alias id of the root {@link QueryContainer} */
    public static final String ROOT_QUERY_CONTAINER_ID = "root";
    
    /** Propery key for read access */
    public static final String READ_ACCESS_PROPERTY = "canRead";
    /** Propery key for write access */
    public static final String WRITE_ACCESS_PROPERTY = "canWrite";
    /** Propery key for rename access */
    public static final String RENAME_ACCESS_PROPERTY = "canRename";
    /** Propery key for delete access */
    public static final String DELETE_ACCESS_PROPERTY = "canDelete";
    /** Propery key for edit rights access */
    public static final String EDIT_RIGHTS_ACCESS_PROPERTY = "canAssignRights";
    
    private static final String __PLUGIN_NODE_NAME = "queriesdirectory";
    
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    
    /** The observation manager */
    protected ObservationManager _observationManager;

    /** The Ametys object resolver */
    private AmetysObjectResolver _resolver;

    private UserHelper _userHelper;

    private RightManager _rightManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _userProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
        _observationManager = (ObservationManager) serviceManager.lookup(ObservationManager.ROLE);
    }
    
    /**
     * Get the root plugin storage object.
     * @return the root plugin storage object.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    public QueryContainer getQueriesRootNode() throws AmetysRepositoryException
    {
        try
        {
            return _getOrCreateRootNode();
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the queries root node", e);
        }
    }
    
    private QueryContainer _getOrCreateRootNode() throws AmetysRepositoryException
    {
        ModifiableTraversableAmetysObject pluginsNode = _resolver.resolveByPath("/ametys:plugins");
        
        ModifiableTraversableAmetysObject pluginNode = (ModifiableTraversableAmetysObject) _getOrCreateNode(pluginsNode, __PLUGIN_NODE_NAME, "ametys:unstructured");
        
        return (QueryContainer) _getOrCreateNode(pluginNode, "ametys:queries", QueryContainerFactory.QUERY_CONTAINER_NODETYPE);
    }
    
    private static AmetysObject _getOrCreateNode(ModifiableTraversableAmetysObject parentNode, String nodeName, String nodeType) throws AmetysRepositoryException
    {
        AmetysObject definitionsNode;
        if (parentNode.hasChild(nodeName))
        {
            definitionsNode = parentNode.getChild(nodeName);
        }
        else
        {
            definitionsNode = parentNode.createChild(nodeName, nodeType);
            parentNode.saveChanges();
        }
        return definitionsNode;
    }
    
    /**
     * Get queries' properties
     * @param queryIds The ids of queries to retrieve
     * @return The queries' properties
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getQueriesProperties(List<String> queryIds)
    {
        Map<String, Object> result = new HashMap<>();
        
        List<Map<String, Object>> queries = new LinkedList<>();
        List<String> notAllowedQueries = new LinkedList<>();
        Set<String> unknownQueries = new HashSet<>();
        
        UserIdentity user = _userProvider.getUser();
        
        for (String id : queryIds)
        {
            try
            {
                Query query = _resolver.resolveById(id);
                
                if (canRead(user, query))
                {
                    queries.add(getQueryProperties(query));
                }
                else
                {
                    notAllowedQueries.add(query.getId());
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                unknownQueries.add(id);
            }
        }
        
        result.put("queries", queries);
        result.put("notAllowedQueries", notAllowedQueries);
        result.put("unknownQueries", unknownQueries);
        
        return result;
    }
    
    /**
     * Get the query properties
     * @param query The query
     * @return The query properties
     */
    public Map<String, Object> getQueryProperties (Query query)
    {
        Map<String, Object> infos = new HashMap<>();
        
        List<String> fullPath = new ArrayList<>();
        fullPath.add(query.getTitle());

        AmetysObject node = query.getParent();
        while (node instanceof QueryContainer
                && node.getParent() instanceof QueryContainer) // The parent must also be a container to avoid the root
        {
            fullPath.add(0, node.getName());
            node = node.getParent();
        }
        
        
        infos.put("isQuery", true);
        infos.put("id", query.getId());
        infos.put("title", query.getTitle());
        infos.put("fullPath", String.join(" > ", fullPath));
        infos.put("type", query.getType());
        infos.put("description", query.getDescription());
        infos.put("documentation", query.getDocumentation());
        infos.put("author", _userHelper.user2json(query.getAuthor()));
        infos.put("contributor", _userHelper.user2json(query.getContributor()));
        infos.put("content", query.getContent());
        infos.put("lastModificationDate", DateUtils.zonedDateTimeToString(query.getLastModificationDate()));
        infos.put("creationDate", DateUtils.zonedDateTimeToString(query.getCreationDate()));
        
        UserIdentity currentUser = _userProvider.getUser();
        infos.put(READ_ACCESS_PROPERTY, canRead(currentUser, query));
        infos.put(WRITE_ACCESS_PROPERTY, canWrite(currentUser, query));
        infos.put(DELETE_ACCESS_PROPERTY, canDelete(currentUser, query));
        infos.put(EDIT_RIGHTS_ACCESS_PROPERTY, canAssignRights(currentUser, query));
        
        return infos;
    }
    
    /**
     * Gets the ids of the path elements of a query or query container, i.e. the parent ids.
     * <br>For instance, if the query path is 'a/b/c', then the result list will be ["id-of-a", "id-of-b", "id-of-c"]
     * @param queryId The id of the query
     * @return the ids of the path elements of a query
     */
    @Callable (rights = {QUERY_CONTAINER_HANDLE_RIGHT_ID, QUERY_HANDLE_RIGHT_ID}, rightContext = QueriesDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public List<String> getIdsOfPath(String queryId)
    {
        AmetysObject queryOrQueryContainer = _resolver.resolveById(queryId);
        QueryContainer queriesRootNode = getQueriesRootNode();
        
        if (!(queryOrQueryContainer instanceof Query) && !(queryOrQueryContainer instanceof QueryContainer))
        {
            throw new IllegalArgumentException("The given id is not a query nor a query container");
        }
        
        List<String> pathElements = new ArrayList<>();
        QueryContainer current = queryOrQueryContainer.getParent();
        while (!queriesRootNode.equals(current))
        {
            pathElements.add(0, current.getId());
            current = current.getParent();
        }
        
        return pathElements;
    }
    
    /**
     * Get the root container properties
     * @return The root container properties
     */
    @Callable(rights = {"QueriesDirectory_Rights_Tool", "CMS_Rights_Delegate_Rights", "Runtime_Rights_Rights_Handle"})
    public Map<String, Object> getRootProperties()
    {
        return getQueryContainerProperties(getQueriesRootNode());
    }
    
    /**
     * Get the query container properties
     * @param id The query container id. Can be {@link #ROOT_QUERY_CONTAINER_ID} for the root container.
     * @return The query container properties
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getQueryContainerProperties(String id)
    {
        UserIdentity user = _userProvider.getUser();
        QueryContainer container = _getQueryContainer(id);
        if (!canRead(user, container))
        {
            throw new AccessDeniedException(user + " tried to access the properties of container " + id + " without sufficients rights");
        }
        return getQueryContainerProperties(container);
    }
    
    /**
     * Get the query container properties
     * @param queryContainer The query container
     * @return The query container properties
     */
    public Map<String, Object> getQueryContainerProperties(QueryContainer queryContainer)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("isQuery", false);
        infos.put("id", queryContainer.getId());
        infos.put("name", queryContainer.getName());
        infos.put("title", queryContainer.getName());
        infos.put("fullPath", _getFullPath(queryContainer));
        
        UserIdentity currentUser = _userProvider.getUser();
        
        infos.put(READ_ACCESS_PROPERTY, canRead(currentUser, queryContainer));
        infos.put(WRITE_ACCESS_PROPERTY, canWrite(currentUser, queryContainer)); // create container, add query
        infos.put(RENAME_ACCESS_PROPERTY, canRename(currentUser, queryContainer)); // rename
        infos.put(DELETE_ACCESS_PROPERTY, canDelete(currentUser, queryContainer)); // delete or move
        infos.put(EDIT_RIGHTS_ACCESS_PROPERTY, canAssignRights(currentUser, queryContainer)); // edit rights

        return infos;
    }
    
    private String _getFullPath(QueryContainer queryContainer)
    {
        List<String> fullPath = new ArrayList<>();
        fullPath.add(queryContainer.getName());

        AmetysObject node = queryContainer.getParent();
        while (node instanceof QueryContainer
                && node.getParent() instanceof QueryContainer) // The parent must also be a container to avoid the root
        {
            fullPath.add(0, node.getName());
            node = node.getParent();
        }
        return String.join(" > ", fullPath);
    }
    
    /**
     * Filter queries on the server side
     * @param rootNode The root node where to seek (to refresh subtrees)
     * @param search Textual search
     * @param ownerOnly Only queries of the owner will be returned
     * @param requestType Only simple/advanced requests will be returned
     * @param solrType Only Solr requests will be returned
     * @param scriptType Only script requests will be returned
     * @param formattingType Only formatting will be returned
     * @return The list of query path
     */
    @Callable (rights = {"QueriesDirectory_Rights_Tool", "CMS_Rights_Delegate_Rights", "Runtime_Rights_Rights_Handle"})
    public List<String> filterQueries(String rootNode, String search, boolean ownerOnly, boolean requestType, boolean solrType, boolean scriptType, boolean formattingType)
    {
        List<String> matchingPaths = new ArrayList<>();
        _getMatchingQueries(_getQueryContainer(rootNode),
                            matchingPaths, StringUtils.stripAccents(search.toLowerCase()), ownerOnly, requestType, solrType, scriptType, formattingType);
        
        return matchingPaths;
    }
    
    private void _getMatchingQueries(QueryContainer queryContainer, List<String> matchingPaths, String search, boolean ownerOnly, boolean requestType, boolean solrType, boolean scriptType, boolean formattingType)
    {
        if (StringUtils.isBlank(search) && !ownerOnly && !requestType && !solrType && !scriptType && !formattingType)
        {
            return;
        }
        
        String containerName = StringUtils.stripAccents(queryContainer.getName().toLowerCase());
        if (containerName.contains(search))
        {
            matchingPaths.add(_getQueryPath(queryContainer));
        }
        
        if (!hasAnyReadableDescendant(_userProvider.getUser(), queryContainer))
        {
            return;
        }
        
        try (AmetysObjectIterable<AmetysObject> children  = queryContainer.getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof QueryContainer childQueryContainer)
                {
                    _getMatchingQueries(childQueryContainer, matchingPaths, search, ownerOnly, requestType, solrType, scriptType, formattingType);
                }
                else if (child instanceof Query childQuery)
                {
                    _getMatchingQuery(childQuery, matchingPaths, search, ownerOnly, requestType, solrType, scriptType, formattingType);
                }
            }
        }
    }
    
    private void _getMatchingQuery(Query query, List<String> matchingPaths, String search, boolean ownerOnly, boolean requestType, boolean solrType, boolean scriptType, boolean formattingType)
    {
        String type = query.getType();
        
        if (!canRead(_userProvider.getUser(), query)
            || formattingType && !"formatting".equals(type)
            || requestType && !Query.Type.SIMPLE.toString().equals(type) && !Query.Type.ADVANCED.toString().equals(type)
            || solrType && !"solr".equals(type)
            || scriptType && !Query.Type.SCRIPT.toString().equals(type)
            || ownerOnly && !query.getAuthor().equals(_userProvider.getUser()))
        {
            return;
        }
        
        if (_contains(query, search))
        {
            matchingPaths.add(_getQueryPath(query));
        }
    }
    
    /**
     * Check it the query contains the search string in its title, content, description or documentation
     */
    private boolean _contains(Query query, String search)
    {
        if (StringUtils.stripAccents(query.getTitle().toLowerCase()).contains(search))
        {
            return true;
        }
        
        if (StringUtils.stripAccents(query.getContent().toLowerCase()).contains(search))
        {
            return true;
        }
        
        if (StringUtils.stripAccents(query.getDescription().toLowerCase()).contains(search))
        {
            return true;
        }
        return false;
    }
    
    private String _getQueryPath(AmetysObject queryOrContainer)
    {
        if (queryOrContainer instanceof Query
            || queryOrContainer instanceof QueryContainer
                && queryOrContainer.getParent() instanceof QueryContainer)
        {
            return _getQueryPath(queryOrContainer.getParent()) + "#" + queryOrContainer.getId();
        }
        else
        {
            return "#root";
        }
    }

    /**
     * Creates a new {@link Query}
     * @param title The title of the query
     * @param desc The description of the query
     * @param documentation The documentation of the query
     * @param type The type of the query
     * @param content The content of the query
     * @param parentId The id of the parent of the query. Use {@link #ROOT_QUERY_CONTAINER_ID} for the root container.
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> createQuery(String title, String desc, String documentation, String type, String content, String parentId)
    {
        Map<String, Object> results = new HashMap<>();

        QueryContainer queriesNode = _getQueryContainer(parentId);
        
        if (!canWrite(_userProvider.getUser(), queriesNode))
        {
            results.put("message", "not-allowed");
            return results;
        }

        String name = NameHelper.filterName(title);
        
        // Find unique name
        String uniqueName = name;
        int index = 2;
        while (queriesNode.hasChild(uniqueName))
        {
            uniqueName = name + "-" + (index++);
        }
        
        Query query = queriesNode.createChild(uniqueName, QueryFactory.QUERY_NODETYPE);
        query.setTitle(title);
        query.setDescription(desc);
        query.setDocumentation(documentation);
        query.setAuthor(_userProvider.getUser());
        query.setContributor(_userProvider.getUser());
        query.setType(type);
        query.setContent(content);
        query.setCreationDate(ZonedDateTime.now());
        query.setLastModificationDate(ZonedDateTime.now());

        queriesNode.saveChanges();

        results.put("id", query.getId());
        results.put("title", query.getTitle());
        results.put("content", query.getContent());
        
        return results;
    }
    
    /**
     * Creates a new {@link QueryContainer}
     * @param parentId The id of the parent. Use {@link #ROOT_QUERY_CONTAINER_ID} for the root container.
     * @param name The desired name for the new {@link QueryContainer}
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> createQueryContainer(String parentId, String name)
    {
        QueryContainer parent = _getQueryContainer(parentId);

        if (!canWrite(_userProvider.getUser(), parent))
        {
            return ImmutableMap.of("message", "not-allowed");
        }
        
        int index = 2;
        String legalName = Text.escapeIllegalJcrChars(name);
        String realName = legalName;
        while (parent.hasChild(realName))
        {
            realName = legalName + " (" + index + ")";
            index++;
        }
        
        QueryContainer createdChild = parent.createChild(realName, QueryContainerFactory.QUERY_CONTAINER_NODETYPE);
        parent.saveChanges();
        
        return getQueryContainerProperties(createdChild);
    }
    
    /**
     * Edits a {@link Query}
     * @param id The id of the query
     * @param title The title of the query
     * @param desc The description of the query
     * @param documentation The documentation of the query
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> updateQuery(String id, String title, String desc, String documentation)
    {
        Map<String, Object> results = new HashMap<>();
        
        Query query = _resolver.resolveById(id);
        
        if (canWrite(_userProvider.getUser(), query))
        {
            query.setTitle(title);
            query.setDescription(desc);
            query.setDocumentation(documentation);
            query.setContributor(_userProvider.getUser());
            query.setLastModificationDate(ZonedDateTime.now());
            query.saveChanges();
        }
        else
        {
            results.put("message", "not-allowed");
        }

        results.put("id", query.getId());
        results.put("title", query.getTitle());
        
        return results;
    }
    
    /**
     * Renames a {@link QueryContainer}
     * @param id The id of the query container
     * @param newName The new name of the container
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> renameQueryContainer(String id, String newName)
    {
        QueryContainer queryContainer = _resolver.resolveById(id);
        
        UserIdentity currentUser = _userProvider.getUser();
        
        if (canRename(currentUser, queryContainer))
        {
            String legalName = Text.escapeIllegalJcrChars(newName);
            Node node = queryContainer.getNode();
            try
            {
                node.getSession().move(node.getPath(), node.getParent().getPath() + '/' + legalName);
                node.getSession().save();
                
                return getQueryContainerProperties((QueryContainer) _resolver.resolveById(id));
            }
            catch (RepositoryException e)
            {
                getLogger().error("Unable to rename query container '{}'", id, e);
                return Map.of("message", "cannot-rename");
            }
        }
        else
        {
            return Map.of("message", "not-allowed");
        }
    }
    
    /**
     * Moves a {@link Query}
     * @param id The id of the query
     * @param newParentId The id of the new parent container of the query. Use {@link #ROOT_QUERY_CONTAINER_ID} for the root container.
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> moveQuery(String id, String newParentId)
    {
        Map<String, Object> results = new HashMap<>();
        Query query = _resolver.resolveById(id);
        QueryContainer queryContainer = _getQueryContainer(newParentId);
        
        if (canDelete(_userProvider.getUser(), query) && canWrite(_userProvider.getUser(), queryContainer))
        {
            if (!_move(query, newParentId))
            {
                results.put("message", "cannot-move");
            }
        }
        else
        {
            results.put("message", "not-allowed");
        }
        
        results.put("id", query.getId());
        return results;
    }
    
    /**
     * Moves a {@link QueryContainer}
     * @param id The id of the query container
     * @param newParentId The id of the new parent container of the query container. Use {@link #ROOT_QUERY_CONTAINER_ID} for the root container.
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> moveQueryContainer(String id, String newParentId)
    {
        QueryContainer queryContainer = _resolver.resolveById(id);
        QueryContainer parentQueryContainer = _getQueryContainer(newParentId);
        UserIdentity currentUser = _userProvider.getUser();
        
        if (canDelete(currentUser, queryContainer) && canWrite(currentUser, parentQueryContainer))
        {
            if (_move(queryContainer, newParentId))
            {
                // returns updated properties
                return getQueryContainerProperties((QueryContainer) _resolver.resolveById(id));
            }
            else
            {
                return Map.of("id", id, "message", "cannot-move");
            }
            
        }
        else
        {
            return Map.of("id", id, "message", "not-allowed");
        }
    }
    
    private boolean _move(MovableAmetysObject obj, String newParentId)
    {
        QueryContainer newParent = _getQueryContainer(newParentId);
        if (obj.canMoveTo(newParent))
        {
            try
            {
                obj.moveTo(newParent, false);
                return true;
            }
            catch (AmetysRepositoryException e)
            {
                getLogger().error("Unable to move '{}' to query container '{}'", obj.getId(), newParentId, e);
            }
        }
        return false;
    }
    
    private QueryContainer _getQueryContainer(String id)
    {
        return ROOT_QUERY_CONTAINER_ID.equals(id)
                ? getQueriesRootNode()
                : _resolver.resolveById(id);
    }
    
    /**
     * Saves a {@link Query}
     * @param id The id of the query
     * @param type The type of the query
     * @param content The content of the query
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> saveQuery(String id, String type, String content)
    {
        Map<String, Object> results = new HashMap<>();
        
        Query query = _resolver.resolveById(id);
        UserIdentity user = _userProvider.getUser();
        if (canWrite(user, query))
        {
            query.setType(type);
            query.setContent(content);
            query.setContributor(user);
            query.setLastModificationDate(ZonedDateTime.now());
            query.saveChanges();
            
            results.put("id", query.getId());
            results.put("content", query.getContent());
            results.put("title", query.getTitle());
        }
        else
        {
            results.put("message", "not-allowed");
        }
        
        return results;
    }
    
    /**
     * Deletes {@link Query}(ies)
     * @param ids The ids of the queries to delete
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteQuery(List<String> ids)
    {
        Map<String, Object> results = new HashMap<>();
        
        List<String> deletedQueries = new ArrayList<>();
        List<String> unknownQueries = new ArrayList<>();
        List<String> notallowedQueries = new ArrayList<>();
         
        for (String id : ids)
        {
            try
            {
                Query query = _resolver.resolveById(id);
                
                if (canDelete(_userProvider.getUser(), query))
                {
                    Map<String, Object> params = new HashMap<>();
                    params.put(ObservationConstants.ARGS_QUERY_ID, query.getId());

                    query.remove();
                    query.saveChanges();
                    deletedQueries.add(id);
                    
                    _observationManager.notify(new Event(ObservationConstants.EVENT_QUERY_DELETED, _userProvider.getUser(), params));
                }
                else
                {
                    notallowedQueries.add(query.getTitle());
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                unknownQueries.add(id);
                getLogger().error("Unable to delete query. The query of id '{}' doesn't exist", id, e);
            }
        }
        
        results.put("deletedQueries", deletedQueries);
        results.put("notallowedQueries", notallowedQueries);
        results.put("unknownQueries", unknownQueries);
        
        return results;
    }
    
    /**
     * Determines if application must warn before deleting the given {@link QueryContainer}s
     * @param ids The {@link QueryContainer} ids
     * @return <code>true</code> if application must warn
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public boolean mustWarnBeforeDeletion(List<String> ids)
    {
        return ids.stream()
                .anyMatch(LambdaUtils.wrapPredicate(this::_mustWarnBeforeDeletion));
    }
    
    private boolean _mustWarnBeforeDeletion(String id)
    {
        QueryContainer container = _resolver.resolveById(id);
        AmetysObjectIterable<Query> allQueries = getChildQueriesForAdministrator(container, false, List.of());
        return _containsNotOwnQueries(allQueries);
    }
    
    private boolean _containsNotOwnQueries(AmetysObjectIterable<Query> allQueries)
    {
        UserIdentity currentUser = _userProvider.getUser();
        return allQueries.stream()
                .map(Query::getAuthor)
                .anyMatch(Predicates.not(currentUser::equals));
    }

    
    /**
     * Deletes {@link QueryContainer}(s)
     * @param ids The ids of the query containers to delete
     * @return A result map
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteQueryContainer(List<String> ids)
    {
        Map<String, Object> results = new HashMap<>();
        
        List<Object> deletedQueryContainers = new ArrayList<>();
        List<String> unknownQueryContainers = new ArrayList<>();
        List<String> notallowedQueryContainers = new ArrayList<>();
         
        for (String id : ids)
        {
            try
            {
                QueryContainer queryContainer = _resolver.resolveById(id);
                
                if (canDelete(_userProvider.getUser(), queryContainer))
                {
                    Map<String, Object> props = getQueryContainerProperties(queryContainer);
                    queryContainer.remove();
                    queryContainer.saveChanges();
                    deletedQueryContainers.add(props);
                }
                else
                {
                    notallowedQueryContainers.add(queryContainer.getName());
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                unknownQueryContainers.add(id);
                getLogger().error("Unable to delete query container. The query container of id '{}' doesn't exist", id, e);
            }
        }
        
        results.put("deletedQueryContainers", deletedQueryContainers);
        results.put("notallowedQueryContainers", notallowedQueryContainers);
        results.put("unknownQueryContainers", unknownQueryContainers);
        
        return results;
    }
    
    /**
     * Gets all queries for administrator for given parent
     * @param parent The {@link QueryContainer}, defining the context from which getting children
     * @param onlyDirect <code>true</code> in order to have only direct child queries from parent path, <code>false</code> otherwise to have all queries at any level underneath the parent path
     * @param acceptedTypes The type of queries. Can be null or empty to accept all types.
     * @return all queries for administrator for given parent
     */
    public AmetysObjectIterable<Query> getChildQueriesForAdministrator(QueryContainer parent, boolean onlyDirect, List<String> acceptedTypes)
    {
        return _resolver.query(QueryHelper.getXPathForQueriesForAdministrator(parent, onlyDirect, acceptedTypes));
    }
    
    /**
     * Gets all queries in READ access for given parent
     * @param parent The {@link QueryContainer}, defining the context from which getting children
     * @param onlyDirect <code>true</code> in order to have only direct child queries from parent path, <code>false</code> otherwise to have all queries at any level underneath the parent path
     * @param user The user
     * @param acceptedTypes The type of queries. Can be null or empty to accept all types.
     * @return all queries in READ access for given parent
     */
    public Stream<Query> getChildQueriesInReadAccess(QueryContainer parent, boolean onlyDirect, UserIdentity user, List<String> acceptedTypes)
    {
        return _resolver.query(QueryHelper.getXPathForQueries(parent, onlyDirect, acceptedTypes))
                        .stream()
                        .filter(Query.class::isInstance)
                        .map(obj -> (Query) obj)
                        .filter(query -> canRead(user, query));
    }

    /**
     * Determine if user has read access on a query
     * @param userIdentity the user
     * @param query the query
     * @return true if the user have read rights on a query
     */
    public boolean canRead(UserIdentity userIdentity, Query query)
    {
        return _rightManager.hasReadAccess(userIdentity, query) || canWrite(userIdentity, query);
    }
    
    /**
     * Determines if the user has read access on a query container
     * @param userIdentity the user
     * @param queryContainer the query container
     * @return true if the user has read access on the query container
     */
    public boolean canRead(UserIdentity userIdentity, QueryContainer queryContainer)
    {
        return _rightManager.hasReadAccess(userIdentity, queryContainer) || canWrite(userIdentity, queryContainer);
    }

    /**
     * Gets all queries in WRITE access for given parent
     * @param parent The {@link QueryContainer}, defining the context from which getting children
     * @param onlyDirect <code>true</code> in order to have only direct child queries from parent path, <code>false</code> otherwise to have all queries at any level underneath the parent path
     * @param user The user
     * @param acceptedTypes The type of queries. Can be null or empty to accept all types.
     * @return all queries in WRITE access for given parent
     */
    public Stream<Query> getChildQueriesInWriteAccess(QueryContainer parent, boolean onlyDirect, UserIdentity user, List<String> acceptedTypes)
    {
        return _resolver.query(QueryHelper.getXPathForQueries(parent, onlyDirect, acceptedTypes))
                .stream()
                .filter(Query.class::isInstance)
                .map(obj -> (Query) obj)
                .filter(query -> canWrite(user, query));
    }
    
    /**
     * Determines if the user has write access on a query
     * @param userIdentity the user
     * @param query the query
     * @return true if the user has write access on query
     */
    public boolean canWrite(UserIdentity userIdentity, Query query)
    {
        return _rightManager.hasRight(userIdentity, QUERY_HANDLE_RIGHT_ID, query) == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Determines if the user can delete a query
     * @param userIdentity the user
     * @param query the query
     * @return true if the user can delete the query
     */
    public boolean canDelete(UserIdentity userIdentity, Query query)
    {
        return canWrite(userIdentity, query) && canWrite(userIdentity, (QueryContainer) query.getParent());
    }
    
    /**
     * Determines if the user can rename a query container
     * @param userIdentity the user
     * @param queryContainer the query container
     * @return true if the user can delete the query
     */
    public boolean canRename(UserIdentity userIdentity, QueryContainer queryContainer)
    {
        return !_isRoot(queryContainer) && canWrite(userIdentity, queryContainer) && canWrite(userIdentity, (QueryContainer) queryContainer.getParent());
    }
    
    /**
     * Determines if the user can delete a query container
     * @param userIdentity the user
     * @param queryContainer the query container
     * @return true if the user can delete the query container
     */
    public boolean canDelete(UserIdentity userIdentity, QueryContainer queryContainer)
    {
        return !_isRoot(queryContainer) // is not root
                && canWrite(userIdentity, (QueryContainer) queryContainer.getParent()) // has write access on parent
                && canWrite(userIdentity, queryContainer, true); // has write access on itselft and each descendant
    }
    
    /**
     * Determines if the query container is the root node
     * @param queryContainer the query container
     * @return true if is root
     */
    protected boolean _isRoot(QueryContainer queryContainer)
    {
        return getQueriesRootNode().equals(queryContainer);
    }

    /**
     * Gets all queries in WRITE access for given parent
     * @param parent The {@link QueryContainer}, defining the context from which getting children
     * @param onlyDirect <code>true</code> in order to have only direct child queries from parent path, <code>false</code> otherwise to have all queries at any level underneath the parent path
     * @param user The user
     * @param acceptedTypes The type of queries. Can be null or empty to accept all types.
     * @return all queries in WRITE access for given parent
     */
    public Stream<Query> getChildQueriesInRightAccess(QueryContainer parent, boolean onlyDirect, UserIdentity user, List<String> acceptedTypes)
    {
        return _resolver.query(QueryHelper.getXPathForQueries(parent, onlyDirect, acceptedTypes))
                .stream()
                .filter(Query.class::isInstance)
                .map(obj -> (Query) obj)
                .filter(query -> canAssignRights(user, query));
    }

    /**
     * Check if a user can edit rights on a query
     * @param userIdentity the user
     * @param query the query
     * @return true if the user can edit rights on a query
     */
    public boolean canAssignRights(UserIdentity userIdentity, Query query)
    {
        return canWrite(userIdentity, query) || _rightManager.hasRight(userIdentity, "Runtime_Rights_Rights_Handle", "/cms") == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Check if a user has creation rights on a query container
     * @param userIdentity the user identity
     * @param queryContainer the query container
     * @return true if the user has creation rights on a query container
     */
    public boolean canCreate(UserIdentity userIdentity, QueryContainer queryContainer)
    {
        return canWrite(userIdentity, queryContainer) || _rightManager.hasRight(userIdentity, QUERY_CONTAINER_HANDLE_RIGHT_ID, "/cms") == RightResult.RIGHT_ALLOW;
    }

    
    /**
     * Check if a user has write access on a query container
     * @param userIdentity the user identity
     * @param queryContainer the query container
     * @return true if the user has write access on the a query container
     */
    public boolean canWrite(UserIdentity userIdentity, QueryContainer queryContainer)
    {
        return canWrite(userIdentity, queryContainer, false);
    }
    
    /**
     * Check if a user has write access on a query container
     * @param userIdentity the user user identity
     * @param queryContainer the query container
     * @param recursively true to check write access on all descendants recursively
     * @return true if the user has write access on the a query container
     */
    public boolean canWrite(UserIdentity userIdentity, QueryContainer queryContainer, boolean recursively)
    {
        boolean hasRight = _rightManager.hasRight(userIdentity, QUERY_CONTAINER_HANDLE_RIGHT_ID, queryContainer) == RightResult.RIGHT_ALLOW;
        if (!hasRight)
        {
            return false;
        }
           
        if (recursively)
        {
            try (AmetysObjectIterable<AmetysObject> children = queryContainer.getChildren())
            {
                for (AmetysObject child : children)
                {
                    if (child instanceof QueryContainer)
                    {
                        hasRight = hasRight && canWrite(userIdentity, (QueryContainer) child, true);
                    }
                    else if (child instanceof Query)
                    {
                        hasRight = hasRight && canWrite(userIdentity, (Query) child);
                    }
                    
                    if (!hasRight)
                    {
                        return false;
                    }
                }
            }
        }
        
        return hasRight;
    }
    
    /**
     * Check if a user can edit rights on a query container
     * @param userIdentity the user
     * @param queryContainer the query container
     * @return true if the user can edit rights on a query
     */
    public boolean canAssignRights(UserIdentity userIdentity, QueryContainer queryContainer)
    {
        return canWrite(userIdentity, queryContainer) || _rightManager.hasRight(userIdentity, "Runtime_Rights_Rights_Handle", "/cms") == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Gets all query containers for given parent
     * @param parent The {@link QueryContainer}, defining the context from which getting children
     * @return all query containers for given parent
     */
    public AmetysObjectIterable<QueryContainer> getChildQueryContainers(QueryContainer parent)
    {
        return _resolver.query(QueryHelper.getXPathForQueryContainers(parent));
    }
    
    /**
     * Check if a folder have a descendant in read access for a given user
     * @param userIdentity the user
     * @param queryContainer the query container
     * @return true if the user have read right for at least one child of this container
     */
    public boolean hasAnyReadableDescendant(UserIdentity userIdentity, QueryContainer queryContainer)
    {
        try (AmetysObjectIterable<AmetysObject> children = queryContainer.getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof QueryContainer)
                {
                    if (canRead(userIdentity, (QueryContainer) child) || hasAnyReadableDescendant(userIdentity, (QueryContainer) child))
                    {
                        return true;
                    }
                }
                else if (child instanceof Query && canRead(userIdentity, (Query) child))
                {
                    return true;
                }
            }
        }
        
        return false;
    }

    /**
     * Check if a query container have descendant in write access for a given user
     * @param userIdentity the user identity
     * @param queryContainer the query container
     * @return true if the user have write right for at least one child of this container
     */
    public Boolean hasAnyWritableDescendant(UserIdentity userIdentity, QueryContainer queryContainer)
    {
        boolean canWrite = false;
        
        try (AmetysObjectIterable<AmetysObject> children = queryContainer.getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof QueryContainer)
                {
                    if (canWrite(userIdentity, (QueryContainer) child) || hasAnyWritableDescendant(userIdentity, (QueryContainer) child))
                    {
                        return true;
                    }
                }
                else if (child instanceof Query && canWrite(userIdentity, (Query) child))
                {
                    return true;
                }
            }
        }
        
        return canWrite;
    }
    
    /**
     * Check if a folder have descendant in right assignment access for a given user
     * @param userIdentity the user identity
     * @param queryContainer the query container
     * @return true if the user have right assignment right for at least one child of this container
     */
    public boolean hasAnyAssignableDescendant(UserIdentity userIdentity, QueryContainer queryContainer)
    {
        try (AmetysObjectIterable<AmetysObject> children = queryContainer.getChildren())
        {
            for (AmetysObject child : children)
            {
                if (child instanceof QueryContainer)
                {
                    if (canAssignRights(userIdentity, (QueryContainer) child) || hasAnyAssignableDescendant(userIdentity, (QueryContainer) child))
                    {
                        return true;
                    }
                }
                else if (child instanceof Query)
                {
                    if (canAssignRights(userIdentity, (Query) child))
                    {
                        return true;
                    }
                }
            }
            return false;
        }
    }
}
