/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.queriesdirectory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.jackrabbit.util.ISO9075;

import org.ametys.cms.repository.Content;
import org.ametys.cms.search.SortOrder;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.cms.search.content.ContentSearcherFactory.ContentSearchSort;
import org.ametys.cms.search.content.ContentSearcherFactory.SearchModelContentSearcher;
import org.ametys.cms.search.ui.model.SearchUIModel;
import org.ametys.cms.search.ui.model.SearchUIModelExtensionPoint;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for manipulating {@link Query}
 *
 */
public class QueryHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = QueryHelper.class.getName();
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** JSON Utils */
    protected JSONUtils _jsonUtils;
    
    /** SearchUI Model Extension Point */
    protected SearchUIModelExtensionPoint _searchUiEP;
    
    /** Content Searcher Factory */
    protected ContentSearcherFactory _contentSearcherFactory;

    /** The service manager */
    protected ServiceManager _manager;

    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _searchUiEP = (SearchUIModelExtensionPoint) manager.lookup(SearchUIModelExtensionPoint.ROLE);
        _contentSearcherFactory = (ContentSearcherFactory) manager.lookup(ContentSearcherFactory.ROLE);
    }
    
    /**
     * Creates the XPath query to get all query containers
     * @param queryContainer The {@link QueryContainer}, defining the context from which getting children 
     * @return The XPath query
     */
    static String getXPathForQueryContainers(QueryContainer queryContainer)
    {
        return _getXPathQuery(queryContainer, true, ObjectToReturn.QUERY_CONTAINER, List.of());
    }
    
    /**
     * Creates the XPath query to get all queries for administrator
     * @param queryContainer The {@link QueryContainer}, defining the context from which getting children 
     * @param onlyDirectChildren <code>true</code> in order to have only direct child queries from parent path, <code>false</code> otherwise to have all queries at any level underneath the parent path
     * @param acceptedTypes The type of queries. Can be null or empty to accept all types.
     * @return The XPath query
     */
    static String getXPathForQueriesForAdministrator(QueryContainer queryContainer, boolean onlyDirectChildren, List<String> acceptedTypes)
    {
        return _getXPathQuery(queryContainer, onlyDirectChildren, ObjectToReturn.QUERY, acceptedTypes);
    }
    
    /**
     * Creates the XPath query to get all queries in WRITE access
     * @param queryContainer The {@link QueryContainer}, defining the context from which getting children 
     * @param onlyDirectChildren <code>true</code> in order to have only direct child queries from parent path, <code>false</code> otherwise to have all queries at any level underneath the parent path
     * @param acceptedTypes The type of queries. Can be null or empty to accept all types.
     * @return The XPath query
     */
    static String getXPathForQueries(QueryContainer queryContainer, boolean onlyDirectChildren, List<String> acceptedTypes)
    {
        return _getXPathQuery(queryContainer, onlyDirectChildren, ObjectToReturn.QUERY, acceptedTypes);
    }
    
    private static StringBuilder _getParentPath(QueryContainer queryContainer)
    {
        try
        {
            StringBuilder parentPath = new StringBuilder("/jcr:root")
                    .append(ISO9075.encodePath(queryContainer.getNode().getPath()));
            return parentPath;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    private static String _getXPathQuery(QueryContainer queryContainer, boolean onlyDirectChildren, ObjectToReturn objectToReturn, List<String> acceptedTypes)
    {
        StringBuilder parentPath = _getParentPath(queryContainer);
        final String slashOrDoubleSlash = onlyDirectChildren ? "/" : "//";
        StringBuilder query = parentPath
                .append(slashOrDoubleSlash)
                .append("element(*, ")
                .append(objectToReturn.toNodetype())
                .append(")");
        if (acceptedTypes != null && !acceptedTypes.isEmpty())
        {
            List<Expression> exprs = new ArrayList<>();
            for (String type : acceptedTypes)
            {
                exprs.add(new StringExpression(Query.TYPE, Operator.EQ, type));
            }
            Expression typeExpression = new OrExpression(exprs.toArray(new Expression[exprs.size()]));
            query.append("[").append(typeExpression.build()).append("]");
        }
        
        return query.toString();
    }
    
    private static enum ObjectToReturn
    {
        QUERY,
        QUERY_CONTAINER;
        
        private String _nodetype;
        
        static
        {
            QUERY._nodetype = QueryFactory.QUERY_NODETYPE;
            QUERY_CONTAINER._nodetype = QueryContainerFactory.QUERY_CONTAINER_NODETYPE;
        }
        
        String toNodetype()
        {
            return _nodetype;
        }
    }
    

    /**
     * Execute a query
     * @param queryId id of the query to execute
     * @return the results of the query
     * @throws Exception something went wrong
     */
    public AmetysObjectIterable<Content> executeQuery(String queryId) throws Exception
    {
        AmetysObject ametysObject = _resolver.resolveById(queryId);
        if (ametysObject instanceof Query)
        {
            return executeQuery((Query) ametysObject);
        }
        else
        {
            return null;
        }
    }

    /**
     * Execute a query
     * @param query the query to execute
     * @return the results of the query, can be null
     * @throws Exception if failed to execute query
     */
    public AmetysObjectIterable<Content> executeQuery(Query query) throws Exception
    {
        Map<String, Object> exportParams = getExportParams(query);
        
        int limit = getLimitForQuery(exportParams);
        List<ContentSearchSort> sort = getSortForQuery(exportParams);
        
        String model = getModelForQuery(exportParams);
        if ("solr".equals(query.getType()))
        {
            String queryStr = getQueryStringForQuery(exportParams);
            Set<String> contentTypeIds = getContentTypesForQuery(exportParams).stream().collect(Collectors.toSet());
            AmetysObjectIterable<Content> results = _contentSearcherFactory.create(contentTypeIds)
                    .withSort(sort)
                    .withLimits(0, limit)
                    .search(queryStr);
                    
            return results;
        }
        else if (Query.Type.SIMPLE.toString().equals(query.getType()) || Query.Type.ADVANCED.toString().equals(query.getType()))
        {
            Map<String, Object> values = getValuesForQuery(exportParams);
            
            Map<String, Object> contextualParameters = getContextualParametersForQuery(exportParams);
            String searchMode = getSearchModeForQuery(exportParams);
            SearchUIModel uiModel = _searchUiEP.getExtension(model);
            
            SearchModelContentSearcher searcher = _contentSearcherFactory.create(uiModel);
            
            return searcher
                    .withLimits(0, limit)
                    .withSort(sort)
                    .withSearchMode(searchMode)
                    .search(values, contextualParameters);
        }
        else
        {
            getLogger().warn("This method should only handle solr, advanced or simple queries. Query id '{}' is type '{}'", query.getId(), query.getType());
            return null;
        }
    }
    
    /**
     * Get the limit of results stored in a query, or -1 if none is found
     * @param exportParams export params of the query, available via {@link QueryHelper#getExportParams(Query)}
     * @return the maximum number of results that this query should return, -1 if no limit
     */
    public int getLimitForQuery(Map<String, Object> exportParams)
    {
        return Optional.of("limit")
            .map(exportParams::get)
            .map(Integer.class::cast)
            .filter(l -> l >= 0)
            .orElse(Integer.MAX_VALUE);
    }
    
    /**
     * Get the model of the query based on exportParams
     * @param exportParams export params of the query, available via {@link QueryHelper#getExportParams(Query)}
     * @return the model of the query
     */
    public String getModelForQuery(Map<String, Object> exportParams)
    {
        if (exportParams.containsKey("model"))
        {
            return (String) exportParams.get("model");
        }
        return null;
    }

    /**
     * Get the values of the query based on exportParams
     * @param exportParams export params of the query, available via {@link QueryHelper#getExportParams(Query)}
     * @return the values of the query
     */
    public Map<String, Object> getValuesForQuery(Map<String, Object> exportParams)
    {
        if (exportParams.containsKey("values"))
        {
            @SuppressWarnings("unchecked")
            Map<String, Object> values = (Map<String, Object>) exportParams.get("values");
            return values;
        }
        return null;
    }

    /**
     * Get the contextual parameters of the query based on exportParams
     * @param exportParams export params of the query, available via {@link QueryHelper#getExportParams(Query)}
     * @return the contextual parameters of the query
     */
    public Map<String, Object> getContextualParametersForQuery(Map<String, Object> exportParams)
    {
        if (exportParams.containsKey("contextualParameters"))
        {
            @SuppressWarnings("unchecked")
            Map<String, Object> contextualParameters = (Map<String, Object>) exportParams.get("contextualParameters");
            return contextualParameters;
        }
        return null;
    }

    /**
     * Get the sort of the query based on exportParams
     * @param exportParams export params of the query, available via {@link QueryHelper#getExportParams(Query)}
     * @return the sort of the query
     */
    public List<ContentSearchSort> getSortForQuery(Map<String, Object> exportParams)
    {
        if (exportParams.containsKey("sort"))
        {
            String sortString = (String) exportParams.get("sort");
            List<Object> sortlist = _jsonUtils.convertJsonToList(sortString);
            return sortlist.stream()
                .map(Map.class::cast)
                .map(map -> new ContentSearchSort((String) map.get("property"), SortOrder.valueOf((String) map.get("direction"))))
                .collect(Collectors.toList());
        }
        return null;
    }
    
    /**
     * Get the Content Types of the query based on exportParams
     * @param exportParams export params of the query, available via {@link QueryHelper#getExportParams(Query)}
     * @return the Content Types of the query
     */
    public List<String> getContentTypesForQuery(Map<String, Object> exportParams)
    {
        Map<String, Object> values = getValuesForQuery(exportParams);
        if (values.containsKey("contentTypes"))
        {
            @SuppressWarnings("unchecked")
            List<String> contentTypes = (List<String>) values.get("contentTypes");
            return contentTypes;
        }
        return null;
    }
    
    /**
     * Get the solr query string of the query based on exportParams
     * @param exportParams export params of the query, available via {@link QueryHelper#getExportParams(Query)}
     * @return the solr query string of the query
     */
    public String getQueryStringForQuery(Map<String, Object> exportParams)
    {
        Map<String, Object> values = getValuesForQuery(exportParams);
        if (values.containsKey("query"))
        {
            return (String) values.get("query");
        }
        return null;
    }
    
    /**
     * Get the search model of the query based on exportParams
     * @param exportParams export params of the query, available via {@link QueryHelper#getExportParams(Query)}
     * @return the search model of the query
     */
    public String getSearchModeForQuery(Map<String, Object> exportParams)
    {
        if (exportParams.containsKey("searchMode"))
        {
            return (String) exportParams.get("searchMode");
        }
        return "simple";
    }
    
    /**
     * Get the export params of the query
     * @param query the query
     * @return the export params of the query
     */
    public Map<String, Object> getExportParams(Query query)
    {
        String queryContent = query.getContent();
        Map<String, Object> jsonMap = _jsonUtils.convertJsonToMap(queryContent);
        if (jsonMap.containsKey("exportParams"))
        {
            Object exportParams = jsonMap.get("exportParams");
            if (exportParams instanceof Map<?, ?>)
            {
                @SuppressWarnings("unchecked")
                Map<String, Object> exportParamsObject = (Map<String, Object>) exportParams;
                return exportParamsObject;
            }
        }
        return new HashMap<>();
    }
}
