/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.query;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;

import org.ametys.cms.search.cocoon.SearchGenerator;
import org.ametys.core.authentication.AuthenticateAction;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.queriesdirectory.QueryDAO;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * Get query parameters from query
 *
 */
public class GetQueryParametersAction extends ServiceableAction
{
    
    private static final String _XSLT_BASE_LOCATION = "context://WEB-INF/stylesheets/export";
    
    private AmetysObjectResolver _resolver;

    /** The Json utils */
    private JSONUtils _jsonUtils;

    private CurrentUserProvider _currentUserProvide;

    private QueryDAO _queryDAO;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
        _currentUserProvide = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _queryDAO = (QueryDAO) smanager.lookup(QueryDAO.ROLE);
    }

    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        Request request = ObjectModelHelper.getRequest(objectModel);

        request.setAttribute(AuthenticateAction.REQUEST_ATTRIBUTE_INTERNAL_ALLOWED, true);
        
        String queryId = request.getParameter("queryId");
        String xslt = parameters.getParameter("xslt", "");
        
        UserIdentity user = _currentUserProvide.getUser();

        Query query = _resolver.resolveById(queryId);
        
        if (!_queryDAO.canRead(user, query))
        {
            throw new AccessDeniedException("The user " + user + " is not allowed to access the query with id " + queryId);
        }
        
        String xsltFile = StringUtils.defaultString(getXsltLocation(resolver, xslt));
        
        Map<String, Object> content = _jsonUtils.convertJsonToMap(query.getContent());
        @SuppressWarnings("unchecked")
        Map<String, Object> params = (Map<String, Object>) content.get("exportParams");
        
        String versionLabel = parameters.getParameter("versionLabel");
        if (StringUtils.isNotBlank(versionLabel))
        {
            params.put(SearchGenerator.CONTENT_VERSION_LABEL, versionLabel);
        }
        
        String pluginName = (String) content.get("exportXMLUrlPlugin");
        String exportUrl = (String) content.get("exportXMLUrl");

        result.put("pluginName", pluginName != null ? pluginName : "cms");
        result.put("exportUrl", exportUrl != null ? exportUrl : "search/export.xml");
        
        result.put("parameters", InternalEncoder.encode(_jsonUtils.convertObjectToJson(params)));
        result.put("xsltFile", xsltFile);
        return result;
    }
    
    /**
     * Get the XSLT location.
     * @param resolver the source resolver.
     * @param xslt the requested xslt.
     * @return the XSLT file location.
     * @throws IOException if an error occurs resolving the XSLT.
     */
    protected String getXsltLocation(SourceResolver resolver, String xslt) throws IOException
    {
        Source xsltSource = null;
        try
        {
            if (StringUtils.isNotBlank(xslt))
            {
                String location = _XSLT_BASE_LOCATION + "/" + xslt;
                xsltSource = resolver.resolveURI(location);
                
                if (xsltSource.exists())
                {
                    return location;
                }
            }
        }
        finally
        {
            if (xsltSource != null)
            {
                resolver.release(xsltSource);
            }
        }
        
        return null;
    }
    
    /**
     * Helper defining methods used to partially encode the query string of
     * internal (cocoon) wrapped requests.
     */
    private static final class InternalEncoder
    {
        private static final char[] NEED_ENCODING = {'%', '&', '=', '?', ' ', '+'};

        private InternalEncoder()
        {
            // Do not instantiate
        }
        
        /**
         * Partially encode a string given the NEED_ENCODING array.
         * @param str The string to encode
         * @return encode String.
         */
        public static String encode(String str)
        {
            boolean changed = false;
            final char escape = '%';
            
            int len = str.length();
            StringBuilder sb = new StringBuilder(len);
            
            for (int i = 0; i < len; i++)
            {
                char ch = str.charAt(i);
                if (ArrayUtils.contains(NEED_ENCODING, ch))
                {
                    sb.append(escape);
                    sb.append(Integer.toHexString(ch));
                    changed = true;
                }
                else
                {
                    sb.append(ch);
                }
            }
            
            return changed ? sb.toString() : str;
        }
    }
}

