/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.skincommons;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.path.PathUtils;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinsManager;

/**
 * DAO for skin edition.
 */
public abstract class AbstractCommonSkinDAO extends AbstractLogEnabled implements Serviceable, Component
{
    private static final String __TEMP_MODE = "temp";
    private static final String __WORK_MODE = "work";
    
    /** The site manager */
    protected SiteManager _siteManager;
    /** The skin helper */
    protected SkinEditionHelper _skinHelper;
    /** The lock manager */
    protected SkinLockManager _lockManager;
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    /** The user manager */
    protected UserManager _userManager;
    /** The skin manager */
    protected SkinsManager _skinsManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _skinsManager = (SkinsManager) smanager.lookup(SkinsManager.ROLE);
        _skinHelper = (SkinEditionHelper) smanager.lookup(SkinEditionHelper.ROLE);
        _lockManager = (SkinLockManager) smanager.lookup(SkinLockManager.ROLE);
        _userProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
    }
    
    /**
     * Get the URI to preview a site
     * @param siteName the site name
     * @param lang the site langage
     * @return The uri
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public String getPreviewURI(String siteName, String lang)
    {
        Site site = _siteManager.getSite(siteName);
        
        String siteLangage = !StringUtils.isEmpty(lang) ? lang : site.getSitemaps().iterator().next().getName();
        
        if (site.getSitemap(siteLangage).hasChild("index"))
        {
            return siteLangage + "/index.html";
        }
        else
        {
            Iterator<? extends Page> it = site.getSitemap(siteLangage).getChildrenPages().iterator();
            
            if (it.hasNext())
            {
                String path = it.next().getPathInSitemap();
                return siteLangage + "/" + path + ".html";
            }
        }
        
        return null;
    }
    
    /**
     * Check if there is unsaved or uncomitted changes
     * @param skinName The skin name
     * @return The result
     * @throws IOException if an error occurs while manipulating files
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> checkUnsaveModifications (String skinName) throws IOException
    {
        Map<String, Object> result = new HashMap<>();
                
        checkUserRight(skinName);
        
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        Path workDir = _skinHelper.getWorkDirectory(skinName);
        Path skinDir = _skinHelper.getSkinDirectory(skinName);
        
        if (!_lockManager.canWrite(tempDir))
        {
            // Unecessary to check unsave modifications, the user has no right anymore
            return result;
        }
        
        long lastModifiedLock = _lockManager.lastModified(tempDir).getTime();
        if (lastModifiedLock <= Files.getLastModifiedTime(workDir).toMillis())
        {
            if (Files.getLastModifiedTime(workDir).toMillis() > Files.getLastModifiedTime(skinDir).toMillis())
            {
                result.put("hasUncommitChanges", true);
            }
            else
            {
                PathUtils.deleteDirectory(workDir);
            }
        }
        else if (lastModifiedLock >= Files.getLastModifiedTime(skinDir).toMillis())
        {
            // The modifications were not saved
            result.put("hasUnsaveChanges", true);
        }
        
        return result;
    }
    
    /**
     * Save the current skin into the skin work folder
     * @param skinName The name of the skin
     * @param quit True if the temp directory can be removed
     * @return The name of the skin
     * @throws IOException if an error occurs while manipulating files
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> saveChanges(String skinName, boolean quit) throws IOException
    {
        checkUserRight(skinName);
        
        Map<String, Object> lockInfos = _checkLock(skinName);
        if (!lockInfos.isEmpty())
        {
            return lockInfos;
        }
        
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        Path workDir = _skinHelper.getWorkDirectory(skinName);
        
        if (Files.exists(workDir))
        {
            // Delete work directory
            _skinHelper.deleteQuicklyDirectory(workDir);
        }
        
        if (quit)
        {
            // Move to work directory
            PathUtils.moveDirectory(tempDir, workDir);
            
            // Remove lock
            PathUtils.deleteQuietly(workDir.resolve(".lock"));
        }
        else
        {
            // Do a copy in work directory
            PathUtils.copyDirectory(tempDir, workDir, file -> !file.getFileName().toString().equals(".lock"), false);
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("skinName", skinName);
        return result;
    }
    
    /**
     * Check user rights and throws {@link AccessDeniedException} if it is not authorized
     * @param skinName the skin name
     */
    protected abstract void checkUserRight(String skinName);
    
    /**
     * Commit the changes made to the skin
     * @param skinName the name of the skin
     * @param quit True to remove the temporary directory
     * @return A map with information
     * @throws Exception if an error occurs when committing the skin changes
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> commitChanges(String skinName, boolean quit) throws Exception
    {
        checkUserRight(skinName);
        
        Skin skin = _skinsManager.getSkin(skinName);
        if (!skin.isModifiable())
        {
            throw new IllegalStateException("The skin '" + skinName + "' is not modifiable and thus cannot be modified.");
        }
        
        Map<String, Object> lockInfos = _checkLock(skinName);
        if (!lockInfos.isEmpty())
        {
            return lockInfos;
        }
        
        Path skinDir = _skinHelper.getSkinDirectory(skinName);
        
        // Do a backup (move skin directory to backup directory)
        Path backupDir = _skinHelper.createBackupFile(skinName);
        
        // Move temporary version to current skin
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        PathUtils.moveDirectory(tempDir, skinDir);
        
        Path workDir = _skinHelper.getWorkDirectory(skinName);
        if (quit)
        {
            // Delete work version
            _skinHelper.deleteQuicklyDirectory(workDir);
        }
        else
        {
            // Do a copy in work directory
            PathUtils.copyDirectory(skinDir, workDir, file -> !file.getFileName().toString().equals(".lock"), true);
            
            // Do a copy in temp directory
            PathUtils.copyDirectory(skinDir, tempDir, true);
        }
        
        // Delete lock file
        PathUtils.deleteQuietly(skinDir.resolve(".lock"));
        
        // Invalidate caches
        _skinHelper.invalidateCaches(skinName);
        _skinHelper.invalidateSkinCatalogues(skinName);
        
        // Remove old backup (keep only the 5 last backup)
        _skinHelper.deleteOldBackup(skinName, 5);
        
        Map<String, Object> result = new HashMap<>();
        
        result.put("backupFilename", backupDir.getFileName().toString());
        
        String mailSysAdmin = Config.getInstance().getValue("smtp.mail.sysadminto");
        if (!mailSysAdmin.isEmpty())
        {
            result.put("adminEmail", mailSysAdmin);
        }
        
        return result;
    }
    
    /**
     * Cancel the current modification to the skin
     * @param skinName The name of the skin
     * @param workVersion True it is the work version, false for the temp version
     * @param toolId the id of the tool
     * @return True if some changes were canceled
     * @throws IOException if an error occurs while manipulating files
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> cancelChanges(String skinName, boolean workVersion, String toolId) throws IOException
    {
        checkUserRight(skinName);
        
        Map<String, Object> lockInfos = _checkLock(skinName);
        if (!lockInfos.isEmpty())
        {
            return lockInfos;
        }
        
        String modelBeforeCancel = _skinHelper.getTempModel(skinName);
        
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        if (Files.exists(tempDir))
        {
            // Delete current temp version
            _skinHelper.deleteQuicklyDirectory(tempDir);
        }
        
        Path workDir = _skinHelper.getWorkDirectory(skinName);
        if (workVersion && Files.exists(workDir))
        {
            // Back from the last saved work version
            PathUtils.copyDirectory(workDir, tempDir);
        }
        else
        {
            if (Files.exists(workDir))
            {
                // Delete work version
                _skinHelper.deleteQuicklyDirectory(workDir);
            }
            
            // Back from current skin
            Path skinDir = _skinHelper.getSkinDirectory(skinName);
            PathUtils.copyDirectory(skinDir, tempDir);
        }
        
        String modelAfterCancel = _skinHelper.getTempModel(skinName);
        
        _lockManager.updateLockFile(tempDir, !toolId.isEmpty() ? toolId : "uitool-skineditor");
        
        // Invalidate i18n.
        _skinHelper.invalidateTempSkinCatalogues(skinName);
        
        Map<String, Object> result = new HashMap<>();
        result.put("hasChanges", modelAfterCancel == null || !modelAfterCancel.equals(modelBeforeCancel));
        return result;
    }
    
    private Map<String, Object> _checkLock (String skinName) throws IOException
    {
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        
        if (!_lockManager.canWrite(tempDir))
        {
            Map<String, Object> result = new HashMap<>();
            
            UserIdentity lockOwner = _lockManager.getLockOwner(tempDir);
            User user = _userManager.getUser(lockOwner.getPopulationId(), lockOwner.getLogin());

            result.put("isLocked", true);
            result.put("lockOwner", user != null ? user.getFullName() + " (" + lockOwner + ")" : lockOwner);
            result.put("success", false);
            
            return result;
        }
        
        // Not lock
        return Collections.EMPTY_MAP;
    }
    
    /**
     * Get the model for the skin
     * @param siteName the site name. Can be null if skinName is not null.
     * @param skinName the skin name. Can be null if siteName is not null.
     * @param mode the edition mode. Can be null.
     * @return the model's name or null if there is no model for this skin
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public String getSkinModel(String siteName, String skinName, String mode)
    {
        String skinId = _getSkinName (siteName, skinName);
        
        if (__TEMP_MODE.equals(mode))
        {
            return _skinHelper.getTempModel(skinId);
        }
        else if (__WORK_MODE.equals(mode))
        {
            return _skinHelper.getWorkModel(skinId);
        }
        else
        {
            return _skinHelper.getSkinModel(skinId);
        }
    }
    
    private String _getSkinName (String siteName, String skinName)
    {
        if (StringUtils.isEmpty(skinName) && StringUtils.isNotEmpty(siteName))
        {
            Site site = _siteManager.getSite(siteName);
            return site.getSkinId();
        }
        
        return skinName;
    }

    /**
     * Get lock informations on a skin
     * @param siteName the site name. Can be null if skinName is not null.
     * @param skinName the skin name. Can be null if siteName is not null.
     * @return Informations about the lock
     * @throws IOException if an error occurs
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getLock(String siteName, String skinName) throws IOException
    {
        String skinId = _getSkinName(siteName, skinName);
        
        checkUserRight(skinId);
        
        Map<String, Object> result = new HashMap<>();
        
        
        Path tempDir = _skinHelper.getTempDirectory(skinId);
        if (Files.exists(tempDir) && _lockManager.isLocked(tempDir))
        {
            UserIdentity lockOwner = _lockManager.getLockOwner(tempDir);
            User user = _userManager.getUser(lockOwner.getPopulationId(), lockOwner.getLogin());

            result.put("isLocked", !_userProvider.getUser().equals(lockOwner));
            result.put("lockOwner", user != null ? user.getFullName() + " (" + lockOwner + ")" : lockOwner);
            result.put("lastModified", DateUtils.dateToString(_lockManager.lastModified(tempDir)));
            result.put("toolId", _lockManager.getLockTool(tempDir));
        }
        else
        {
            result.put("isLocked", false);
        }
        
        Path workDir = _skinHelper.getWorkDirectory(skinId);
        if (Files.exists(workDir))
        {
            result.put("lastSave", DateUtils.dateToString(new Date(Files.getLastModifiedTime(workDir).toMillis())));
        }
        
        return result;
    }
    
    /**
     * Revert changes and back to last work version or to current skin
     * @param skinName The skin name
     * @param workVersion true to get back the work version
     * @return The skin name in case of success or lock infos if the skin is locked.
     * @throws IOException if an error occurs
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> clearModifications (String skinName, boolean workVersion) throws IOException
    {
        checkUserRight(skinName);
        
        Map<String, Object> lockInfos = _checkLock(skinName);
        if (!lockInfos.isEmpty())
        {
            return lockInfos;
        }
        
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        if (Files.exists(tempDir))
        {
            // Delete current temp version
            _skinHelper.deleteQuicklyDirectory(tempDir);
        }
        
        if (workVersion)
        {
            Path workDir = _skinHelper.getWorkDirectory(skinName);
            if (Files.exists(workDir))
            {
                // Delete current work version
                _skinHelper.deleteQuicklyDirectory(workDir);
            }
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("skinName", skinName);
        return result;
    }
}
