/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.skincommons;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.List;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.path.PathUtils;

/**
 * Component to manage lock on skin directories.
 *
 */
public class SkinLockManager extends AbstractLogEnabled implements Component, ThreadSafe, Serviceable
{
    /** The Avalon role name */
    public static final String ROLE = SkinLockManager.class.getName();
    
    private CurrentUserProvider _userProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _userProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Determines if the current user can write in skin directory
     * @param file The skin directory
     * @return <code>true</code> if the skin directory is not locked by another user
     * @throws IOException If an error occurred
     */
    public boolean canWrite (Path file) throws IOException
    {
        return getLockOwner(file).equals(_userProvider.getUser());
    }
    
    /**
     * Create or update .lock file
     * @param file The skin directory
     * @param toolId The id of tool responsible for lock.
     * @throws IOException If an error occurred
     */
    public void updateLockFile (Path file, String toolId) throws IOException
    {
        Path lockFile = file.resolve(".lock");
        if (Files.exists(lockFile))
        {
            Files.delete(lockFile);
        }
        
        String content = UserIdentity.userIdentityToString(_userProvider.getUser()) + "\n"
                + DateUtils.getISODateTimeFormatter().format(ZonedDateTime.now()) + "\n"
                + toolId;
        Files.writeString(lockFile, content, StandardCharsets.UTF_8); 
    }
    
    /**
     * Remove lock file
     * @param file The skin directory
     */
    public void unlock (Path file)
    {
        Path lockFile = file.resolve(".lock");
        if (Files.exists(lockFile))
        {
            PathUtils.deleteQuietly(lockFile);
        }
    }
    
    /**
     * Determines if the skin directory is locked
     * @param file The skin directory
     * @return <code>true</code> if the skin directory is locked
     */
    public boolean isLocked (Path file)
    {
        Path lockFile = file.resolve(".lock");
        return Files.exists(lockFile);
    }
    
    /**
     * Get lock owner
     * @param file The skin directory
     * @return The lock owner or null.
     * @throws IOException if an error occurs while manipulating files
     */
    public UserIdentity getLockOwner(Path file) throws IOException
    {
        Path lockFile = file.resolve(".lock");
        if (!Files.exists(lockFile))
        {
            return null;
        }
        
        List<String> lines = Files.readAllLines(lockFile, StandardCharsets.UTF_8);
        if (!lines.isEmpty())
        {
            return UserIdentity.stringToUserIdentity(lines.get(0));
        }
        
        return null;
    }
    
    /**
     * Get tool responsible for lock
     * @param file The skin directory
     * @return The tool responsible for lock or <code>null</code>
     * @throws IOException if an error occurs while manipulating files
     */
    public String getLockTool(Path file)  throws IOException
    {
        Path lockFile = file.resolve(".lock");
        if (!Files.exists(lockFile))
        {
            return null;
        }
        
        List<String> lines = Files.readAllLines(lockFile, StandardCharsets.UTF_8);
        if (!lines.isEmpty() && lines.size() > 2)
        {
            return lines.get(2);
        }
            
        return null;
    }
    
    /**
     * Get the last modified date
     * @param file The skin directory
     * @return the last modified date
     * @throws IOException if an error occurs while manipulating files
     */
    public Date lastModified(Path file) throws IOException
    {
        Path lockFile = file.resolve(".lock");
        if (!Files.exists(lockFile))
        {
            return null;
        }
        
        List<String> lines = Files.readAllLines(lockFile, StandardCharsets.UTF_8);
        if (!lines.isEmpty() && lines.size() > 1)
        {
            return Date.from(ZonedDateTime.parse(lines.get(1), DateTimeFormatter.ISO_DATE_TIME).toInstant());
        }

        return null;
    }
}
