/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * DAO for manipulating a skin with a model
 */
Ext.define("Ametys.plugins.skinfactory.skin.SkinDAO",
{
    singleton: true,
	
	SERVER_ROLE: "org.ametys.skinfactory.skins.SkinDAO",

    constructor: function ()
    {
        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method affectDesign
         * This calls the method 'affectDesign' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Affect a new design configuration.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.skinName The skin name.
         * @param {String} parameters.designId The design id.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue the information on the design, or the error. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "affectDesign",
            callback: {
                handler: this._affectDesignCb
            },
            waitMessage: "{{i18n PLUGINS_SKINFACTORY_AFFECT_DESIGN_WAIT_MSG}}",
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_AFFECT_DESIGN_ERROR}}",
                category: Ext.getClassName(this)
            }
         });

        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method changeModelChange the model of a skin
         * This calls the method 'changeModel' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Change the model of a skin.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.modelName The model name.
         * @param {String} parameters.skinName The skin name.
         * @param {boolean} parameters.useDefaults True to use default model parameters.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The skin parameters, or the error informations. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "changeModel",
            callback: {
                handler: this._changeModelCb
            },
            waitMessage: "{{i18n PLUGINS_SKINFACTORY_CHANGE_MODEL_WAIT_MSG}}",
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_CHANGE_MODEL_ERROR}}",
                category: Ext.getClassName(this)
            }
         });

         /**
          * @callable
          * @member Ametys.plugins.skinfactory.skin.SkinDAO
          * @method getLanguages
          * This calls the method 'getLanguages' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
          * Get the languages available on a site.
          * @param {Object[]} parameters The parameters to transmit to the server method.
          * @param {String} parameters.siteName The site name.
          * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
          * @param {Object} callback.returnedValue The languages informations (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
          * @param {Object} callback.arguments Other arguments specified in option.arguments                 
          * @param {Object} [options] Advanced options for the call.
          * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
          * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
          * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
          * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
          * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
          * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
          * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
          */
          this.addCallables(
          {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "getLanguages",
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_GET_LANGUAGES_ERROR}}",
                category: Ext.getClassName(this)
            }
          });

        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method getColors
         * This calls the method 'getColors' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Get the colors of a model and its theme for a site.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.siteName The site name.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The colors and theme informations. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "getColors",
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_GET_COLOR_THEME_ERROR}}",
                category: Ext.getClassName(this)
            }
         });
            
        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method getCssStyleItems
         * This calls the method 'getCssStyleItems' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Get the css style items used by a site.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.siteName The site name.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The css style items. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "getCssStyleItems",
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_GET_CSS_STYLE_ITEMS_ERROR}}",
                category: Ext.getClassName(this)
            }
         });
            
        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method getParametersValues
         * This calls the method 'getParametersValues' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Get the parameters of the skin of a site
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.siteName The site name.
         * @param {String[]} parameters.paramIds If not null, specify the ids of the parameters to retrieve.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The parameters (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "getParametersValues",
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_GET_PARAMETER_ERROR}}",
                category: Ext.getClassName(this)
            }
         });
            
        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method openSkin
         * This calls the method 'openSkin' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Open the skin of a site for edition.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.siteName The site name.
         * @param {String} parameters.mode The open mode.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The skin id, or an error message. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "openSkin",
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_TOOL_OPEN_SKIN_ERROR}}",
                category: Ext.getClassName(this)
            }
         });
        
        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method restoreDefaults
         * This calls the method 'restoreDefaults' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Restore the default parameters for a skin.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.skinName The skin name.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The skin informations, or an error code. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "restoreDefaults",
            callback: {
                handler: this._restoreDefaultsCb
            },
            waitMessage: "{{i18n PLUGINS_SKINFACTORY_RESTORE_DEFAULTS_WAIT_MSG}}",
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_RESTORE_DEFAULTS_ERROR}}",
                category: Ext.getClassName(this)
            }
         });
        
        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method updateColorTheme
         * This calls the method 'updateColorTheme' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Set the theme used by a skin.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.skinName The skin name.
         * @param {String} parameters.themeId The theme id.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The theme informations, or an error code. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "updateColorTheme",
            callback: {
                handler: this._updateColorThemeCb
            },
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_UPDATE_COLOR_THEME_ERROR}}",
                category: Ext.getClassName(this)
            }
         });
        
        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method updateParameter
         * This calls the method 'updateParameter' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Update a parameter of the skin
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.skinName The skin name.
         * @param {String} parameters.lang The current language.
         * @param {String} parameters.parameterId The parameter id to update.
         * @param {String} parameters.value The new value for the parameter.
         * @param {boolean} parameters.uploaded True if the file was uploaded.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The skin parameters updated, or an error code. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "updateParameter",
            callback: {
                handler: this._updateParameterCb
            },
            waitMessage: false,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_UPDATE_PARAMETER_ERROR}}",
                category: Ext.getClassName(this)
            }
         });
         
         /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method uploadLocalImage
         * This calls the method 'uploadLocalImage' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Upload a local image and update a parameter of the skin
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.skinName The skin name.
         * @param {String} parameters.parameterId The parameter id to update.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The skin parameters updated, or an error code. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "uploadLocalImage",
            callback: {
                handler: this._uploadLocalImageCb
            },
            waitMessage: false,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_UPDATE_PARAMETER_ERROR}}",
                category: Ext.getClassName(this)
            }
         });

        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method getGalleryImages
         * This calls the method 'getGalleryImages' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Retrieve the list of images for the skin and parameter
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.skinName The skin name.
         * @param {String} parameters.paramId The parameter id.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The map of images informations (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "getGalleryImages",
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_IMAGESGALLERY_ERROR}}",
                category: Ext.getClassName(this)
            }
         });

        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method getGalleryVariants
         * This calls the method 'getGalleryVariants' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Retrieve the list of gallery variants available for the specified skin and parameter.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.skinName The skin name.
         * @param {String} parameters.paramId The parameter id.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object[]} callback.returnedValue The list of gallery variants (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "getGalleryVariants",
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_VARIANTGALLERY_ERROR}}",
                category: Ext.getClassName(this)
            }
         });

        /**
         * @callable
         * @member Ametys.plugins.skinfactory.skin.SkinDAO
         * @method getThemeColors
         * This calls the method 'getThemeColors' of the server DAO 'org.ametys.skinfactory.skins.SkinDAO'.
         * Retrieve the list of themes' colors for a site.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.siteName The site name.
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object[]} callback.returnedValue The model's themes colors (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
         this.addCallables(
         {
            role: "org.ametys.skinfactory.skins.SkinDAO",
            methodName: "getThemeColors",
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_COLORMENU_THEME_COLORS_ERROR}}",
                category: Ext.getClassName(this)
            }
         });
    },

    /**
     * Callback after affecting a design to a skin. Send a MODIFIED Message on skin
     * @param {Object} response The informations from the server
     * @param {Object} args The initial arguments
     * @param {Object} params The callback parameters
     * @private
     */
    _affectDesignCb: function(response, args, params)
    {
        if (!response.unknownModel)
        {
            var skinId = params[0];
            
            Ext.create("Ametys.message.Message", {
        		type: Ametys.message.Message.MODIFIED,
        		targets: {
        		    id: Ametys.message.MessageTarget.SKIN,
        			parameters: {
        				id: skinId
        			}
        		}
        	});
        }
    },

    /**
     * Callback after changing the model affected to a skin. Send a MODIFIED Message on skin
     * @param {Object} response The informations from the server
     * @param {Object} args The initial arguments
     * @param {Object} params The callback parameters
     * @private
     */
    _changeModelCb: function(response, args, params)
    {
        var skinId = params[1];
        Ext.create("Ametys.message.Message",
        {
            type: Ametys.message.Message.MODIFIED,
            targets:
            {
                id: Ametys.message.MessageTarget.SKIN,
                parameters:
                {
                    id: skinId
                }
            }
        });
    },

    /**
     * Callback after restoring the default parameters of a skin. Send a MODIFIED Message on skin
     * @param {Object} response The informations from the server
     * @param {Object} args The initial arguments
     * @param {Object} params The callback parameters
     * @private
     */
    _restoreDefaultsCb: function(response, args, params)
    {
        if (!response.unknownModel && !response.isLocked)
        {
            var skinId = params[0];
            
            Ext.create("Ametys.message.Message", {
        		type: Ametys.message.Message.MODIFIED,
        		targets: {
        		    id: Ametys.message.MessageTarget.SKIN,
        			parameters: {
        				id: skinId
        			}
        		}
        	});
        }
    },

    /**
     * Callback after updating the color theme of a skin. Send a MODIFIED Message on skin
     * @param {Object} response The informations from the server
     * @param {Object} args The initial arguments
     * @param {Object} params The callback parameters
     * @private
     */
    _updateColorThemeCb: function(response, args, params)
    {
        if (!response.unknownModel && !response.isLocked)
        {
            var skinId = params[0];
            
            Ext.create("Ametys.message.Message", {
        		type: Ametys.message.Message.MODIFIED,
        		targets: {
        		    id: Ametys.message.MessageTarget.SKIN_COLOR_THEME,
        			parameters: {
        				id: skinId
        			}
        		}
        	});
        }
    },

    /**
     * Callback after updating a parameter of a skin. Send a MODIFIED Message on skin
     * @param {Object} response The informations from the server
     * @param {Object} args The initial arguments
     * @param {Object} params The callback parameters
     * @private
     */
    _updateParameterCb: function(response, args, params)
    {
        if (!response.unknownModel && !response.isLocked)
        {
            var skinId = params[0];
            
            Ext.create("Ametys.message.Message", {
        		type: Ametys.message.Message.MODIFIED,
        		targets: {
        		    id: Ametys.message.MessageTarget.SKIN_PARAMETER,
        			parameters: {
        				id: skinId
        			}
        		}
        	});
        }
    },
    
    /**
     * Callback after uploading a image and updating a parameter of a skin. Send a MODIFIED Message on skin
     * @param {Object} response The informations from the server
     * @param {Object} args The initial arguments
     * @param {Object} params The callback parameters
     * @private
     */
    _uploadLocalImageCb: function(response, args, params)
    {
        if (!response.unknownModel && !response.isLocked)
        {
            var skinId = params[0];
            
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MODIFIED,
                targets: {
                    id: Ametys.message.MessageTarget.SKIN_PARAMETER,
                    parameters: {
                        id: skinId,
                        uploaded: true
                    }
                }
            });
        }
    }
    
});
