/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * @private
 * This tool displays the available skins and models 
 */
Ext.define('Ametys.plugins.skinfactory.skin.SkinTool', {
    extend: 'Ametys.plugins.web.skin.SkinTool',

    /**
     * @private
     * @property {Ext.XTemplate} _filterHintTpl Template used for filter hint
     */
    _filterHintTpl: Ext.create('Ext.Template',
            '{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_FILTER_HINT1}}<b>{label}</b> <em> ({id})</em>{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_FILTER_HINT2}}',
            '<br/>{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_FILTER_HINT_LINK}}'
    ),

    /**
     * Called when a message is received
     * @param {Ametys.message.Message} message The message received.
     * @private
     */
    _onMessage: function(message)
    {
        if (message.getTarget(new RegExp('^(' + Ametys.message.MessageTarget.SKIN + "|" + Ametys.message.MessageTarget.SKINMODEL + "|" + Ametys.message.MessageTarget.SITE + ')$')) != null)
        {
            this.showOutOfDate();
        }
        
        var skinFilter = message.getTarget(Ametys.message.MessageTarget.SKINFILTER);
        if (skinFilter != null)
        {
            if (message.getType() == Ametys.message.Message.CREATED)
            {
                this._applyFilter(skinFilter.getParameters());
            }
            else if (message.getType() == Ametys.message.Message.DELETED)
            {
                this._removeFilter();
            }
        }
    },
    
    _getDockedItems: function ()
    {
    	return [{
    		dock: 'top',
    		xtype: 'button',
    		ui: 'tool-hintmessage',
            itemId: 'filter-hint',
            hidden: true,
            text: '',
            handler: function () {
	    		Ext.create("Ametys.message.Message", {
	                type: Ametys.message.Message.DELETED,
	                targets: {
	                    id: Ametys.message.MessageTarget.SKINFILTER,
	                    parameters: {}
	                }
	            });
	    	}
    	}]
    },

    _getItems: function ()
    {
        return [ this._createDataView("models", "Ametys.plugins.skinfactory.skin.SkinModel", "{{i18n PLUGINS_SKINFACTORY_ADMINISTRATOR_SKINS_MODELS_TITLE}}", Ametys.message.MessageTarget.SKINMODEL), ...this.callParent(arguments)];
    },

    refresh: function ()
    {
        this.showRefreshing();
        Ametys.plugins.skinfactory.model.SkinModelDAO.getSkinsAndModels([true], this._refreshCb, {
            waitMessage: false,
            scope: this
        });
    },

    /**
     * Callback on tool refresh
     * @param {Object} response The list of models and skins
     * @param {Object[]} response.models The list of models
     * @param {Object[]} response.skins The list of skins
     * @private
     */
    _refreshCb: function (response)
    {
        var modelStore = this.getContentPanel().getComponent("dataview-models").getComponent("models").getStore();
        modelStore.removeAll();

        if (response.models)
        {
            modelStore.add(response.models);
        }
        
        this.callParent(arguments);
    },

    /**
     * Set a filter to display only a model and its skins
     * @param {Object} modelData The model to filter
     * @param {String} modelData.id The id of the model
     * @param {String} modelData.label The label of the model
     * @private
     */
    _applyFilter: function (modelData)
    {
        var filterModel = new Ext.util.Filter({
            filterFn: function(item)
            {
                if (item.data.type == "model")
                {
                    // Filter the models by data.id
                    return item.data.id == modelData.id;
                }
                else if (item.data.type == "skin" && item.data.model)
                {
                    // Filter the skins by data.model.id
                    return item.data.model.id == modelData.id;
                }
                return false;
            }
        });
        
        Ext.Array.each (this._dataViews, function (dv) {
        	var total = dv.getStore().getCount();
        	dv.getStore().addFilter(filterModel);
        	var count = dv.getStore().getCount();
        	dv.ownerCt.setTitle (dv._title + " (" + count + "/" + total + ")");
        })

        var filterHint = this.getContentPanel().getComponent("filter-hint");
        filterHint.setText(this._filterHintTpl.apply(modelData))
        filterHint.show();
    },

    /**
     * Remove all filters
     * @private
     */
    _removeFilter: function ()
    {
    	Ext.Array.each (this._dataViews, function (dv) {
        	dv.getStore().clearFilter();
        	
        	var count = dv.getStore().getCount();
    		dv.ownerCt.setTitle (dv._title + " (" + count + ")")
        })
        
        this.getContentPanel().getComponent("filter-hint").hide();
    	
    }
});
