/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 *  Menu widget for loading a set of different parameters 
 */
Ext.define('Ametys.plugins.skinfactory.widgets.menu.ParametersMenu', {
	extend: 'Ext.menu.Menu',
	
	 /**
	 * @cfg {Object[]} parameters The list of parameters displayed by the menu.
	 */
	
	constructor: function (config) 
	{
		this.callParent(arguments);
		this.on("show", this._onMenuShow, this);
		this.on('afterrender', this._onAfterRender, this);
		
		// Create the menu items.
		this._items = {};
		
		var items = [];
		var parameters = this.getInitialConfig("parameters") || {};
		
		for (var i in parameters)
		{
			var cssProperty = parameters[i];
			this._items[i] = this._getMenuItem(i, cssProperty);
			if (this._items[i] != null)
			{
				items.push(this._items[i]);
			}
		}
		
		this.add(items);
	},
	
	/**
	 * @private
	 * Listener invoked after rendering item.
	 * Apply the parent menu UI to the item
	 */
	_onAfterRender: function()
	{
		if (this.parentMenu && this.parentMenu.ui)
		{
			this.setUI(this.parentMenu.ui);
		}
	},
	
	/**
	 * Listener when the menu is showed. Set the checkboxes values.
	 */
	_onMenuShow: function ()
	{
		var currentValues = Ametys.plugins.skinfactory.SkinParametersManager.getParametersValues();
		for (var i in this._items)
		{
			if (currentValues[i] != null)
			{
				var item = this._items[i];
				if (item != null)
				{
					if (item.ctype == 'Ext.menu.CheckItem')
					{
						item.setChecked("underline" == currentValues[i] || "bold" == currentValues[i] || "italic" == currentValues[i]);
					}
				}
			}
		}
	},
	
	/**
	 * Get the menu item corresponding to the id and configuration
	 * @param {String} paramId The parameter id
	 * @param {Object} parameterConfig The configuration
	 * @return {Ext.menu.Item} The new menu item
	 */
	_getMenuItem: function (paramId, parameterConfig)
	{
		if (parameterConfig.type == 'item')
		{
			var cssProperty = parameterConfig.cssProperty;
			var text = parameterConfig.label != '' && parameterConfig.label != paramId ? parameterConfig.label : null;
			switch (cssProperty) 
			{
				case "color":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						menuClass: "Ametys.plugins.skinfactory.widgets.menu.ColorSelectorMenu",
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_FONTCOLOR_LABEL}}",
						iconCls: 'ametysicon-color-text'
					});
				case "background-color":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						menuClass: "Ametys.plugins.skinfactory.widgets.menu.ColorSelectorMenu",
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_BGCOLOR}}", 
						iconCls: 'ametysicon-paint-bucket'
					});
				case "font-family":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_FONTFAMILY_LABEL}}", 
						cssStyle: "font-family",
						iconCls: 'ametysicon-font-family'
					});
				case "font-size":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_FONTSIZE_LABEL}}", 
						cssStyle: "font-size",
						iconCls: 'ametysicon-font-size'
					});
				case "font-weight":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.FontStyleBoldItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_FONTSTYLE_BOLD}}"
					});
				case "font-style":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.FontStyleItalicItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_FONTSTYLE_ITALIC}}"
					});
				case "text-decoration":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.FontStyleUnderlineItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_FONTSTYLE_UNDERLINE}}"
					});
				case "margin-left":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_MARGIN_LEFT}}", 
						cssStyle: "margin",
						iconCls: 'ametysicon-margin-left'
					});
				case "margin-right":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_MARGIN_RIGHT}}", 
						cssStyle: "margin",
						iconCls: 'ametysicon-margin-right'
					});
				case "margin":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_MARGIN}}", 
						cssStyle: "margin",
						iconCls: 'ametysicon-margin'
					});
				case "padding":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_PADDING}}", 
						cssStyle: "margin",
						iconCls: 'ametysicon-padding'
					});
				case "padding-left":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_PADDING_LEFT}}", 
						cssStyle: "margin",
						iconCls: 'ametysicon-padding'
					});
				case "padding-right":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_PADDING_RIGHT}}", 
						cssStyle: "margin",
						iconCls: 'ametysicon-padding'
					});
				case "letter-spacing":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_LETTER_SPACING}}", 
						cssStyle: "letter-spacing",
						iconCls: 'ametysicon-letter-spacing'
					});
				case "line-height":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_LINE_HEIGHT}}", 
						cssStyle: "line-height",
						iconCls: 'ametysicon-text-height'
					});
				case "border-color":
				case "border-bottom-color":
				case "border-left-color":
				case "border-top-color":
				case "border-right-color":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						menuClass: "Ametys.plugins.skinfactory.widgets.menu.ColorSelectorMenu",
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_BORDER_COLOR}}", 
						iconCls: 'ametysicon-painter-border'
					});
				case "border-width":
				case "border-bottom-width":
				case "border-top-width":
				case "border-left-width":
				case "border-right-width":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_BORDER_WIDTH}}", 
						cssStyle: "border-width",
						iconCls: 'ametysicon-border-width'
					});
				case "border-style":
				case "border-bottom-style":
				case "border-top-style":
				case "border-left-style":
				case "border-right-style":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_BORDER_STYLE}}",
						cssStyle: "border-style",
						iconCls: 'ametysicon-border-style'
					});
				case "border-radius":
				case "border-top-left-radius":
				case "border-top-right-radius":
				case "border-bottom-left-radius":
				case "border-bottom-right-radius":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId, 
						text: text || "{{i18n PLUGINS_SKINFACTORY_BORDER_RADIUS}}", 
						cssStyle: "border-radius",
						iconCls: 'ametysicon-rounded-corners'
					});
				case "text-indent":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_TEXTINDENT}}",
						cssStyle: "text-indent",
						iconCls: 'ametysicon-word2'
					});
				case "background-position":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_BACKGROUND_POSITION}}",
						cssStyle: "background-position",
						iconCls: 'ametysicon-background'
					});
				case "background-repeat":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_BACKGROUND_REPEAT}}",
						cssStyle: "background-repeat",
						iconCls: 'ametysicon-background'
					});
				case "text-align":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						menuClass: "Ametys.plugins.skinfactory.widgets.menu.TextAlignMenu",
						cssStyle: "text-align",
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_TEXTALIGN}}",
						iconCls: 'ametysicon-alignment'
					});
				case "height":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_HEIGHT}}",
						cssStyle: "height",
						iconCls: 'ametysicon-height-rectangular'
					});
				case "top":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_TOP}}",
						cssStyle: "top"
					});
				case "bottom":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_BOTTOM}}",
						cssStyle: "bottom"
					});
				case "width":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_WIDTH}}",
						cssStyle: "width",
						iconCls: 'ametysicon-width-rectangular'
					});
				case "left":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_LEFT}}",
						cssStyle: "left"
					});
				case "right":
					return Ext.create("Ametys.plugins.skinfactory.widgets.item.CssStyleWidgetItem", {
						paramId: paramId,
						text: text || "{{i18n PLUGINS_SKINFACTORY_RIGHT}}",
						cssStyle: "right"
					});
				default:
					return null;
			}
		}
		else if (parameterConfig.type == 'image-gallery')
		{
			return Ext.create("Ext.menu.Item", {
				text: parameterConfig.label,
				iconCls: parameterConfig.iconCls,
				icon: !parameterConfig.iconCls ? Ametys.CONTEXT_PATH + parameterConfig.icon : null,
				menu: Ext.create("Ametys.plugins.skinfactory.widgets.gallery.ImagesGallery", {
					parentControl: this.parentControl,
					paramId: paramId, 
					label: parameterConfig.label,
					libraryPath: parameterConfig.libraryPath,
					enableLocalUpload: true
				})
			});
		}
		else if (parameterConfig.type == 'variant-gallery')
		{
			return Ext.create("Ext.menu.Item", {
				text: parameterConfig.label,
				iconCls: parameterConfig.iconCls,
				icon: !parameterConfig.iconCls ? Ametys.CONTEXT_PATH + parameterConfig.icon : null,
				menu: Ext.create("Ametys.plugins.skinfactory.widgets.gallery.VariantsGallery", {
					parentControl: this.parentControl,
					paramId: paramId, 
					label: parameterConfig.label
				})
			});
		}
		else if (parameterConfig.type == 'menu')
		{
			return Ext.create("Ext.menu.Item", {
				text: parameterConfig.label,
				iconCls: parameterConfig.iconCls,
				icon: !parameterConfig.iconCls ? Ametys.CONTEXT_PATH + parameterConfig.icon : null,
				menu: Ext.create("Ametys.plugins.skinfactory.widgets.menu.ParametersMenu", {
					parentControl: this.parentControl,
					parameters: parameterConfig.parameters
				})
			});
		}
	}
});