/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.SourceUtil;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.skinfactory.filefilter.FileFilter;
import org.ametys.skinfactory.parameters.AbstractSkinParameter;
import org.ametys.skinfactory.parameters.CSSColorParameter;
import org.ametys.skinfactory.parameters.CSSParameter;
import org.ametys.skinfactory.parameters.I18nizableTextParameter;
import org.ametys.skinfactory.parameters.ImageParameter;
import org.ametys.skinfactory.parameters.ImageParameter.FileValue;
import org.ametys.skinfactory.parameters.TextParameter;
import org.ametys.skinfactory.parameters.Variant;
import org.ametys.skinfactory.parameters.VariantParameter;
import org.ametys.web.skin.SkinModel;
import org.ametys.web.skin.SkinModelsManager;

/**
 * Manager for skin parameters
 * 
 */
public class SkinFactoryComponent extends AbstractLogEnabled implements Component, ThreadSafe, Serviceable
{
    /** Avalon role */
    public static final String ROLE = SkinFactoryComponent.class.getName();

    /** Pattern for CSS parameters. E.g: color: #EEEEEE /* AMETYS("myuniqueid", "Label", DESCRIPTION_I18N_KEY) *\/; **/ 
    public static final Pattern CSS_PARAM_PATTERN = Pattern.compile("\\s*([^,:\\s]*)\\s*:\\s*([^:;!]*)\\s*(?:!important)?\\s*\\/\\*\\s*AMETYS\\s*\\(\\s*\"([^\"]+)\"\\s*(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?\\)\\s*\\*\\/\\s*;?\\s*", Pattern.MULTILINE); 
    
    /** Pattern for i18n text parameters. E.g: &lt;message key="SKIN_TITLE"&gt;Ametys, Web Java Open Source CMS&lt;!-- Ametys("text.i18n.site.title", "Titre du site") --&gt;&lt;/message&gt; */
    public static final Pattern I18N_PARAM_PATTERN = Pattern.compile("^\\s*<message key=\"([^,:\\s]*)\">([^<]*)<!--\\s*AMETYS\\s*\\(\\s*\"([^\"]+)\"\\s*(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?\\)\\s*--></message>\\s*$", Pattern.MULTILINE); 
    
    /** Pattern for text parameters. E.g: &lt;xsl:variable name="foo"&gt;test&lt;!-- Ametys("variable.foo", "Foo", "Foo") --&gt;&lt;/xsl:variable&gt; */
    public static final Pattern TXT_PARAM_PATTERN = Pattern.compile("^[^>]*>([^<]*)<!--\\s*AMETYS\\s*\\(\\s*\"([^\"]+)\"\\s*(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?\\)\\s*-->.*$", Pattern.MULTILINE); 
    
    private static final Pattern __I18N_CATALOG_LANGUAGE = Pattern.compile("^\\s*<catalogue xml:lang=\"([a-z]{2})\">\\s*$", Pattern.MULTILINE); 
    
    private final Map<String, Map<String, AbstractSkinParameter>> _modelsParameters = new HashMap<>();
    private final Map<String, String> _modelsHash = new HashMap<>();
    
    private SkinModelsManager _modelsManager;
    private SourceResolver _resolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _modelsManager = (SkinModelsManager) smanager.lookup(SkinModelsManager.ROLE);
        _resolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
    }

    /**
     * Remove all model parameters from cache
     */
    public void clearModelsParametersCache()
    {
        _modelsParameters.clear();
    }
    
    /**
     * Get the skin parameter of the given model
     * @param modelName The model name
     * @param paramId The id of the parameter
     * @return The skin parameter or <code>null</code> if it doesn't exist
     */
    public AbstractSkinParameter getModelParamater (String modelName, String paramId)
    {
        Map<String, AbstractSkinParameter> modelParameters = getModelParameters(modelName);
        return modelParameters.get(paramId);
    }
    
    /**
     * Determines if the model was changed
     * @param modelName The model name
     * @return <code>true</code> if the model was changed
     */
    public boolean isModelUpToDate (String modelName)
    {
        String hash = _modelsManager.getModelHash(modelName);
        if (!_modelsHash.containsKey(modelName) || !_modelsHash.get(modelName).equals(hash))
        {
            // The model is not up-to-date, clear cache
            _modelsHash.put(modelName, hash);
            return false;
        }
        return true;
    }
    
    /**
     * Get the skin parameters of a model
     * @param modelName the model name
     * @return skin parameters in a List
     */
    public Map<String, AbstractSkinParameter> getModelParameters (String modelName)
    {
        try
        {
            if (!isModelUpToDate(modelName))
            {
                // The model is not up-to-date, clear cache
                _modelsParameters.remove(modelName);
            }
            
            if (_modelsParameters.containsKey(modelName))
            {
                return _modelsParameters.get(modelName);
            }
            
            _modelsParameters.put(modelName, new LinkedHashMap<>());
            
            Map<String, AbstractSkinParameter> skinParams = _modelsParameters.get(modelName);
            
            Path modelDir = _modelsManager.getModel(modelName).getPath();
    
            // Variants parameters
            skinParams.putAll(_getVariantParameters(modelDir, modelName));
            
            // Images Parameters
            skinParams.putAll(_getImageParameters(modelDir, modelName));
            
            // CSS parameters
            skinParams.putAll(_getCSSParameters(modelDir, modelName));
            
            // I18n parameters
            skinParams.putAll(_getI18nTextParameters(modelDir, modelName));
            
            // Text parameters
            skinParams.putAll(_getTextParameters(modelDir, modelName));
            
            return skinParams;
        }
        catch (IOException e)
        {
            throw new RuntimeException("Cannot find parameters for model " + modelName, e);
        }
    }
    
    private Map<String, VariantParameter> _getVariantParameters (Path modelDir, String modelName) throws IOException
    {
        Map<String, VariantParameter> params = new HashMap<>();
        
        Path variantsDir = modelDir.resolve("model/variants");
        if (Files.exists(variantsDir))
        {
            _findVariantsParameters (variantsDir, params, modelName);
        }
        return params;
    }
    
    private void _findVariantsParameters(Path variantsDir, Map<String, VariantParameter> params, String modelName) throws IOException
    {
        try (Stream<Path> files = Files.list(variantsDir))
        {
            files
                 .filter(FileFilter.getSkinDirectoryFilter())
                 .forEach(child -> 
                 {
                     String filename = child.getFileName().toString();
                     List<Variant> values = _getVariantValues(child, modelName);
                     
                     // avoid false directory such as CVS
                     if (values.size() != 0)
                     {
                         VariantParameter variantParameter = new VariantParameter(filename, new I18nizableText(filename), new I18nizableText(""), values);
                         _configureVariant(child, variantParameter, modelName);
                         params.put(variantParameter.getId(), variantParameter);
                     }
                     else
                     {
                         getLogger().debug("Discarding variant " + child.toAbsolutePath().toString() + " because it has no sub directories as values");
                     }
                 });
        }
    }
    
    private List<Variant> _getVariantValues(Path variant, String modelName)
    {
        List<Variant> values = new ArrayList<>();
        
        try (Stream<Path> files = Files.list(variant))
        {
            files
                 .filter(FileFilter.getSkinDirectoryFilter())
                 .forEach(child ->
                 {
                     String id = child.getFileName().toString();
                     
                     // Thumbnail
                     String thumbnailPath = null;
                     Path thumbnailFile = child.resolve(id + ".png");
                     if (Files.exists(thumbnailFile))
                     {
                         thumbnailPath = variant.getParent().relativize(thumbnailFile).toString();
                     }
                     
                     I18nizableText label = new I18nizableText(id);
                     I18nizableText description = new I18nizableText("");
                     Path file = child.resolve(id + ".xml");
                     if (Files.exists(file))
                     {
                         try (InputStream is = Files.newInputStream(file))
                         {
                             Configuration configuration = new DefaultConfigurationBuilder().build(is);
                             label = _configureI18nizableText(configuration.getChild("label"), modelName);
                             
                             description = _configureI18nizableText(configuration.getChild("description"), modelName);
                         }
                         catch (Exception e)
                         {
                             getLogger().error("Unable to configure variant '" + id + "'", e);
                         }
                     }
                     
                     values.add(new Variant(id, label, description, thumbnailPath));
                 });
        }
        catch (IOException e)
        {
            throw new RuntimeException("Unable to get variants from " + variant.toAbsolutePath().toString(), e);
        }
        
        return values;
    }
    
    private void _configureVariant(Path variantFile, VariantParameter param, String modelName)
    {
        Path file = variantFile.resolve(variantFile.getFileName().toString() + ".xml");
        
        if (Files.exists(file))
        {
            try (InputStream is = Files.newInputStream(file))
            {
                Configuration configuration = new DefaultConfigurationBuilder().build(is);

                I18nizableText label = _configureI18nizableText(configuration.getChild("label"), modelName);
                param.setLabel(label);
                
                I18nizableText description = _configureI18nizableText(configuration.getChild("description"), modelName);
                param.setDescription(description);
                
                param.setIconGlyph(configuration.getChild("icon-glyph").getValue(null));
            }
            catch (Exception e)
            {
                getLogger().error("Unable to configure variant '" + param.getId() + "'", e);
            }
        }
        
        File iconSmall = new File(variantFile + "/thumbnail_16.png");
        if (iconSmall.exists())
        {
            param.setIconSmall(iconSmall.getName());
        }
        
        File iconLarge = new File(variantFile + "/thumbnail_32.png");
        if (iconLarge.exists())
        {
            param.setIconLarge(iconLarge.getName());
        }
    }
    
    private Map<String, ImageParameter> _getImageParameters (Path modelDir, String modelName)
    {
        Map<String, ImageParameter> params = new HashMap<>();
        
        Path imagesDir = modelDir.resolve("model/images");
        if (Files.exists(imagesDir))
        {
            _findImagesParameters(modelDir, imagesDir, imagesDir, params, modelName);
        }
        return params;
    }
    
    private void _findImagesParameters(Path modelDir, Path imagesDir, Path file, Map<String, ImageParameter> params, String modelName)
    {
        try (Stream<Path> files = Files.list(file))
        {
            files
                 .filter(Files::isDirectory)
                 .forEach(child -> 
                 {
                     String filename = child.getFileName().toString();
                     String lcFilename = filename.toLowerCase();
                     if (lcFilename.endsWith(".png") || lcFilename.endsWith(".jpg") || lcFilename.endsWith(".jpeg") || lcFilename.endsWith(".gif"))
                     {
                         String imagePath = imagesDir.relativize(child).toString();
                         ImageParameter imageParameter = new ImageParameter(imagePath, new I18nizableText(filename), new I18nizableText(""));
                         _configureImage(child, imageParameter, modelName);
                         params.put(imageParameter.getId(), imageParameter);
                     }
                     else
                     {
                         _findImagesParameters (modelDir, imagesDir, child, params, modelName);
                     }
                 });
        }
        catch (IOException e)
        {
            throw new RuntimeException("Cannot find images parameters in directory " + imagesDir.toString(), e);
        }
    }
    
    private void _configureImage(Path imageFile, ImageParameter param, String modelName)
    {
        String filename = imageFile.getFileName().toString();
        int i = filename.lastIndexOf(".");
        if (i > 0)
        {
            filename = filename.substring(0, i);
        }
        Path file = imageFile.resolve(filename + ".xml");
        
        if (Files.exists(file))
        {
            try (InputStream is = Files.newInputStream(file))
            {
                Configuration configuration = new DefaultConfigurationBuilder().build(is);

                I18nizableText label = _configureI18nizableText(configuration.getChild("label"), modelName);
                param.setLabel(label);
                
                I18nizableText description = _configureI18nizableText(configuration.getChild("description"), modelName);
                param.setDescription(description);
                
                param.setIconGlyph(configuration.getChild("icon-glyph").getValue(null));
            }
            catch (Exception e)
            {
                getLogger().error("Unable to configure image parameter '" + param.getId() + "'", e);
            }
        }
        
        Path iconSmall = imageFile.resolve("thumbnail_16.png");
        if (Files.exists(iconSmall))
        {
            param.setIconSmall(iconSmall.getFileName().toString());
        }
        Path iconLarge = imageFile.resolve("thumbnail_32.png");
        if (Files.exists(iconLarge))
        {
            param.setIconLarge(iconLarge.getFileName().toString());
        }
    }
    
    
    private Map<String, CSSParameter> _getCSSParameters(Path modelDir, String modelName) throws IOException
    {
        Map<String, CSSParameter> cssParams = new HashMap<>();

        // Parse css files in the root resources directory.
        for (Path file : _listFiles(modelDir.resolve("resources"), "css"))
        {
            _parseCSSFile(cssParams, modelName, file);
        }
        
        // Parse css files for each templates.
        Path templatesDir = modelDir.resolve("templates");
        if (Files.isDirectory(templatesDir))
        {
            try (Stream<Path> files = Files.list(templatesDir))
            {
                files
                     .forEach(template -> 
                     {
                         for (Path file : _listFiles(template.resolve("resources"), "css"))
                         {
                             _parseCSSFile(cssParams, modelName, file);
                         }
                     });
            }
        }
        
        // Parse xsl files for inline css 
        for (Path file : _listFiles(modelDir, "xsl"))
        {
            if (Files.isRegularFile(file))
            {
                _parseCSSFile(cssParams, modelName, file);
            }
        }

        return cssParams;
    }

    private List<Path> _listFiles(Path file, String extension)
    {
        try
        {
            if (Files.exists(file))
            {
                return Files.walk(file)
                        .filter(Files::isRegularFile)
                        .filter(f -> f.getFileName().toString().endsWith("."  + extension))
                        .collect(Collectors.toList());
            }
            else
            {
                return Collections.EMPTY_LIST;
            }
            
        }
        catch (IOException e)
        {
            throw new RuntimeException("Cannot list " + extension + " files in " + file.toString(), e);
        }
    }

    private void _parseCSSFile (Map<String, CSSParameter> cssParams, String modelName, Path cssFile)
    {
        try (InputStream is = Files.newInputStream(cssFile))
        {
            String string = IOUtils.toString(is, "UTF-8");
            
            Matcher m = CSS_PARAM_PATTERN.matcher(string);
            while (m.find())
            {
                String id = m.group(3);
                if (cssParams.containsKey(id))
                {
                    CSSParameter cssParameter = cssParams.get(id);
                    cssParameter.addCSSFile(cssFile);
                    
                    I18nizableText label = m.group(4) != null ? _parseI18nizableText(m.group(4), modelName) : null;
                    if (label != null)
                    {
                        cssParameter.setLabel(label);
                    }
                    
                    I18nizableText description = m.group(5) != null ? _parseI18nizableText(m.group(5), modelName) : null;
                    if (description != null)
                    {
                        cssParameter.setDescription(description);
                    }
                }
                else
                {
                    String cssProperty = m.group(1);
                    String defaultValue = m.group(2).trim();

                    I18nizableText label = m.group(4) != null ? _parseI18nizableText(m.group(4), modelName) : null;
                    I18nizableText description = m.group(5) != null ? _parseI18nizableText(m.group(5), modelName) : null;
                    
                    if (cssProperty.equals("color") || cssProperty.equals("background-color") || cssProperty.equals("border-color"))
                    {
                        CSSColorParameter cssParameter = new CSSColorParameter (id, label, description, cssFile, cssProperty, defaultValue, _modelsManager.getModel(modelName), this);
                        cssParams.put(id, cssParameter);
                    }
                    else
                    {
                        CSSParameter cssParameter = new CSSParameter(id, label, description, cssFile, cssProperty, defaultValue);
                        cssParams.put(id, cssParameter);
                    }
                }
            }
        }
        catch (IOException e)
        {
            getLogger().error("Unable to parse file '" + cssFile.getFileName().toString() + "'", e);
        }
    }
    
    private Map<String, TextParameter> _getTextParameters(Path modelDir, String modelName)
    {
        Map<String, TextParameter> textParams = new HashMap<>();

        for (Path file : _listFiles(modelDir, "xsl"))
        {
            if (Files.isRegularFile(file))
            {
                textParams.putAll(_parseXSLFile(modelName, file));
            }
        }

        return textParams;
    }
    
    private Map<String, TextParameter> _parseXSLFile (String skinName, Path xslFile)
    {
        Map<String, TextParameter> txtParams = new LinkedHashMap<>();
        
        try (InputStream is = Files.newInputStream(xslFile))
        {
            String string = IOUtils.toString(is, "UTF-8");
            
            Matcher m = TXT_PARAM_PATTERN.matcher(string);
            while (m.find())
            {
                String id = m.group(2);
                String defaultValue = m.group(1);
                
                I18nizableText label = m.group(3) != null ? _parseI18nizableText(m.group(3), skinName) : new I18nizableText(id);
                I18nizableText description = m.group(4) != null ? _parseI18nizableText(m.group(4), skinName) : new I18nizableText("");
                
                TextParameter txtParam = new TextParameter(id, label, description, xslFile, defaultValue);
                txtParams.put(id, txtParam);
            }
        }
        catch (IOException e)
        {
            getLogger().error("Unable to parse file '" + xslFile.getFileName().toString() + "'", e);
        }
        
        return txtParams;
    }
    
    private Map<String, I18nizableTextParameter> _getI18nTextParameters(Path modelDir, String modelName)
    {
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();

        Path file = modelDir.resolve("i18n/messages.xml");
        if (Files.exists(file))
        {
            i18nParams.putAll(_parseI18nFile(modelName, file));
        }

        return i18nParams;
    }
    

    private Map<String, I18nizableTextParameter> _parseI18nFile (String skinName, Path i18nFile)
    {
        Map<String, I18nizableTextParameter> i18nParams = new LinkedHashMap<>();
        
        try (InputStream is = Files.newInputStream(i18nFile))
        {
            String string = IOUtils.toString(is, StandardCharsets.UTF_8);
            
            String defaultLang = _getCatalogLanguage(string, i18nFile.getFileName().toString());
            
            Matcher m = I18N_PARAM_PATTERN.matcher(string);
            while (m.find())
            {
                String i18nKey = m.group(1);
                String defaultValue = m.group(2).trim();
                String id = m.group(3);
                I18nizableText label = m.group(4) != null ? _parseI18nizableText(m.group(4), skinName) : new I18nizableText(id);
                I18nizableText description = m.group(5) != null ? _parseI18nizableText(m.group(5), skinName) : new I18nizableText("");
                
                // Default values
                Map<String, String> defaultValues = new HashMap<>();
                defaultValues.put(defaultLang, defaultValue);
                defaultValues.putAll(_getI18nOtherDefaultValues(i18nFile, i18nKey));
                
                I18nizableTextParameter i18nParam = new I18nizableTextParameter(id, label, description, i18nKey, defaultValues);
                i18nParams.put(id, i18nParam);
            }
        }
        catch (IOException e)
        {
            getLogger().error("Unable to parse file '" + i18nFile.getFileName().toString() + "'", e);
        }
        
        return i18nParams;
    }
    
    private String _getCatalogLanguage (String string, String fileName)
    {
        Matcher m = __I18N_CATALOG_LANGUAGE.matcher(string);
        if (m.find())
        {
            return m.group(1);
        }
        else if (fileName.startsWith("messages_"))
        {
            return fileName.substring("messages_".length(), 2);
        }
        return "en";
    }
    
    private Map<String, String> _getI18nOtherDefaultValues(Path defaultCatalog, String i18nKey) throws IOException
    {
        Pattern pattern = Pattern.compile("^\\s*<message key=\"" + i18nKey + "\">([^<]*)</message>\\s*$", Pattern.MULTILINE); 
        
        Map<String, String> defaultValues = new HashMap<>();
        
        try (Stream<Path> files = Files.list(defaultCatalog.getParent()))
        {
            files
                .filter(f -> !f.equals(defaultCatalog))
                .filter(Files::isRegularFile)
                .filter(f -> f.getFileName().toString().startsWith("messages"))
                .forEach(file ->
                {
                    try (InputStream is = Files.newInputStream(file))
                    {
                        String string = org.apache.commons.io.IOUtils.toString(is, "UTF-8");
                        
                        String lang = _getCatalogLanguage(string, file.getFileName().toString());
                        
                        Matcher m = pattern.matcher(string);
                        if (m.find())
                        {
                            String value = m.group(1);
                            defaultValues.put(lang, value);
                        }
                    }
                    catch (IOException e)
                    {
                        getLogger().error("Unable to parse file '" + file.getFileName().toString() + "'", e);
                    }
                });
        }
        
        return defaultValues;
    }
    
    private I18nizableText _parseI18nizableText (String label, String modelName)
    {
        if (label.startsWith("\"") && label.endsWith("\""))
        {
            return new I18nizableText(label.substring(1, label.length() - 1));
        }
        else
        {
            return new I18nizableText("model." + modelName, label);
        }
    }
    
    /**
     * Get the parameter's value
     * @param skinDir The skin directory (could be in temp, work or skin)
     * @param modelName The model name
     * @param id The parameter id
     * @return The parameter's value
     */
    public Object getParameterValue (Path skinDir, String modelName, String id)
    {
        List<String> ids = new ArrayList<>();
        ids.add(id);
        
        return getParameterValues(skinDir, modelName, ids).get(id);
    }
    
    /**
     * Get the parameters' values
     * @param skinDir The skin directory (could be in temp, work or skin)
     * @param modelName The model name
     * @param ids The parameters' id
     * @return The parameters' values
     */
    public Map<String, Object> getParameterValues (Path skinDir, String modelName, List<String> ids)
    {
        Map<String, Object> values = new HashMap<>();
        
        Path modelFile = skinDir.resolve("model.xml");
        
        try (InputStream is = Files.newInputStream(modelFile))
        {
            Configuration configuration = new DefaultConfigurationBuilder(true).build(is);
            Configuration[] parametersConf = configuration.getChild("parameters").getChildren("parameter");
            
            for (Configuration paramConf : parametersConf)
            {
                String id = paramConf.getAttribute("id");
                if (ids.contains(id))
                {
                    AbstractSkinParameter modelParam = getModelParamater(modelName, id);
                    if (modelParam instanceof I18nizableTextParameter)
                    {
                        Configuration[] children = paramConf.getChildren();
                        Map<String, String> langValues = new HashMap<>();
                        for (Configuration langConfig : children)
                        {
                            langValues.put(langConfig.getName(), langConfig.getValue(""));
                        }
                        values.put(id, langValues);
                    }
                    else
                    {
                        values.put(id, paramConf.getValue(""));
                    }
                }
            }
            
            return values;
        }
        catch (Exception e)
        {
            getLogger().error("Unable to get values for parameters '" + StringUtils.join(ids, ", ") + "'", e);
            return new HashMap<>();
        }
    }
    
    /**
     * Get the current theme
     * @param skinDir The skin directory (could be in temp, work or skin)
     * @return The current theme id
     */
    public String getColorTheme (Path skinDir)
    {
        Path modelFile = skinDir.resolve("model.xml");
        
        try (InputStream is = Files.newInputStream(modelFile))
        {
            Configuration configuration = new DefaultConfigurationBuilder(true).build(is);
            return configuration.getChild("color-theme").getValue(null);
        }
        catch (Exception e)
        {
            getLogger().error("Unable to get theme value", e);
            return null;
        }
    }
    
    /**
     * Get all parameters' values
     * @param skinDir The skin directory (could be in temp, work or skin)
     * @param modelName The model name
     * @return The all parameters' values
     */
    public Map<String, Object> getParameterValues (Path skinDir, String modelName)
    {
        Map<String, Object> values = new HashMap<>();
        
        Path modelFile = skinDir.resolve("model.xml");
        
        try (InputStream is = Files.newInputStream(modelFile))
        {
            Configuration configuration = new DefaultConfigurationBuilder(true).build(is);
            Configuration[] parametersConf = configuration.getChild("parameters").getChildren("parameter");
            
            Map<String, AbstractSkinParameter> modelParameters = getModelParameters(modelName);
            
            for (Configuration paramConf : parametersConf)
            {
                String id = paramConf.getAttribute("id");
                AbstractSkinParameter modelParam = modelParameters.get(id);
                if (modelParam != null)
                {
                    if (modelParam instanceof I18nizableTextParameter)
                    {
                        Configuration[] children = paramConf.getChildren();
                        Map<String, String> langValues = new HashMap<>();
                        for (Configuration langConfig : children)
                        {
                            langValues.put(langConfig.getName(), langConfig.getValue(""));
                        }
                        values.put(id, langValues);
                    }
                    else if (modelParam instanceof ImageParameter)
                    {
                        boolean uploaded = Boolean.valueOf(paramConf.getAttribute("uploaded", "false"));
                        values.put(id, new ImageParameter.FileValue(paramConf.getValue(""), uploaded));
                    }
                    else
                    {
                        values.put(id, paramConf.getValue(""));
                    }
                }
            }
            
            return values;
        }
        catch (Exception e)
        {
            getLogger().error("Unable to get values of all parameters", e);
            return new HashMap<>();
        }
    }
    
    /**
     * Modify the color theme
     * @param skinDir The skin directory (could be in temp, work or skin)
     * @param themeId The id of the theme. Can be null to clear theme.
     * @return <code>true</code> is modification success
     */
    public boolean saveColorTheme (Path skinDir, String themeId)
    {
        Path currentFile = skinDir.resolve("model.xml");
        
        Source currentSrc = null;
        Source src = null;
        try
        {
            currentSrc = _resolver.resolveURI("file://" + currentFile.toFile().getAbsolutePath());
            
            Map<String, Object> parentContext = new HashMap<>();
            parentContext.put("modelUri", currentSrc.getURI());
            if (themeId != null)
            {
                parentContext.put("themeId", themeId);
            }
            
            src = _resolver.resolveURI("cocoon://_plugins/skinfactory/change-color-theme", null, parentContext);
            
            SourceUtil.copy(src, currentSrc);
            
            return true;
        }
        catch (IOException e)
        {
            getLogger().error("Unable to update color theme for skin '" + skinDir.getFileName().toString() + "'", e);
            return false;
        }
        finally
        {
            _resolver.release(src);
            _resolver.release(currentSrc);
        }
    }
    
    /**
     * Modify a skin parameter's value
     * @param skinDir The skin directory (could be in temp, work or skin)
     * @param id the id of the parameter
     * @param value the new value
     * @return <code>true</code> is modification success
     */
    public boolean saveParameter(Path skinDir, String id, Object value)
    {
        Map<String, Object> parameters = new HashMap<>();
        parameters.put(id, value);
        
        return saveParameters (skinDir, parameters);
    }
    
    /**
     * Save skin parameters
     * @param skinDir The skin directory (could be in temp, work or skin)
     * @param parameters The skins parameters to save
     * @return <code>true</code> is modification success
     */
    public boolean saveParameters(Path skinDir, Map<String, Object> parameters)
    {
        Path currentFile = skinDir.resolve("model.xml");
        
        Source currentSrc = null;
        Source src = null;
        try
        {
            currentSrc = _resolver.resolveURI("file://" + currentFile.toFile().getAbsolutePath());
            
            Map<String, Object> parentContext = new HashMap<>();
            parentContext.put("modelUri", currentSrc.getURI());
            parentContext.put("skinParameters", parameters);
            
            src = _resolver.resolveURI("cocoon://_plugins/skinfactory/change-parameters", null, parentContext);
            
            SourceUtil.copy(src, currentSrc);
            
            return true;
        }
        catch (IOException e)
        {
            getLogger().error("Unable to save parameters from skin '" + skinDir.getFileName().toString() + "'", e);
            return false;
        }
        finally
        {
            _resolver.release(src);
            _resolver.release(currentSrc);
        }
    }
    
    /**
     * Apply model parameters in given skin
     * @param modelName The model name
     * @param skinDir The skin directory (could be temp, work or skins)
     */
    public void applyModelParameters (String modelName, Path skinDir)
    {
        Map<String, Object> currentValues = getParameterValues(skinDir, modelName);
        applyModelParameters(modelName, skinDir, currentValues);
    }
    
    /**
     * Apply model parameters in given skin
     * @param modelName The model name
     * @param skinDir The skin directory
     * @param values The values to set
     */
    public void applyModelParameters (String modelName, Path skinDir, Map<String, Object> values)
    {
        Map<String, Object> parameterValues = new HashMap<>();
        SkinModel model = _modelsManager.getModel(modelName);
        
        Map<String, String> modelDefaultValues = model.getDefaultValues();
        
        Map<String, AbstractSkinParameter> modelParameters = getModelParameters(modelName);
        for (AbstractSkinParameter skinParameter : modelParameters.values())
        {
            String paramId = skinParameter.getId();
            
            if (skinParameter instanceof I18nizableTextParameter)
            {
                Map<String, String> defaultValues = ((I18nizableTextParameter) skinParameter).getDefaultValues();
                @SuppressWarnings("unchecked")
                Map<String, String> currentValues = (Map<String, String>) values.get(paramId);
                if (currentValues == null)
                {
                    currentValues = new HashMap<>();
                }
                for (String lang : defaultValues.keySet())
                {
                    if (currentValues.get(lang) != null)
                    {
                        applyParameter(skinParameter, skinDir, modelName, currentValues.get(lang), lang);
                    }
                    else
                    {
                        // Default value
                        applyParameter(skinParameter, skinDir, modelName, defaultValues.get(lang), lang);
                        currentValues.put(lang, defaultValues.get(lang));
                    }
                }
                parameterValues.put(skinParameter.getId(), currentValues);
            }
            else if (skinParameter instanceof ImageParameter)
            {
                FileValue imgValue = (FileValue) _getValue(model, skinParameter, values, modelDefaultValues);
                applyParameter(skinParameter, skinDir, modelName, imgValue, null);
                parameterValues.put(skinParameter.getId(), imgValue);
            }
            else
            {
                String value = (String) _getValue (model, skinParameter, values, modelDefaultValues);
                applyParameter(skinParameter, skinDir, modelName, value, null);
                parameterValues.put(skinParameter.getId(), value);
            }
        }
        
        saveParameters(skinDir, parameterValues);
    }
    
    private Object _getValue (SkinModel model, AbstractSkinParameter param, Map<String, Object> values, Map<String, String> defaultValues)
    {
        String id = param.getId();
        
        // First search in current values
        if (values.containsKey(id))
        {
            return values.get(id);
        }
        
        // Then search in default values from model
        if (defaultValues.containsKey(id))
        {
            String valueAsStr =  defaultValues.get(id);
            
            if (param instanceof ImageParameter)
            {
                return new FileValue (valueAsStr, false);
            }
            else
            {
                return valueAsStr;
            }
        }
        
        // If nor found, get the parameter default value
        return param.getDefaultValue(model);
    }
    /**
     * Update hash
     * @param xmlFile the xml {@link File}
     * @param hash the new hash of the file
     * @throws IOException if an error occurs while manipulating the file
     */
    public void updateHash (Path xmlFile, String hash) throws IOException
    {
        Source currentSrc = null;
        Source src = null;
        try
        {
            currentSrc = _resolver.resolveURI("file://" + xmlFile.toFile().getAbsolutePath());
            
            Map<String, Object> parentContext = new HashMap<>();
            parentContext.put("modelUri", currentSrc.getURI());
            parentContext.put("hash", hash);
            
            src = _resolver.resolveURI("cocoon://_plugins/skinfactory/change-hash", null, parentContext);
            
            SourceUtil.copy(src, currentSrc);
        }
        finally
        {
            _resolver.release(src);
            _resolver.release(currentSrc);
        }
        
    }
    
    /**
     * Apply parameter
     * @param parameter the skin parameter
     * @param skinDir the skin directory (could be temp, work or skins)
     * @param modelName the model name
     * @param value the parameter value
     * @param lang The language
     */
    public void applyParameter (AbstractSkinParameter parameter, Path skinDir, String modelName, Object value, String lang)
    {
        Path modelDir = _modelsManager.getModel(modelName).getPath();
        parameter.apply(skinDir, modelDir, value, lang);
    }
    
    /**
     * Apply new color theme 
     * @param skinDir the skin directory (could be temp, work or skins)
     * @param modelName the model name
     */
    public void applyColorTheme (String modelName, Path skinDir)
    {
        Path modelDir = _modelsManager.getModel(modelName).getPath();
        
        Map<String, Object> currentValues = getParameterValues(skinDir, modelName);
        
        Map<String, AbstractSkinParameter> modelParameters = getModelParameters(modelName);
        for (AbstractSkinParameter skinParameter : modelParameters.values())
        {
            if (skinParameter instanceof CSSColorParameter)
            {
                String value = (String) currentValues.get(skinParameter.getId());
                if (StringUtils.isEmpty(value))
                {
                    value = (String) skinParameter.getDefaultValue(_modelsManager.getModel(modelName));
                }
                
                skinParameter.apply(skinDir, modelDir, value, null);
            }
        }
    }
    
    private I18nizableText _configureI18nizableText(Configuration configuration, String modelName) throws ConfigurationException
    {
        boolean i18nSupported = configuration.getAttributeAsBoolean("i18n", false);

        if (i18nSupported)
        {
            String catalogue = configuration.getAttribute("catalogue", null);
            if (catalogue == null)
            {
                catalogue = "model." + modelName;
            }

            return new I18nizableText(catalogue, configuration.getValue());
        }
        else
        {
            return new I18nizableText(configuration.getValue(""));
        }
    }
}
