/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.filefilter;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.function.Predicate;

/**
 * File filters
 */
public final class FileFilter
{
    private FileFilter()
    {
        // emtpy
    }
    
    /**
     * This filter accepts all <code>File</code>s excepted CVS and SVN directories
     * @return The filter
     */
    public static Predicate<Path> getSkinDirectoryFilter()
    {
        return path -> Files.isDirectory(path) 
                        && !path.getFileName().toString().equals("CVS") 
                        && !path.getFileName().toString().equals(".svn"); 
    }
    
    /**
     * This filter accepts all <code>File</code>s excepted CVS and SVN directories
     * @return The filter
     */
    public static Predicate<Path> getSkinFileFilter()
    {
        return path -> !path.getFileName().toString().equals("CVS") 
                        && !path.getFileName().toString().equals(".svn"); 
    }
    
    /**
     * This filter accepts all <code>File</code>s excepted CVS and SVN directories and root directory named "model"
     * @param modelDir The model root directory
     * @return the filter
     */
    public static final Predicate<Path> getModelFilter(Path modelDir)
    {
        return f -> 
        !f.getFileName().toString().equals("CVS")
        && !f.getFileName().toString().equals(".svn")
        && (!f.getParent().equals(modelDir)  || !f.getFileName().toString().equals("model"));
    }

    /**
     * This filter accepts <code>File</code>s that are not the root variant's description files ([VARIANT_NAME].xml and [VARIANT_NAME].png) and that are not a CVS or SVN directory
     * @param variantName The variant name
     * @param variantDir The variant root dir
     * @return the filter
     */
    public static final Predicate<Path> getModelVariantFilter(String variantName, Path variantDir)
    {
        return f -> 
                    !f.getFileName().toString().equals("CVS")
                    && !f.getFileName().toString().equals(".svn")
                    && (!f.getParent().equals(variantDir) 
                        || !f.getFileName().toString().equals(variantName + ".xml")
                            && !f.getFileName().toString().equals(variantName + ".png"));
    }

}
