/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.generators;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.xml.sax.SAXParser;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.skinfactory.SkinFactoryComponent;
import org.ametys.skinfactory.parameters.AbstractSkinParameter;
import org.ametys.skinfactory.parameters.AbstractSkinParameter.SkinParameterType;
import org.ametys.web.skin.SkinModel;
import org.ametys.web.skin.SkinModelsManager;

/**
 * Generate the ribbon file
 */
public class GenerateRibbonFile extends ServiceableGenerator
{
    private SkinFactoryComponent _skinFactoryManager;
    private SkinModelsManager _modelsManager;
    private I18nUtils _i18nUtils;
    
    private final HashMap<Pattern, String> _glyphAssociations = new LinkedHashMap<>();
    {
        _glyphAssociations.put(Pattern.compile("^(h1|h2|h3|h4)$"), "ametysicon-header");
        _glyphAssociations.put(Pattern.compile("text"), "ametysicon-text");
        _glyphAssociations.put(Pattern.compile("link"), "ametysicon-link23");
        _glyphAssociations.put(Pattern.compile("list"), "ametysicon-list4");
        _glyphAssociations.put(Pattern.compile("vmenu^$"), "ametysicon-vmenu");
        _glyphAssociations.put(Pattern.compile("menu"), "ametysicon-menu");
        _glyphAssociations.put(Pattern.compile("^(item|subitem)$"), "ametysicon-menu-element");
        _glyphAssociations.put(Pattern.compile("title|header"), "ametysicon-header");
        _glyphAssociations.put(Pattern.compile("table"), "ametysicon-tables1");
        _glyphAssociations.put(Pattern.compile("border"), "ametysicon-tables1"); // FIXME when new icon has arrived
        _glyphAssociations.put(Pattern.compile("image"), "ametysicon-image2");
        // Default glyph for unknown style
        _glyphAssociations.put(Pattern.compile("."), "ametysicon-designer");
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException 
    {
        super.service(smanager);
        _skinFactoryManager = (SkinFactoryComponent) smanager.lookup(SkinFactoryComponent.ROLE);
        _modelsManager = (SkinModelsManager)  smanager.lookup(SkinModelsManager.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "CMS");
        
        String modelName = parameters.getParameter("modelName", null);
        SkinModel model = _modelsManager.getModel(modelName);
        
        Path ribbonFile = model.getPath().resolve("model/cms-ribbon.xml");
        if (Files.exists(ribbonFile))
        {
            SAXParser saxParser = null;
            try (InputStream is = Files.newInputStream(ribbonFile))
            {
                saxParser = (SAXParser) manager.lookup(SAXParser.ROLE);
                saxParser.parse(new InputSource(is), new IgnoreRootHandler(contentHandler));
            }
            catch (ServiceException e)
            {
                throw new ProcessingException("Unable to get a SAX parser", e);
            }
            finally
            {
                manager.release(saxParser);
            }
        }
        
        Map<String, RibbonTab> tabs = _getGroupedParams(_skinFactoryManager.getModelParameters(modelName));
        
        XMLUtils.startElement(contentHandler, "parameters");
        for (RibbonTab tab : tabs.values())
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", tab.getId());
            attrs.addCDATAAttribute("label", _i18nUtils.translate(tab.getLabel()));
            XMLUtils.startElement(contentHandler, "tab", attrs);
            
            XMLUtils.startElement(contentHandler, "groups");
            
            Collection<RibbonGroup> groups = tab.getGroups();
            for (RibbonGroup group : groups)
            {
                attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", group.getId());
                attrs.addCDATAAttribute("label", _i18nUtils.translate(group.getLabel()));
                XMLUtils.startElement(contentHandler, "group", attrs);
                
                Collection<RibbonElement> elmts = group.getRibbonElements();
                for (RibbonElement elt : elmts)
                {
                    if (elt instanceof ParameterControl)
                    {
                        attrs = new AttributesImpl();
                        attrs.addCDATAAttribute("id", elt.getId());
                        XMLUtils.createElement(contentHandler, "parameter", attrs);
                    }
                    else if (elt instanceof Menu)
                    {
                        _saxMenu ((Menu) elt);
                    }
                }
                XMLUtils.endElement(contentHandler, "group");
            }
            XMLUtils.endElement(contentHandler, "groups");
            
            XMLUtils.endElement(contentHandler, "tab");
        }
        XMLUtils.endElement(contentHandler, "parameters");
        
        XMLUtils.endElement(contentHandler, "CMS");
        contentHandler.endDocument();
        
    }
    
    private void _saxMenu (Menu menu) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", menu.getId());
        attrs.addCDATAAttribute("label", _i18nUtils.translate(menu.getLabel()));
        _saxMenuIcons (menu, attrs);
        XMLUtils.startElement(contentHandler, "menu", attrs);
        
        for (String itemId : menu.getItems())
        {
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", itemId);
            XMLUtils.createElement(contentHandler, "parameter", attrs);
        }
        
        for (Menu submenu : menu.getMenus())
        {
            _saxMenu(submenu);
        }
        
        XMLUtils.endElement(contentHandler, "menu");
    }
    
    private Map<String, RibbonTab> _getGroupedParams (Map<String, AbstractSkinParameter> skinParameters)
    {
        Map<String, RibbonTab> tabs = new HashMap<>();
        
        for (AbstractSkinParameter param : skinParameters.values())
        {   
            String id = param.getId();
            String[] parts = id.split("\\.");
            
            String tabId = "org.ametys.skinfactory.Tab." + (parts.length >= 3 ? parts[0] : "default");
            String groupId = parts.length >= 3 ? StringUtils.join(parts, ".", 0, 2) : "default.default";
            String eltId = null;
            
            String itemId = null;
            String subItemId = null;
            
            if (parts.length <= 3 || !param.getType().equals(SkinParameterType.CSS))
            {
                eltId = StringUtils.join(parts, ".");
            }
            else
            {
                eltId = StringUtils.join(parts, ".", 0, 3);
                itemId = parts.length ==  4 ? StringUtils.join(parts, ".") : StringUtils.join(parts, ".", 0, 4);
                
                if (parts.length > 4)
                {
                    subItemId = StringUtils.join(parts, ".");
                }
            }
            
            _addGroupedParam(tabs, tabId, groupId, eltId, itemId, subItemId);
        }
        
        return tabs;
    }
    
    private void _addGroupedParam (Map<String, RibbonTab> tabs, String tabId, String groupId, String controlId, String itemId, String subItemId)
    {
        if (!tabs.containsKey(tabId))
        {
            tabs.put(tabId, new RibbonTab(tabId, new I18nizableText(tabId.substring(tabId.lastIndexOf(".") + 1))));
        }
        
        RibbonTab ribbonTab = tabs.get(tabId);
        if (!ribbonTab.hasGroup(groupId))
        {
            ribbonTab.addGroup(new RibbonGroup(groupId, new I18nizableText(groupId.substring(groupId.lastIndexOf(".") + 1))));
        }
        
        RibbonGroup ribbonGroup = ribbonTab.getGroup(groupId);
        
        if (itemId != null)
        {
            if (!ribbonGroup.hasRibbonElement(controlId))
            {
                ribbonGroup.addRibbonElement(new Menu(controlId, new I18nizableText(controlId.substring(controlId.lastIndexOf(".") + 1))));
            }
            
            RibbonElement ribbonControl = ribbonGroup.getRibbonElement(controlId);
            
            if (ribbonControl instanceof Menu)
            {
                Menu menu = (Menu) ribbonControl;
                
                if (subItemId == null)
                {
                    menu.addItem(itemId);
                }
                else
                {
                    if (!menu.hasMenu(itemId))
                    {
                        menu.addMenu(new Menu(itemId, new I18nizableText(itemId.substring(itemId.lastIndexOf(".") + 1))));
                    }
                    
                    Menu subMenu = menu.getMenu(itemId);
                    subMenu.addItem(subItemId);
                }
            }
        }
        else
        {
            ribbonGroup.addRibbonElement(new ParameterControl(controlId));
        }
    }
    
    private void _saxMenuIcons (Menu menu, AttributesImpl attrs)
    {
        String label = _i18nUtils.translate(menu.getLabel());
        
        for (Map.Entry<Pattern, String> entry : _glyphAssociations.entrySet())
        {
            Pattern pattern = entry.getKey();
            if (pattern.matcher(label).matches())
            {
                attrs.addCDATAAttribute("iconGlyph", entry.getValue());
                return;
            }
        }
    }
    
    class RibbonTab
    {
        private I18nizableText _label;
        private String _id;
        private Map<String, RibbonGroup> _groups;
        
        public RibbonTab(String id, I18nizableText label)
        {
            _id = id;
            _label = label;
            _groups = new HashMap<>();
        }
        
        String getId ()
        {
            return _id;
        }
        
        I18nizableText getLabel ()
        {
            return _label;
        }
        
        Collection<RibbonGroup> getGroups ()
        {
            return _groups.values();
        }
        
        void addGroup (RibbonGroup group)
        {
            _groups.put(group.getId(), group);
        }
        
        boolean hasGroup (String groupId)
        {
            return _groups.containsKey(groupId);
        }
        
        RibbonGroup getGroup (String groupId)
        {
            return _groups.get(groupId);
        }
    }
    
    class RibbonGroup 
    {
        private String _id;
        private I18nizableText _label;
        private Map<String, RibbonElement> _ribbonElts;
        
        public RibbonGroup(String id, I18nizableText label)
        {
            _id = id;
            _label = label;
            _ribbonElts = new HashMap<>();
        }
        
        String getId()
        {
            return _id;
        }
        
        I18nizableText getLabel ()
        {
            return _label;
        }
        
        void addRibbonElement (RibbonElement control)
        {
            _ribbonElts.put(control.getId(), control);
        }
        
        boolean hasRibbonElement (String controlId)
        {
            return _ribbonElts.containsKey(controlId);
        }
        
        RibbonElement getRibbonElement (String controlId)
        {
            return _ribbonElts.get(controlId);
        }
        
        Collection<RibbonElement> getRibbonElements ()
        {
            return _ribbonElts.values();
        }
    }
    
    interface RibbonElement
    {
        public String getId();
    }
    
    class Menu implements RibbonElement
    {
        private String _id;
        private I18nizableText _label;
        private List<String> _items;
        private Map<String, Menu> _menus;
        
        public Menu(String id, I18nizableText label)
        {
            _id = id;
            _label = label;
            _items = new ArrayList<>();
            _menus = new HashMap<>();
        }
        
        @Override
        public String getId()
        {
            return _id;
        }
        
        I18nizableText getLabel ()
        {
            return _label;
        }
        
        void addItem (String id)
        {
            _items.add(id);
        }
        
        void addMenu (Menu menu)
        {
            _menus.put(menu.getId(), menu);
        }
        
        boolean hasMenu (String menuId)
        {
            return _menus.containsKey(menuId);
        }
        
        Menu getMenu (String menuId)
        {
            return _menus.get(menuId);
        }
        
        List<String> getItems ()
        {
            return _items;
        }
        
        Collection<Menu> getMenus()
        {
            return _menus.values();
        }
    }
    
    class ParameterControl implements RibbonElement
    {
        private String _id;
        public ParameterControl(String id)
        {
            _id = id;
        }
        
        @Override
        public String getId()
        {
            return _id;
        }
    }
}
