/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.ui.ClientSideElement;
import org.ametys.core.ui.SimpleMenu;

/**
 * This implementation looks at existing {@link ImportModelActionExtensionPoint} to create a button for import skin. 
 */
public class ImportModelClientSideElement extends SimpleMenu
{
    private ImportModelActionExtensionPoint _importModelActionEP;

    private boolean _importMenuItemsInitialized;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _importModelActionEP = (ImportModelActionExtensionPoint) smanager.lookup(ImportModelActionExtensionPoint.ROLE);
    }
    
    private synchronized void _resolveImportMenuItems ()
    {
        if (!_importMenuItemsInitialized)
        {
            List<UnresolvedItem> unresolvedImportMenuItems = new ArrayList<>();
            
            Set<String> importActionIds = _importModelActionEP.getExtensionsIds();
            
            if (importActionIds.size() > 1)
            {
                for (String id : importActionIds)
                {
                    unresolvedImportMenuItems.add(new UnresolvedItem(id, false, false));
                }
            }
            
            for (UnresolvedItem unresolvedItem : unresolvedImportMenuItems)
            {
                String id = unresolvedItem.getId();
                ClientSideElement element = _importModelActionEP.getExtension(id);
                
                if (unresolvedItem.isPrimary())
                {
                    _primaryMenuItem = element;
                }
                
                _menuItems.add(element);
            }
            
            _importMenuItemsInitialized = true;
        }
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        List<Script> scripts = new ArrayList<>();
        
        Set<String> importActionIds = _importModelActionEP.getExtensionsIds();
        
        if (importActionIds.size() == 0)
        {
            // Non button
            return scripts;
        }
        else if (importActionIds.size() == 1)
        {
            for (String id : importActionIds)
            {
                ClientSideElement importAction = _importModelActionEP.getExtension(id);
                return importAction.getScripts(ignoreRights, contextParameters);
            }
        }
        
        _resolveImportMenuItems();
        
        List<ScriptFile> scriptFiles = new ArrayList<>();
        List<ScriptFile> cssFiles = new ArrayList<>();
        Map<String, Object> parameters = new HashMap<>();
        
        for (Script script : super.getScripts(ignoreRights, contextParameters))
        {
            scriptFiles.addAll(script.getScriptFiles());
            cssFiles.addAll(script.getCSSFiles());
            parameters.putAll(script.getParameters());
        }
        
        for (String id : importActionIds)
        {
            ClientSideElement importAction = _importModelActionEP.getExtension(id);
            for (Script script : importAction.getScripts(ignoreRights, contextParameters))
            {
                scriptFiles.addAll(script.getScriptFiles());
                cssFiles.addAll(script.getCSSFiles());
                parameters.putAll(script.getParameters());
            }
        }
        
        scripts.add(new Script(this.getId(), _script.getScriptClassname(), scriptFiles, cssFiles, parameters));
        return scripts;
    }
    
}
