/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.model;

import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.skinfactory.SkinFactoryComponent;
import org.ametys.skinfactory.parameters.AbstractSkinParameter;
import org.ametys.skinfactory.parameters.I18nizableTextParameter;
import org.ametys.skinfactory.parameters.ImageParameter;
import org.ametys.web.skin.SkinModel;
import org.ametys.web.skin.SkinModelsManager;

/**
 * Manages the design conceptions of a model
 */
public class ModelDesignsManager extends AbstractLogEnabled implements ThreadSafe, Serviceable, Initializable, Component
{
    /** The avalon role name */
    public static final String ROLE = ModelDesignsManager.class.getName();
    
    private static final String __MODEL_DESIGN_CACHE = ModelDesignsManager.class.getName() + "$modelDesign";
    
    private SkinModelsManager _modelsManager;
    private SkinFactoryComponent _skinFactoryManager;

    private AbstractCacheManager _cacheManager;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _modelsManager = (SkinModelsManager) smanager.lookup(SkinModelsManager.ROLE);
        _skinFactoryManager = (SkinFactoryComponent) smanager.lookup(SkinFactoryComponent.ROLE);
        _cacheManager = (AbstractCacheManager) smanager.lookup(AbstractCacheManager.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createMemoryCache(__MODEL_DESIGN_CACHE,
                new I18nizableText("plugin.skinfactory", "PLUGINS_SKINFACTORY_CACHE_MODEL_DESIGN_LABEL"),
                new I18nizableText("plugin.skinfactory", "PLUGINS_SKINFACTORY_CACHE_MODEL_DESIGN_DESCRIPTION"),
                true,
                null);
    }
    
    /**
     * Get all design instances for given model
     * @param modelName The model name
     * @return all design instances for given model
     */
    public Set<Design> getDesigns (String modelName)
    {
        return _getDesignCache().get(modelName, this::_getDesigns);
    }
    
    /**
     * Get design instance of given id and model name
     * @param modelName The model name
     * @param id The id
     * @return design instance
     */
    public Design getDesign (String modelName, String id)
    {
        Set<Design> designs = _getDesignCache().get(modelName, this::_getDesigns);
        return designs.stream()
            .filter(design -> StringUtils.equals(design.getId(), id))
            .findFirst().orElse(null);
    }
    
    /**
     * Apply a design
     * @param modelName The model name
     * @param id Id of design
     * @param skinDir The skin directory (could be temp, work or skins)
     */
    public void applyDesign (String modelName, String id, Path skinDir)
    {
        SkinModel model = _modelsManager.getModel(modelName);
     
        Path file = model.getPath().resolve("model/designs/" + id + ".xml");
        if (Files.exists(file))
        {
            // Apply color theme
            String themeId = _getColorTheme(file);
            if (themeId != null)
            {
                _skinFactoryManager.saveColorTheme(skinDir, themeId);
            }
            
            // Apply values
            Map<String, Object> values = _getParameterValues (modelName, file);
            _skinFactoryManager.applyModelParameters(modelName, skinDir, values);
        }
    }
    
    
    private Set<Design> _getDesigns (String modelName)
    {
        SkinModel model = _modelsManager.getModel(modelName);
        
        Set<Design> designs;
        
        Path designDir = model.getPath().resolve("model/designs");
        if (Files.exists(designDir))
        {
            try
            {
                designs = Files.walk(designDir, 1)
                     .filter(f -> f.getFileName().toString().toLowerCase().endsWith(".xml"))
                     .map(f -> _configureDesign(modelName, f))
                     .filter(design -> design != null)
                     .collect(Collectors.toSet());
            }
            catch (Exception e)
            {
                throw new RuntimeException("Cannot read the configuration file model/designs for the model " + modelName);
            }
        }
        else
        {
            designs = new HashSet<>();
        }
        return designs;
    }
    
    
    private Design _configureDesign (String modelName, Path configurationFile)
    {
        try (InputStream is = Files.newInputStream(configurationFile))
        {
            String fileName = configurationFile.getFileName().toString();
            String id = fileName.substring(0, fileName.lastIndexOf("."));
            
            Configuration configuration = new DefaultConfigurationBuilder().build(is);
            I18nizableText label = _configureI18nizableText(configuration.getChild("label", false), new I18nizableText(id), modelName);
            I18nizableText description = _configureI18nizableText(configuration.getChild("description", false), new I18nizableText(id), modelName);
            
            String iconName = id + ".png";
            String icon = "/plugins/skinfactory/resources/img/actions/designs_32.png";
            Path iconFile = configurationFile.getParent().resolve(iconName);
            if (Files.exists(iconFile))
            {
                icon = "/plugins/skinfactory/" + modelName + "/_thumbnail/32/32/model/designs/" + iconName;
            }
            
            return new Design(id, label, description, icon);
        }
        catch (Exception e)
        {
            if (getLogger().isWarnEnabled())
            {
                getLogger().warn("Cannot read the configuration file model/designs/" + configurationFile.getFileName().toString()  + " for the model '" + modelName + "'. Continue as if file was not existing", e);
            }
            return null;
        }
        
    }
    
    private String _getColorTheme(Path file)
    {
        try (InputStream is = Files.newInputStream(file))
        {
            
            Configuration configuration = new DefaultConfigurationBuilder(true).build(is);
            return configuration.getChild("color-theme").getValue(null);
        }
        catch (Exception e)
        {
            getLogger().error("Unable to get color theme", e);
            return null;
        }
    }
    
    private Map<String, Object> _getParameterValues (String modelName, Path file)
    {
        Map<String, Object> values = new HashMap<>();
        
        try (InputStream is = Files.newInputStream(file))
        {
            
            Configuration configuration = new DefaultConfigurationBuilder(true).build(is);
            Configuration[] parametersConf = configuration.getChild("parameters").getChildren("parameter");
            
            Map<String, AbstractSkinParameter> modelParameters = _skinFactoryManager.getModelParameters(modelName);
            
            for (Configuration paramConf : parametersConf)
            {
                String id = paramConf.getAttribute("id");
                AbstractSkinParameter modelParam = modelParameters.get(id);
                if (modelParam != null)
                {
                    if (modelParam instanceof I18nizableTextParameter)
                    {
                        Configuration[] children = paramConf.getChildren();
                        Map<String, String> langValues = new HashMap<>();
                        for (Configuration langConfig : children)
                        {
                            langValues.put(langConfig.getName(), langConfig.getValue(""));
                        }
                        values.put(id, langValues);
                    }
                    else if (modelParam instanceof ImageParameter)
                    {
                        values.put(id, new ImageParameter.FileValue(paramConf.getValue(""), false));
                    }
                    else
                    {
                        values.put(id, paramConf.getValue(""));
                    }
                }
            }
            
            return values;
        }
        catch (Exception e)
        {
            getLogger().error("Unable to get values of all parameters", e);
            return new HashMap<>();
        }
    }
    
    
    private I18nizableText _configureI18nizableText(Configuration configuration, I18nizableText defaultValue, String modelName) throws ConfigurationException
    {
        if (configuration != null)
        {
            boolean i18nSupported = configuration.getAttributeAsBoolean("i18n", false);
            if (i18nSupported)
            {
                String catalogue = configuration.getAttribute("catalogue", null);
                if (catalogue == null)
                {
                    catalogue = "model." + modelName;
                }

                return new I18nizableText(catalogue, configuration.getValue());
            }
            else
            {
                return new I18nizableText(configuration.getValue(""));
            }
        }
        else
        {
            return defaultValue;
        }
        
    }
    
    private Cache<String, Set<Design>> _getDesignCache()
    {
        return _cacheManager.get(__MODEL_DESIGN_CACHE);
    }
    
    /**
     * Bean representing a model design
     *
     */
    public static class Design 
    {
        private String _id;
        private I18nizableText _label;
        private I18nizableText _description;
        private String _icon;

        /**
         * Constructor
         * @param id the theme id
         * @param label the theme's label
         * @param description the theme's description
         * @param icon the icon
         */
        public Design (String id, I18nizableText label, I18nizableText description, String icon)
        {
            _id = id;
            _label = label;
            _description = description;
            _icon = icon;
        }
        
        /**
         * Get the id
         * @return the id
         */
        public String getId ()
        {
            return _id;
        }
        
        /**
         * Get the label
         * @return the label
         */
        public I18nizableText getLabel ()
        {
            return _label;
        }
        
        /**
         * Get the description
         * @return the description
         */
        public I18nizableText getDescription ()
        {
            return _description;
        }
        
        /**
         * Get the icon
         * @return the icon
         */
        public String getIcon ()
        {
            return _icon;
        }
    }        

}
