/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.parameters;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.io.IOUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.skin.SkinModel;

/**
 * Implementation of {@link AbstractSkinParameter} for a CSS property
 */
public class CSSParameter extends AbstractSkinParameter
{
    private final List<Path> _cssFiles;
    private final String _cssProperty;
    private final String _defaultValue;
    
    /**
     * Constructor
     * @param id the unique id
     * @param label the label
     * @param description the description
     * @param cssFile the css file
     * @param cssProperty the css property
     * @param defaultValue the default value
     */
    public CSSParameter(String id, I18nizableText label, I18nizableText description, Path cssFile, String cssProperty, String defaultValue)
    {
        super(id, label, description);
        _cssFiles = new ArrayList<>();
        _cssFiles.add(cssFile);
        _cssProperty = cssProperty;
        _defaultValue = defaultValue;
    }
    
    /**
     * Constructor
     * @param id the unique id
     * @param label the label
     * @param description the description
     * @param cssFiles the css files
     * @param cssProperty the css property
     * @param defaultValue the default value
     */
    public CSSParameter(String id, I18nizableText label, I18nizableText description, List<Path> cssFiles, String cssProperty, String defaultValue)
    {
        super(id, label, description);
        _cssFiles = cssFiles;
        _cssProperty = cssProperty;
        _defaultValue = defaultValue;
    }
    
    @Override
    public SkinParameterType getType()
    {
        return SkinParameterType.CSS;
    }
    
    /**
     * Get the CSS files path
     * @return the CSS files path
     */
    public List<Path> getCSSFiles ()
    {
        return _cssFiles;
    }
    
    /**
     * Add a CSS file
     * @param cssFile The CSS file
     */
    public void addCSSFile (Path cssFile)
    {
        for (Path f : _cssFiles)
        {
            if (f.normalize().toString().equals(cssFile.normalize().toString()))
            {
                // The file is already registered
                return;
            }
        }
        _cssFiles.add(cssFile);
    }
    
    /**
     * Get the CSS property
     * @return the CSS file path
     */
    public String getCSSProperty ()
    {
        return _cssProperty;
    }
    
    @Override
    public void apply(Path tempDir, Path modelDir, Object value, String lang) throws SkinParameterException
    {
        for (Path file : _cssFiles)
        {
            String relPath = modelDir.relativize(file).toString();
            Path cssFile = tempDir.resolve(relPath);
            String string = cssFileToString(cssFile);
            
            Pattern pattern = getCSSPattern();
            Matcher matcher = pattern.matcher(string);
            
            StringBuffer sb = new StringBuffer();
            int startIndex = 0;
            while (matcher.find())
            {
                int propertyStart = matcher.start(2);
                
                String beforeValue = string.substring(startIndex, propertyStart);
                String oldValue = string.substring(propertyStart, matcher.end(2));
                sb.append(beforeValue);

                sb.append((String) value + (oldValue.endsWith(" ") ? " " : ""));
                sb.append(string.substring(matcher.end(2), matcher.end(3)));
                
                startIndex = matcher.end(3);
            }
            
            sb.append(string.substring(startIndex));

            try (OutputStream os = Files.newOutputStream(cssFile))
            {
                IOUtils.write(sb.toString(), os, StandardCharsets.UTF_8);
            }
            catch (IOException e)
            {
                throw new SkinParameterException ("Unable to apply css parameter '" + getId() + "'", e);
            }
        }
    }
    
    /**
     * Get the CSS pattern
     * @return The CSS pattern
     */
    protected Pattern getCSSPattern ()
    {
        return  Pattern.compile("\\s*([^,:\\s]*)\\s*:\\s*([^:;!]*)\\s*(?:!important)?\\s*\\/\\*\\s*AMETYS\\s*\\(\\s*\"(" + getId() + ")\"\\s*(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?\\)\\s*\\*\\/\\s*;\\s*?", Pattern.MULTILINE); 
    }
    
    /**
     * Get the css file input stream as String
     * @param cssFile The css file
     * @return The css file content as String
     * @throws SkinParameterException if the css file was not parsable
     */
    protected String cssFileToString (Path cssFile) throws SkinParameterException
    {
        try (InputStream is = Files.newInputStream(cssFile))
        {
            return org.apache.commons.io.IOUtils.toString(is, StandardCharsets.UTF_8);
        }
        catch (IOException e)
        {
            throw new SkinParameterException ("Unable to parse file '" + cssFile.getFileName().toString() + "'", e);
        }
    }
    
    @Override
    public void toSAX(ContentHandler contentHandler, String modelName) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", getId());
        attrs.addCDATAAttribute("type", SkinParameterType.CSS.name().toLowerCase());
        
        XMLUtils.startElement(contentHandler, "parameter", attrs);
        
        getLabel().toSAX(contentHandler, "label");
        getDescription().toSAX(contentHandler, "description");
        XMLUtils.createElement(contentHandler, "property", getCSSProperty());
        
        XMLUtils.endElement(contentHandler, "parameter");
    }
    
    @Override
    public Map<String, Object> toJson(String modelName)
    {
        Map<String, Object> jsonObject = new HashMap<>();
        
        jsonObject.put("id", getId());
        jsonObject.put("type", SkinParameterType.CSS.name().toLowerCase());
        jsonObject.put("label", getLabel());
        jsonObject.put("description", getDescription());
        jsonObject.put("property", getCSSProperty());
        
        return jsonObject;
    }
    
    @Override
    public String getDefaultValue(SkinModel model)
    {
        return _defaultValue;
    }
    
    @Override
    public String getDefaultValue(SkinModel model, String lang)
    {
        return getDefaultValue(model);
    }
}
