/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The survey question DAO.
 */
 Ext.define('Ametys.cms.survey.QuestionDAO', {
 	singleton: true,
 	
 	constructor: function(config)
 	{
		/**
		 * @callable
		 * @member Ametys.cms.survey.QuestionDAO
		 * @method createQuestion 
		 * Creates a survey question.
		 * This calls the method 'createQuestion' of the server DAO 'QuestionDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.pageId The id of the parent page
		 * @param {String} parameters.type The type of question
		 * @param {String} parameters.label The label
 		 * @param {String} parameters.title The title
 		 * @param {String} parameters.regexp The regexp type
 		 * @param {String} parameters.mandatory "true" if the question is mandatory
 		 * @param {String} parameters.columns The columns
 		 * @param {String} parameters.options The options
 		 * @param {String} parameters.otherOption "true" if the question has an "other" option
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.QuestionDAO",
				methodName: "createQuestion",
	     		callback: {
	         		handler: this._createQuestionCb
	     		},
				errorMessage: {
				    msg: "{{i18n PLUGINS_SURVEY_ADD_QUESTION_ERROR}}",
				    category: Ext.getClassName(this)
				}
			}
		);
		
		/**
		 * @callable
		 * @member Ametys.cms.survey.QuestionDAO
		 * @method editQuestion 
		 * Edits a survey question.
		 * This calls the method 'editQuestion' of the server DAO 'QuestionDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the question to edit
		 * @param {String} parameters.type The type of question
		 * @param {String} parameters.label The label
 		 * @param {String} parameters.title The title
 		 * @param {String} parameters.regexp The regexp type
 		 * @param {String} parameters.mandatory "true" if the question is mandatory
 		 * @param {String} parameters.columns The columns
 		 * @param {String} parameters.options The options
 		 * @param {String} parameters.otherOption "true" if the question has an "other" option
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
				role: "org.ametys.plugins.survey.dao.QuestionDAO",
	     		methodName: "editQuestion",
	     		callback: {
	         		handler: this._editQuestionCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_EDIT_QUESTION_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
			}
		);
		
		/**
		 * @callable
		 * @member Ametys.cms.survey.QuestionDAO
		 * @method copyQuestion 
		 * Copies and pastes a survey question.
		 * This calls the method 'copyQuestion' of the server DAO 'QuestionDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.pageId The id of the page, target of the copy
		 * @param {String} parameters.questionId The id of the question to copy
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
				role: "org.ametys.plugins.survey.dao.QuestionDAO",
	     		methodName: "copyQuestion",
	     		callback: {
	         		handler: this._copyQuestionCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_COPY_QUESTION_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
			}
		);
		
		/**
		 * @callable
		 * @member Ametys.cms.survey.QuestionDAO
		 * @method deleteQuestion 
		 * Deletes a survey question.
		 * This calls the method 'deleteQuestion' of the server DAO 'QuestionDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the question to delete
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.QuestionDAO",
	     		methodName: "deleteQuestion",
	     		callback: {
	         		handler: this._deleteQuestionCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_DELETE_QUESTION_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.QuestionDAO
		 * @method getQuestion 
		 * Gets properties of a survey question
		 * This calls the method 'getQuestion' of the server DAO 'QuestionDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the question to retrieve
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.QuestionDAO",
	     		methodName: "getQuestion",
	     		callback: {
	         		handler: Ext.emptyFn
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_TOOL_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.QuestionDAO
		 * @method addRule 
		 * Adds a new rule to a question.
		 * This calls the method 'addRule' of the server DAO 'QuestionDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the question
		 * @param {String} parameters.option The option
		 * @param {String} parameters.rule The rule type
		 * @param {String} parameters.page The page to jump or skip
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.QuestionDAO",
	     		methodName: "addRule",
	     		callback: {
	         		handler: this._addRuleCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_BRANCHES_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.QuestionDAO
		 * @method deleteRule
		 * Deletes a rule to a question. 
		 * This calls the method 'deleteRule' of the server DAO 'QuestionDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the question
		 * @param {String} parameters.option The option to delete
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.QuestionDAO",
	     		methodName: "deleteRule",
	     		callback: {
	         		handler: Ext.emptyFn
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_DELETE_RULE_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.QuestionDAO
		 * @method getRules 
		 * Gets the rules for a survey question.
		 * This calls the method 'getRules' of the server DAO 'QuestionDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the question
		 * @param {String} parameters.number The question number
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.QuestionDAO",
	     		methodName: "getRules",
	     		callback: {
	         		handler: Ext.emptyFn
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_BRANCHES_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
 	},
 	
 	/**
 	 * Callback function called after {@link #createQuestion} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_createQuestionCb: function(response)
 	{
    	Ext.create("Ametys.message.Message", {
    		type: Ametys.message.Message.CREATED,
    		targets: {
    		    id: Ametys.message.MessageTarget.SURVEY_QUESTION,
    			parameters: {
    				id: response.id,
    				pageId: response.pageId,
    				surveyId: response.surveyId,
    				type: response.type
    			}
    		}
    	});
 	},
 	
 	/**
 	 * Callback function called after {@link #editQuestion} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_editQuestionCb: function(response)
 	{
		Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.MODIFIED,
			targets: {
			    id: Ametys.message.MessageTarget.SURVEY_QUESTION,
				parameters: {
					id: response.id,
    				pageId: response.pageId,
    				surveyId: response.surveyId,
    				type: response.type
				}
			}
		});
 	},
 	
 	/**
 	 * Callback function called after {@link #copyQuestion} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_copyQuestionCb: function(response)
 	{
    	Ext.create("Ametys.message.Message", {
    		type: Ametys.message.Message.CREATED,
    		targets: {
    		    id: Ametys.message.MessageTarget.SURVEY_QUESTION,
    			parameters: {
    				id: response.id,
    				pageId: response.pageId,
    				surveyId: response.surveyId,
    				type: response.type
    			}
    		}
    	});
 	},
 	
 	/**
 	 * Callback function called after {@link #deleteQuestion} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_deleteQuestionCb: function(response)
 	{
	    Ext.create("Ametys.message.Message", {
	    	type: Ametys.message.Message.DELETED,
	    	targets: {
	    	    id: Ametys.message.MessageTarget.SURVEY_QUESTION,
	    		parameters: {
					id: response.id,
    				pageId: response.pageId,
    				surveyId: response.surveyId,
    				type: response.type
				}
	    	}
	    });
	 },
	 
	 /**
 	 * Callback function called after {@link #addRule} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
	 _addRuleCb: function(response)
	 {
	 	var error = response['error'];
	 	if (error == 'already-exists')
		{
			Ametys.Msg.show({
	            title: "{{i18n PLUGINS_SURVEY_ADD_RULE_LABEL}}",
	            msg: "{{i18n PLUGINS_SURVEY_ADD_RULE_ALREADY_EXISTS}}",
	            buttons: Ext.Msg.OK,
	            icon: Ext.MessageBox.ERROR
	        });
		}
	 }
 	
 });