/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class to handle actions on survey pages.
 * @private
 */
 Ext.define('Ametys.plugins.survey.PageActions', {
 	singleton: true,
 	
 	/**
 	 * @private
 	 * @property {String} _mode Can be 'new or 'edit'.
 	 */
 	
 	/**
 	 * @private
 	 * @property {Ametys.window.DialogBox} _editDialogBox The dialog box for creating/editing a page.
 	 */
 	 
 	/**
 	 * @private
 	 * @property {Ext.form.Panel} _formPanel The form panel of the dialog box.
 	 */
 	 
 	/**
 	 * @private
 	 * @property {Boolean} _initialized Indicates if the dialog box is initialized.
 	 */
 	
 	/**
 	 * @private
 	 * @property {String} _pageToCopy The id of the page in the clipboard (copied and ready to be pasted).
 	 */
 	
 	/**
 	 * Creates a new page.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	add: function(controller)
 	{
 		this._mode = 'new';
 		this._delayedInitialize();
 		
 		this._initForm(controller.getMatchingTargets()[0]);
 	},
 	
 	/**
 	 * Edits a given page.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	edit: function(controller)
 	{
		this._mode = 'edit';
 		this._delayedInitialize();
 		
 		this._initForm(controller.getMatchingTargets()[0]);
 	},
 	
 	/**
 	 * Copies a given page.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	copy: function(controller)
 	{
 		var target = controller.getMatchingTargets()[0];
 		this._pageToCopy = target.getParameters().id
 	},
 	
 	/**
 	 * Pastes a given page.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	paste: function(controller)
 	{
 		var target = controller.getMatchingTargets()[0];
 		if (!this._pageToCopy)
 		{
 			return;
 		}
 		
 		var surveyId = target.getParameters().id;
 		var pageId = this._pageToCopy;
 		Ametys.cms.survey.PageDAO.copyPage([surveyId, pageId]);
 	},
 	
 	/**
 	 * Removes a given page.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	remove: function(controller)
 	{
 		var target = controller.getMatchingTargets()[0];
 		
 		Ametys.Msg.confirm("{{i18n PLUGINS_SURVEY_DELETE_PAGE_LABEL}}",
 				"{{i18n PLUGINS_SURVEY_DELETE_PAGE_CONFIRM}}",
 				Ext.bind(this._doRemove, this, [target], 1),
 				this
 		);
 	},
 	
 	/**
     * Initializes the dialog box to create or edit a page. 
     * @private
     */
 	_delayedInitialize: function()
 	{
 		if (!this._initialized)
		{
			this._formPanel = this._createFormPanel();
			
			this._editDialogBox = Ext.create('Ametys.window.DialogBox', {
				title: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE}}",
	            iconCls: "ametysicon-website38",
	            
	            width: 550,
		        scrollable: false,
	            
	            items: [this._formPanel],
	            
	            closeAction: 'hide',
	            
	            defaultFocus: 'label',
	            selectDefaultFocus: true,
	            
	            referenceHolder: true,
	            defaultButton: 'validate',
	            
	            buttons: [{
	            	reference: 'validate',
	                text: "{{i18n PLUGINS_SURVEY_DIALOG_BUTTON_OK}}",
	                handler: Ext.bind(this._validate, this)
	            }, {
	                text: "{{i18n PLUGINS_SURVEY_DIALOG_BUTTON_CANCEL}}",
	                handler: Ext.bind(this._cancel, this)
	            }]    
			});
			
			this._initialized = true;
		}
 	},
 	
 	/**
 	 * Creates the form panel of this dialog box.
 	 * @return {Ext.form.Panel} The form panel
 	 * @private
 	 */
 	_createFormPanel: function()
 	{
 		var formPanel = Ext.create('Ext.form.Panel', {
	        defaultType: 'textfield',
	        defaults: {
	        	cls: 'ametys',
	        	labelSeparator: '',
	        	labelAlign: 'right',
	        	labelWidth: 120,
	        	width: '100%',
	        	msgTarget: 'side'
	        },
	        
	        border: false,
	        scrollable: true,
	        
	        items: [
		        {
		        	xtype: 'hidden',
		        	name: 'id'
		        },
		        {
		        	xtype: 'hidden',
		        	name: 'surveyId'
		        },
		        {
		        	itemId: 'label',
		        	name: 'label',
		        	fieldLabel: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE_LABEL}}",
	                ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE_LABEL_DESC}}",
	                allowBlank: false,
	                maxLength: 20
		        },
		        {
		        	name: 'title',
	                fieldLabel: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE_TITLE}}",
	                ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE_TITLE_DESC}}"
		        },
		        {
                    xtype: 'edition.file',
                    name: 'picture',
                    allowSources: [Ametys.form.widget.File.External.SOURCE, Ametys.cms.widget.File.Resource.SOURCE],
                    filter: Ametys.form.widget.File.IMAGE_FILTER,
                    fieldLabel: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE_PICTURE_LABEL}}",
                    ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE_PICTURE_DESC}}"
                },
                {
                    xtype: 'textfield',
                    name: 'picture-alternative',
                    fieldLabel: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE_PICTURE_ALT_LABEL}}",
                    ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE_PICTURE_ALT_DESC}}"
                },
                {
		        	xtype: 'textarea',
		        	name: 'description',
	                fieldLabel: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE_DESCRIPTION}}",
	                ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_PAGE_DESCRIPTION_DESC}}",
	                height: 140
		        }
	        ]
        	
 		});
 		
 		return formPanel;
 	},
 	
 	/**
 	 * @private
 	 * Initializes the form with some optional values.
 	 * @param {Ametys.message.MessageTarget} target If 'new' mode, the parent target. If 'edit' mode, the target to edit.
 	 */
	_initForm: function(target)
 	{
 		var form = this._formPanel.getForm();
		form.reset();
 		
 		if (this._mode === "new") 
        {
 			form.findField('id').setValue('');
 			form.findField('surveyId').setValue(target.getParameters().id);
		    form.findField('label').setValue("{{i18n PLUGINS_SURVEY_DIALOG_PAGE_LABEL_DEFAULT}}");
		    form.findField('label').clearInvalid();
		    form.findField('title').setValue('');
		    form.findField('description').setValue('');
		    form.findField('picture-alternative').setValue();
		    form.findField('picture').setValue();
		    
		    this._editDialogBox.show();
        }
 		else
 		{
 			form.findField('id').setValue(target.getParameters().id);
 			form.findField('surveyId').setValue('');
 			
 			Ametys.cms.survey.PageDAO.getPage([target.getParameters().id], this._setValues, {scope: this});
 		}
 	},
 	
 	/**
 	 * Fill the form with some values.
 	 * @param {Object} data The page.
 	 * @private
 	 */
 	_setValues: function(data)
 	{
 		var form = this._formPanel.getForm();
 		
	    form.findField('title').setValue(data.title);
	    form.findField('label').setValue(data.label);
	    form.findField('description').setValue(data.description);
	    
	    form.findField('picture-alternative').setValue(data.pictureAlternative);
		    
	    if (!Ext.Object.isEmpty(data.picture))
    	{
	    	data.picture.id = 'untouched';
		    form.findField('picture').setValue(data.picture);
    	}
	    else
	    {
	    	form.findField('picture').setValue();
	    }
	    
	    this._editDialogBox.show();
 	},
 	
 	/**
    * @private
    * Handler for the 'ok' button of the dialog box
    */
    _validate: function()
    {
        var form = this._formPanel.getForm();
        if (!form.isValid())
        {
        	return;
        }
        
        var values = form.getValues();
 		
 		if (this._mode == 'new')
 		{
 			var params = [values];
 			Ametys.cms.survey.PageDAO.createPage(params, this._validateCb, {scope: this} );
 		}
 		else
 		{
 			var params = [values];
 			Ametys.cms.survey.PageDAO.editPage(params, this._validateCb, {scope: this} );
 		}
    },
    
    /**
     * Callback function called after creation or edition process
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @param {Object} args The callback arguments
     */
    _validateCb: function(response, args)
    {
    	if (!response['error'])
    	{
    		this._editDialogBox.close();
    	}
    },
 	
 	/**
     * @private
     * Callback for the "cancel" button of the dialog. Close the dialog.
     */
    _cancel: function()
    {
        this._editDialogBox.close();
    },
 	 	
 	/**
 	 * @private
 	 * The action to perform when the user clicks on a button from the removing message box.
 	 * @param {String} btn The pressed button. Can only be 'yes'/'no'
 	 * @param {Ametys.message.MessageTarget} target The target to remove.
 	 */
 	_doRemove: function(btn, target)
 	{
	 	if (btn == 'yes')
	    {
	        if (target == null)
	        {
	            return;
	        }
	        
	        Ametys.cms.survey.PageDAO.deletePage([target.getParameters().id]);
	    }
 	}
 	
 });