/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class to handle actions on survey questions.
 * @private
 */
 Ext.define('Ametys.plugins.survey.QuestionActions', {
 	singleton: true,
 	
 	/**
 	 * @private
 	 * @property {String} _questionToCopy The id of the question in the clipboard (copied and ready to be pasted).
 	 */
 	
 	/**
 	 * Creates a new question of type input text.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	insertInputText: function(controller)
 	{
 		var mode = 'new';
 		Ametys.plugins.survey.question.InputTextDialog.open({
 			mode: mode,
 			target: controller.getMatchingTargets()[0]
 		});
 	},
 	
 	/**
 	 * Creates a new question of type choice.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	insertChoice: function(controller)
 	{
 		var mode = 'new';
 		Ametys.plugins.survey.question.InputChoicesDialog.open({
 			mode: mode,
 			target: controller.getMatchingTargets()[0]
 		});
 	},
 	
 	/**
 	 * Creates a new question of type matrix.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	insertMatrix: function(controller)
 	{
 		var mode = 'new';
 		Ametys.plugins.survey.question.MatrixDialog.open({
 			mode: mode,
 			target: controller.getMatchingTargets()[0]
 		});
 	},
 	
 	/**
 	 * Edits a given question.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	edit: function(controller)
 	{
		var mode = 'edit';
		var target = controller.getMatchingTargets()[0];
		var questionType = target.getParameters().type;
		
		if (questionType == 'FREE_TEXT' || questionType == 'MULTILINE_FREE_TEXT')
		{
			Ametys.plugins.survey.question.InputTextDialog.open({
	 			mode: mode,
	 			target: target
	 		});
		}
		else if (questionType == 'SINGLE_CHOICE' || questionType == 'MULTIPLE_CHOICE')
		{
			Ametys.plugins.survey.question.InputChoicesDialog.open({
	 			mode: mode,
	 			target: target
	 		});
		}
		else if (questionType == 'SINGLE_MATRIX' || questionType == 'MULTIPLE_MATRIX')
		{
			Ametys.plugins.survey.question.MatrixDialog.open({
				mode: mode,
				target: target
			});
		}
 	},
 	
 	/**
 	 * Removes a given question.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	remove: function(controller)
 	{
 		var target = controller.getMatchingTargets()[0];
 		
 		Ametys.Msg.confirm("{{i18n PLUGINS_SURVEY_DELETE_QUESTION_LABEL}}",
 				"{{i18n PLUGINS_SURVEY_DELETE_QUESTION_CONFIRM}}",
 				Ext.bind(this._doRemove, this, [target], 1),
 				this
 		);
 	},
 	
 	/**
 	 * Copies a given question.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	copy: function(controller)
 	{
 		var target = controller.getMatchingTargets()[0];
 		this._questionToCopy = target.getParameters().id
 	},
 	
 	/**
 	 * Pastes a given question.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function.
 	 */
 	paste: function(controller)
 	{
 		var target = controller.getMatchingTargets()[0];
 		if (!this._questionToCopy)
 		{
 			return;
 		}
 		
 		var pageId = target.getParameters().id;
 		var questionId = this._questionToCopy;
 		Ametys.cms.survey.QuestionDAO.copyQuestion([pageId, questionId]);
 	},
 	
 	/**
 	 * @private
 	 * The action to perform when the user clicks on a button from the removing message box.
 	 * @param {String} btn The pressed button. Can only be 'yes'/'no'
 	 * @param {Ametys.message.MessageTarget} target The target to remove.
 	 */
 	_doRemove: function(btn, target)
 	{
	 	if (btn == 'yes')
	    {
	        if (target == null)
	        {
	            return;
	        }
	        
	        Ametys.cms.survey.QuestionDAO.deleteQuestion([target.getParameters().id]);
	    }
 	}
 	
 });