/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.answer;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * Survey errors.
 */
public class SurveyErrors
{
    /** The errors as a Map of field ID -&gt; error messages. */
    protected Map<String, List<I18nizableText>> _errors;
    
    /**
     * Default constructor.
     */
    public SurveyErrors()
    {
        this(new LinkedHashMap<>());
    }
    
    /**
     * Constructor with parameters.
     * @param errors the errors.
     */
    public SurveyErrors(Map<String, List<I18nizableText>> errors)
    {
        this._errors = errors;
    }
    
    /**
     * Get the errors.
     * @return the errors
     */
    public Map<String, List<I18nizableText>> getErrors()
    {
        return _errors;
    }
    
    /**
     * Set the errors.
     * @param errors the errors to set
     */
    public void setErrors(Map<String, List<I18nizableText>> errors)
    {
        this._errors = errors;
    }
    
    /**
     * Add an error.
     * @param fieldId the field ID.
     * @param error the error message.
     */
    public void addError(String fieldId, I18nizableText error)
    {
        if (StringUtils.isNotEmpty(fieldId) && error != null)
        {
            if (_errors.containsKey(fieldId))
            {
                _errors.get(fieldId).add(error);
            }
            else
            {
                List<I18nizableText> errors = new ArrayList<>();
                errors.add(error);
                
                _errors.put(fieldId, errors);
            }
        }
    }
    
    /**
     * Add an error list.
     * @param fieldId the field ID.
     * @param errors the error messages.
     */
    public void addErrors(String fieldId, List<I18nizableText> errors)
    {
        if (StringUtils.isNotEmpty(fieldId) && !errors.isEmpty())
        {
            if (_errors.containsKey(fieldId))
            {
                _errors.get(fieldId).addAll(errors);
            }
            else
            {
                _errors.put(fieldId, errors);
            }
        }
    }
    
    /**
     * Tests if the form has errors.
     * @return true if there are errors, false otherwise.
     */
    public boolean hasErrors()
    {
        for (List<I18nizableText> errList : _errors.values())
        {
            if (!errList.isEmpty())
            {
                return true;
            }
        }
        return false;
    }
    
}
