/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.clientsideelement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.survey.dao.SurveyDAO;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;

/**
 * This element creates a button to send invitations
 *
 */
public class InvitationsClientSideElement extends StaticClientSideElement
{
    /** Repository content */
    protected AmetysObjectResolver _resolver;
    private SurveyDAO _surveyDAO;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _surveyDAO = (SurveyDAO) smanager.lookup(SurveyDAO.ROLE);
    }
    
    /**
     * Get the invitations status for a given survey
     * @param surveyId The survey id
     * @return The invitations status
     */
    @Callable(rights = "Plugins_Survey_Right_LimitAccess", context = "/cms")
    public Map<String, Object> getStatus(String surveyId)
    {
        Map<String, Object> results = new HashMap<>();
        
        Survey survey = _resolver.resolveById(surveyId);
        
        boolean isPrivate = _surveyDAO.isPrivate(survey);
        boolean isValided = survey.isValidated();
        
        String siteName = survey.getSiteName();
        String language = survey.getLanguage();
        Page page = null;
        
        String xpathQuery = "//element(" + siteName + ", ametys:site)/ametys-internal:sitemaps/" + language
                        + "//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.survey.service.Display' and ametys:service_parameters/@ametys:surveyId = '" + surveyId + "']";
        
        AmetysObjectIterable<ZoneItem> zoneItems = _resolver.query(xpathQuery);
        Iterator<ZoneItem> it = zoneItems.iterator();
        if (it.hasNext())
        {
            SitemapElement sitemapElement = it.next().getZone().getSitemapElement();
            if (sitemapElement instanceof Page)
            {
                page = (Page) sitemapElement;
            }
        }
        
        if (!isPrivate)
        {
            List<String> i18nParameters = new ArrayList<>();
            i18nParameters.add(survey.getLabel());
            
            I18nizableText ed = (I18nizableText) this._script.getParameters().get("no-private-description");
            I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
            
            results.put("status", "no-private");
            results.put("description", msg);
        }
        
        if (!isValided)
        {
            List<String> i18nParameters = new ArrayList<>();
            i18nParameters.add(survey.getLabel());
            
            I18nizableText ed = (I18nizableText) this._script.getParameters().get("no-validated-description");
            I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
            
            results.put("status", "no-validated");
            results.put("description", msg);
        }
        
        if (page == null)
        {
            List<String> i18nParameters = new ArrayList<>();
            i18nParameters.add(survey.getLabel());
            
            I18nizableText ed = (I18nizableText) this._script.getParameters().get("no-page-description");
            I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
            
            results.put("status", "no-page");
            results.put("description", msg);
        }
        
        if (isPrivate && isValided && page != null)
        {
            List<String> i18nParameters = new ArrayList<>();
            i18nParameters.add(survey.getLabel());
            
            I18nizableText ed = (I18nizableText) this._script.getParameters().get("all-right-description");
            I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
            
            results.put("page-id", new I18nizableText(page.getId()));
            results.put("status", "all-right");
            results.put("description", msg);
        }
        
        
        return results;
    }
    
}
