/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.dao;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.util.log.SLF4JLoggerAdapter;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import org.ametys.core.observation.Event;
import org.ametys.core.right.ProfileAssignmentStorageExtensionPoint;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.survey.SurveyEvents;
import org.ametys.plugins.survey.data.SurveyAnswer;
import org.ametys.plugins.survey.data.SurveyAnswerDao;
import org.ametys.plugins.survey.data.SurveySession;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyPage;
import org.ametys.plugins.survey.repository.SurveyQuestion;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.ModifiableSitemapElement;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.repository.site.Site;
import org.ametys.web.site.SiteConfigurationExtensionPoint;

import jakarta.mail.MessagingException;

/**
 * DAO for manipulating surveys.
 *
 */
public class SurveyDAO extends AbstractDAO 
{
    /** The Avalon role */
    public static final String ROLE = SurveyDAO.class.getName();
    
    private static final String __OTHER_OPTION = "__opt_other";
    
    /** The survey answer dao. */
    protected SurveyAnswerDao _surveyAnswerDao;
    
    /** The page DAO */
    protected PageDAO _pageDAO;
    
    /** The site configuration. */
    protected SiteConfigurationExtensionPoint _siteConfiguration;
    
    private I18nUtils _i18nUtils;
    private RightManager _rightManager;
    private UserManager _userManager;
    private ProfileAssignmentStorageExtensionPoint _profileAssignmentStorageEP;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _surveyAnswerDao = (SurveyAnswerDao) serviceManager.lookup(SurveyAnswerDao.ROLE);
        _pageDAO = (PageDAO) serviceManager.lookup(PageDAO.ROLE);
        _siteConfiguration = (SiteConfigurationExtensionPoint) serviceManager.lookup(SiteConfigurationExtensionPoint.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _profileAssignmentStorageEP = (ProfileAssignmentStorageExtensionPoint) serviceManager.lookup(ProfileAssignmentStorageExtensionPoint.ROLE);
    }
    
    /**
     * Gets properties of a survey
     * @param id The id of the survey
     * @return The properties
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> getSurvey (String id)
    {
        Survey survey = _resolver.resolveById(id);
        
        return getSurvey(survey);
    }
    
    /**
     * Gets properties of a survey
     * @param survey The survey
     * @return The properties
     */
    public Map<String, Object> getSurvey (Survey survey)
    {
        Map<String, Object> properties = new HashMap<>();
        
        properties.put("id", survey.getId());
        properties.put("label", survey.getLabel());
        properties.put("title", survey.getTitle());
        properties.put("description", survey.getDescription());
        properties.put("endingMessage", survey.getEndingMessage());
        properties.put("private", isPrivate(survey));
        
        if (survey.getRedirection() == null)
        {
            properties.put("redirection", "");
        }
        else
        {
            properties.put("redirection", survey.getRedirection());
        }
        
        properties.putAll(getPictureInfo(survey));
        
        return properties;
    }
    
    /**
     * Determines if the survey is private
     * @param survey The survey
     * @return true if the survey is reading restricted
     */
    public boolean isPrivate (Survey survey)
    {
        return !_rightManager.hasAnonymousReadAccess(survey);
    }
    
    /**
     * Gets the online status of a survey
     * @param id The id of the survey
     * @return A map indicating if the survey is valid and if it is online
     */
    @Callable(rights = {"Plugins_Survey_Right_Handle", "Plugins_Survey_Right_ExportHtml"}, context = "/cms")
    public Map<String, String> isOnline (String id)
    {
        Map<String, String> result = new HashMap<>();
        
        Survey survey = _resolver.resolveById(id);
        
        String xpathQuery = "//element(" + survey.getSiteName() + ", ametys:site)/ametys-internal:sitemaps/" + survey.getLanguage()
                + "//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.survey.service.Display' and ametys:service_parameters/@ametys:surveyId = '" + id + "']";

        AmetysObjectIterable<ZoneItem> zoneItems = _resolver.query(xpathQuery);
        
        result.put("isValid", String.valueOf(survey.isValidated()));
        result.put("isOnline", String.valueOf(zoneItems.iterator().hasNext()));
        
        return result;
    }
    
    /**
     * Gets the children pages of a survey
     * @param id The id of the survey
     * @return A map of pages properties
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public List<Object> getChildren (String id)
    {
        List<Object> result = new ArrayList<>();
        
        Survey survey = _resolver.resolveById(id);
        AmetysObjectIterable<SurveyPage> pages = survey.getChildren();
        for (SurveyPage page : pages)
        {
            result.add(_pageDAO.getPage(page));
        }
        
        return result;
    }
    
    /**
     * Creates a survey.
     * @param values The survey values
     * @param siteName The site name
     * @param language The language
     * @return The id of the created survey
     * @throws Exception if an error occurs during the survey creation process
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> createSurvey (Map<String, Object> values, String siteName, String language) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        ModifiableTraversableAmetysObject rootNode = getSurveyRootNode(siteName, language);
        
        String label = StringUtils.defaultString((String) values.get("label"));
        
        // Find unique name
        String originalName = NameHelper.filterName(label);
        String name = originalName;
        int index = 2;
        while (rootNode.hasChild(name))
        {
            name = originalName + "-" + (index++);
        }
        
        Survey survey = rootNode.createChild(name, "ametys:survey");
        _setValues(survey, values);
        
        rootNode.saveChanges();

        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", survey);
        _observationManager.notify(new Event(SurveyEvents.SURVEY_CREATED, _getCurrentUser(), eventParams));
        
        // Set public access
        _setPublicAccess(survey);
        
        result.put("id", survey.getId());
        
        return result;
    }
    
    /**
     * Edits a survey.
     * @param values The survey values
     * @param siteName The site name
     * @param language The language
     * @return The id of the edited survey
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> editSurvey (Map<String, Object> values, String siteName, String language)
    {
        Map<String, String> result = new HashMap<>();
        
        String id = StringUtils.defaultString((String) values.get("id"));
        Survey survey = _resolver.resolveById(id);
        
        _setValues(survey, values);
        
        survey.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", survey);
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", survey.getId());
        
        return result;
    }
    
    private void _setPublicAccess (Survey survey)
    {
        _profileAssignmentStorageEP.allowProfileToAnonymous(RightManager.READER_PROFILE_ID, survey);
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT, survey);
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_PROFILES, Collections.singleton(RightManager.READER_PROFILE_ID));
        
        _observationManager.notify(new Event(org.ametys.core.ObservationConstants.EVENT_ACL_UPDATED, _currentUserProvider.getUser(), eventParams));
    }
    
    private void _setValues (Survey survey, Map<String, Object> values)
    {
        survey.setTitle(StringUtils.defaultString((String) values.get("title")));
        survey.setLabel(StringUtils.defaultString((String) values.get("label")));
        survey.setDescription(StringUtils.defaultString((String) values.get("description")));
        survey.setEndingMessage(StringUtils.defaultString((String) values.get("endingMessage")));
        
        survey.setPictureAlternative(StringUtils.defaultString((String) values.get("picture-alternative")));
        setPicture(survey, StringUtils.defaultString((String) values.get("picture")));
    }
    
    /**
     * Copies and pastes a survey.
     * @param surveyId The id of the survey to copy
     * @param label The label 
     * @param title The title
     * @return The id of the created survey
     * @throws Exception if an error occurs during the survey copying process
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> copySurvey(String surveyId, String label, String title) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        String originalName = NameHelper.filterName(label);
        
        Survey surveyToCopy = _resolver.resolveById(surveyId);
        
        ModifiableTraversableAmetysObject rootNode = getSurveyRootNode(surveyToCopy.getSiteName(), surveyToCopy.getLanguage());
        
        // Find unique name
        String name = originalName;
        int index = 2;
        while (rootNode.hasChild(name))
        {
            name = originalName + "-" + (index++);
        }
        
        Survey survey = surveyToCopy.copyTo(rootNode, name);
        survey.setLabel(label);
        survey.setTitle(title);
        
        // Update rules references after copy
        updateReferencesAfterCopy (surveyToCopy, survey);
        
        rootNode.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", survey);
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        _setPublicAccess(survey);
        
        result.put("id", survey.getId());
        
        return result;
    }
    
    /**
     * Deletes a survey.
     * @param id The id of the survey to delete
     * @return The id of the deleted survey
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> deleteSurvey (String id)
    {
        Map<String, String> result = new HashMap<>();
        
        Survey survey = _resolver.resolveById(id);
        ModifiableAmetysObject parent = survey.getParent();
        
        String siteName = survey.getSiteName();
        
        survey.remove();
        
        _surveyAnswerDao.deleteSessions(id);
        
        parent.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("siteName", siteName);
        _observationManager.notify(new Event(SurveyEvents.SURVEY_DELETED, _getCurrentUser(), eventParams));
        
        result.put("id", id);
        
        return result;
    }
    
    /**
     * Validates a survey.
     * @param id The id of the survey to validate
     * @return The id of the validated survey
     */
    @Callable(rights = "Plugins_Survey_Right_Validate", context = "/cms")
    public Map<String, String> validateSurvey (String id)
    {
        Map<String, String> result = new HashMap<>();
        
        Survey survey = _resolver.resolveById(id);
        survey.setValidated(true);
        survey.setValidationDate(new Date());
        survey.saveChanges();
        
        result.put("id", survey.getId());
        
        return result;
    }
    
    /**
     * Reinitializes a survey.
     * @param id The id of the survey to validate
     * @param invalidate True to invalidate the survey
     * @return The id of the reinitialized survey
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> reinitSurvey (String id, boolean invalidate)
    {
        Map<String, Object> result = new HashMap<>();
        
        Survey survey = _resolver.resolveById(id);
        
        if (invalidate)
        {
            // Invalidate survey
            survey.setValidated(false);
            survey.setValidationDate(null);
            
            result.put("modifiedPages", removeExistingServices (survey.getSiteName(), survey.getLanguage(), id));
        }
        
        // Re-initialize the survey
        survey.reinit();
        survey.saveChanges();
        
        // Send observer to clear survey service page cache
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", survey);
        _observationManager.notify(new Event(SurveyEvents.SURVEY_REINITIALIZED, _getCurrentUser(), eventParams));

        // Delete all answers
        _surveyAnswerDao.deleteSessions(id);
        
        
        result.put("id", survey.getId());
        
        return result;
    }
    
    /**
     * Sets a new redirection page to the survey.
     * @param surveyId The id of the survey to edit.
     * @param pageId The id of the redirection page.
     * @return The id of the edited survey
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> setRedirection (String surveyId, String pageId)
    {
        Map<String, String> result = new HashMap<>();
        
        Survey survey = _resolver.resolveById(surveyId);
        if (StringUtils.isNotEmpty(pageId))
        {
            survey.setRedirection(pageId);
        }
        else
        {
            // Remove redirection
            survey.setRedirection(null);
        }
        survey.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", survey);
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", survey.getId());
        
        return result;
    }
    
    /**
     * Moves an element of the survey.
     * @param id The id of the element to move.
     * @param oldParent The id of the element's parent.
     * @param newParent The id of the new element's parent.
     * @param index The index where to move. null to place the element at the end.
     * @return A map with the ids of the element, the old parent and the new parent
     * @throws Exception if an error occurs when moving an element of the survey
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> moveObject (String id, String oldParent, String newParent, long index) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        JCRAmetysObject aoMoved = _resolver.resolveById(id);
        DefaultTraversableAmetysObject newParentAO = _resolver.resolveById(newParent);
        JCRAmetysObject  brother = null;
        long size = newParentAO.getChildren().getSize();
        if (index != -1 && index < size)
        {
            brother = newParentAO.getChildAt(index);
        }
        else if (index >= size)
        {
            brother = newParentAO.getChildAt(Math.toIntExact(size) - 1);
        }
        Survey oldSurvey = getParentSurvey(aoMoved);
        if (oldSurvey != null)
        {
            result.put("oldSurveyId", oldSurvey.getId());
        }
        
        if (oldParent.equals(newParent) && brother != null)
        {
            Node node = aoMoved.getNode();
            String name = "";
            try
            {
                name = brother.getName();
                node.getParent().orderBefore(node.getName(), name);
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException(String.format("Unable to order AmetysOject '%s' before sibling '%s'", this, name), e);
            }
        }
        else
        {
            Node node = aoMoved.getNode();
            
            String name = node.getName();
            // Find unused name on new parent node
            int localIndex = 2;
            while (newParentAO.hasChild(name))
            {
                name = node.getName() + "-" + localIndex++;
            }
            
            node.getSession().move(node.getPath(), newParentAO.getNode().getPath() + "/" + name);
            
            if (brother != null)
            {
                node.getParent().orderBefore(node.getName(), brother.getName());
            }
        }
        
        if (newParentAO.needsSave())
        {
            newParentAO.saveChanges();
        }
        
        Survey survey = getParentSurvey(aoMoved);
        if (survey != null)
        {
            result.put("newSurveyId", survey.getId());
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put("survey", survey);
            _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        }
        
        result.put("id", id);
        
        if (aoMoved instanceof SurveyPage)
        {
            result.put("type", "page");
        }
        else if (aoMoved instanceof SurveyQuestion)
        {
            result.put("type", "question");
            result.put("questionType", ((SurveyQuestion) aoMoved).getType().name());
        }
        
        result.put("newParentId", newParentAO.getId());
        result.put("oldParentId", oldParent);
        
        return result;
    }
    
    
    
    /**
     * Sends invitations emails.
     * @param surveyId The id of the survey.
     * @param message The message content.
     * @param siteName The site name.
     * @return An empty map
     */
    @Callable(rights = "Plugins_Survey_Right_LimitAccess", context = "/cms")
    public Map<String, Object> sendInvitations (String surveyId, String message, String siteName)
    {
        String subject = getMailSubject();
        String body = getMailBody(surveyId, message, siteName);
        
        Site site = _siteManager.getSite(siteName);
        String defaultFromValue = Config.getInstance().getValue("smtp.mail.from");
        String from = site.getValue("site-mail-from", false, defaultFromValue);
       
        Survey survey = _resolver.resolveById(surveyId);
        Set<UserIdentity> allowedUsers = _rightManager.getReadAccessAllowedUsers(survey).resolveAllowedUsers(false);
        
        for (UserIdentity userIdentity : allowedUsers)
        {
            User user = _userManager.getUser(userIdentity);
            if (user != null && StringUtils.isNotEmpty(user.getEmail()) && !hasAlreadyAnswered(surveyId, userIdentity))
            {
                try
                {
                    String finalMessage = StringUtils.replace(body, "[name]", user.getFullName());
                    
                    SendMailHelper.newMail()
                                  .withSubject(subject)
                                  .withTextBody(finalMessage)
                                  .withSender(from)
                                  .withRecipient(user.getEmail())
                                  .sendMail();
                }
                catch (MessagingException | IOException e) 
                {
                    new SLF4JLoggerAdapter(LoggerFactory.getLogger(this.getClass())).error("Unable to send mail to user " + user.getEmail(), e);
                }
            }
        }
        
        return new HashMap<>();
    }
    
    /**
     * Generates statistics on each question of a survey.
     * @param id The survey id
     * @return A map containing the statistics
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> getStatistics(String id)
    {
        Map<String, Object> statistics = new HashMap<>();
        
        Survey survey = _resolver.resolveById(id);
        
        int sessionCount = _surveyAnswerDao.getSessionCount(id);
        List<SurveySession> sessions = _surveyAnswerDao.getSessionsWithAnswers(id);
        
        statistics.put("id", id);
        statistics.put("title", survey.getTitle());
        statistics.put("sessions", sessionCount);
        
        Map<String, Map<String, Map<String, Object>>> statsMap = createStatsMap(survey);
        
        dispatchStats(survey, sessions, statsMap);
        
        List statsList = statsToArray(survey, statsMap);
        
        statistics.put("questions", statsList);
        
        return statistics;
    }
    
    /**
     * Remove the existing services if exists
     * @param siteName The site name
     * @param lang The language
     * @param surveyId The id of survey
     * @return The list of modified pages ids
     */
    protected List<String> removeExistingServices (String siteName, String lang, String surveyId)
    {
        List<String> modifiedPages = new ArrayList<>();
        for (ModifiableZoneItem zoneItem : getSurveyZoneItems(siteName, lang, surveyId))
        {
            ModifiableSitemapElement sitemapElement = (ModifiableSitemapElement) zoneItem.getZone().getSitemapElement();
            
            String id = zoneItem.getId();
            ZoneType type = zoneItem.getType();
            
            zoneItem.remove();
            sitemapElement.saveChanges();
            modifiedPages.add(sitemapElement.getId());
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_SITEMAP_ELEMENT, sitemapElement);
            eventParams.put(ObservationConstants.ARGS_ZONE_ITEM_ID, id);
            eventParams.put(ObservationConstants.ARGS_ZONE_TYPE, type);
            _observationManager.notify(new Event(ObservationConstants.EVENT_ZONEITEM_DELETED, _getCurrentUser(), eventParams));
        }
        
        return modifiedPages;
    }
    
    /**
     * Get all zone items which contains the survey
     * @param siteName the site name
     * @param lang the language
     * @param surveyId the survey id
     * @return the zone items
     */
    public AmetysObjectIterable<ModifiableZoneItem> getSurveyZoneItems(String siteName, String lang, String surveyId)
    {
        String xpathQuery = "//element(" + siteName + ", ametys:site)/ametys-internal:sitemaps/" + lang
                + "//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.survey.service.Display' and ametys:service_parameters/@ametys:surveyId = '" + surveyId + "']";

        return _resolver.query(xpathQuery);
    }
    
    /**
     * Get the survey containing the given object.
     * @param obj the object.
     * @return the parent Survey.
     */
    protected Survey getParentSurvey(JCRAmetysObject obj)
    {
        try
        {
            JCRAmetysObject currentAo = obj.getParent();
            
            while (!(currentAo instanceof Survey))
            {
                currentAo = currentAo.getParent();
            }
            
            if (currentAo instanceof Survey)
            {
                return (Survey) currentAo;
            }
        }
        catch (AmetysRepositoryException e)
        {
            // Ignore, just return null.
        }
        
        return null;
    }
    
    /**
     * Create the statistics Map for a survey.
     * @param survey the survey.
     * @return the statistics Map. It is of the following form: questionId -&gt; optionId -&gt;choiceId -&gt; count.
     */
    protected Map<String, Map<String, Map<String, Object>>> createStatsMap(Survey survey)
    {
        Map<String, Map<String, Map<String, Object>>> stats = new LinkedHashMap<>();
        
        for (SurveyQuestion question : survey.getQuestions())
        {
            Map<String, Map<String, Object>> questionValues = new LinkedHashMap<>();
            stats.put(question.getName(), questionValues);
            
            switch (question.getType())
            {
                case FREE_TEXT:
                case MULTILINE_FREE_TEXT:
                    Map<String, Object> values = new LinkedHashMap<>();
                    questionValues.put("values", values);
                    values.put("answered", 0);
                    values.put("empty", 0);
                    break;
                case SINGLE_CHOICE:
                case MULTIPLE_CHOICE:
                    values = new LinkedHashMap<>();
                    questionValues.put("values", values);
                    
                    for (String option : question.getOptions().keySet())
                    {
                        values.put(option, 0);
                    }
                    
                    if (question.hasOtherOption())
                    {
                        // Add other option
                        values.put(__OTHER_OPTION, 0);
                    }
                    break;
                case SINGLE_MATRIX:
                case MULTIPLE_MATRIX:
                    for (String option : question.getOptions().keySet())
                    {
                        values = new LinkedHashMap<>();
                        questionValues.put(option, values);
                        
                        for (String column : question.getColumns().keySet())
                        {
                            values.put(column, 0);
                        }
                    }
                    break;
                default:
                    break;
            }
        }
        
        return stats;
    }
    
    /**
     * Dispatch the survey user sessions (input) in the statistics map.
     * @param survey the survey.
     * @param sessions the user sessions.
     * @param stats the statistics Map to fill.
     */
    protected void dispatchStats(Survey survey, Collection<SurveySession> sessions, Map<String, Map<String, Map<String, Object>>> stats)
    {
        for (SurveySession session : sessions)
        {
            for (SurveyAnswer answer : session.getAnswers())
            {
                SurveyQuestion question = survey.getQuestion(answer.getQuestionId());
                if (question != null)
                {
                    Map<String, Map<String, Object>> questionStats = stats.get(answer.getQuestionId());
                    
                    Map<String, Set<String>> valueMap = getValueMap(question, answer.getValue());
                    
                    switch (question.getType())
                    {
                        case FREE_TEXT:
                        case MULTILINE_FREE_TEXT:
                            dispatchTextStats(session, questionStats, valueMap);
                            break;
                        case SINGLE_CHOICE:
                        case MULTIPLE_CHOICE:
                            dispatchChoiceStats(session, questionStats, valueMap);
                            break;
                        case SINGLE_MATRIX:
                        case MULTIPLE_MATRIX:
                            dispatchMatrixStats(session, questionStats, valueMap);
                            break;
                        default:
                            break;
                    }
                }
            }
        }
    }

    /**
     * Dispatch stats on a text question.
     * @param session the survey session.
     * @param questionStats the Map to fill with the stats.
     * @param valueMap the value map.
     */
    protected void dispatchTextStats(SurveySession session, Map<String, Map<String, Object>> questionStats, Map<String, Set<String>> valueMap)
    {
        Map<String, Object> optionStats = questionStats.get("values");
        
        if (valueMap.containsKey("values"))
        {
            String singleValue = valueMap.get("values").iterator().next();
            boolean isBlank = StringUtils.isBlank(singleValue);
            String stat = isBlank ? "empty" : "answered";
            
            int iValue = (Integer) optionStats.get(stat);
            optionStats.put(stat, iValue + 1);
            
            if (!isBlank)
            {
                optionStats.put(Integer.toString(session.getId()), singleValue);
            }
        }
    }
    
    /**
     * Dispatch stats on a choice question.
     * @param session the survey session.
     * @param questionStats the Map to fill with the stats.
     * @param valueMap the value map.
     */
    protected void dispatchChoiceStats(SurveySession session, Map<String, Map<String, Object>> questionStats, Map<String, Set<String>> valueMap)
    {
        Map<String, Object> optionStats = questionStats.get("values");
        
        if (valueMap.containsKey("values"))
        {
            for (String value : valueMap.get("values"))
            {
                if (optionStats.containsKey(value))
                {
                    int iValue = (Integer) optionStats.get(value);
                    optionStats.put(value, iValue + 1);
                }
                else
                {
                    int iValue = (Integer) optionStats.get(__OTHER_OPTION);
                    optionStats.put(__OTHER_OPTION, iValue + 1);
                }
            }
        }
    }
    
    /**
     * Dispatch stats on a matrix question.
     * @param session the survey session.
     * @param questionStats the Map to fill with the stats.
     * @param valueMap the value map.
     */
    protected void dispatchMatrixStats(SurveySession session, Map<String, Map<String, Object>> questionStats, Map<String, Set<String>> valueMap)
    {
        for (String option : valueMap.keySet())
        {
            Map<String, Object> optionStats = questionStats.get(option);
            if (optionStats != null)
            {
                for (String value : valueMap.get(option))
                {
                    if (optionStats.containsKey(value))
                    {
                        int iValue = (Integer) optionStats.get(value);
                        optionStats.put(value, iValue + 1);
                    }
                }
            }
            
        }
    }
    
    /**
     * Transforms the statistics map into an array with some info.
     * @param survey The survey
     * @param stats The filled statistics Map.
     * @return A list of statistics.
     */
    protected List<Map<String, Object>> statsToArray (Survey survey, Map<String, Map<String, Map<String, Object>>> stats)
    {
        List<Map<String, Object>> result = new ArrayList<>();
        
        for (String questionId : stats.keySet())
        {
            Map<String, Object> questionMap = new HashMap<>();
            
            SurveyQuestion question = survey.getQuestion(questionId);
            Map<String, Map<String, Object>> questionStats = stats.get(questionId);
            
            questionMap.put("id", questionId);
            questionMap.put("title", question.getTitle());
            questionMap.put("type", question.getType());
            questionMap.put("mandatory", question.isMandatory());
            
            List<Object> options = new ArrayList<>();
            for (String optionId : questionStats.keySet())
            {
                Map<String, Object> option = new HashMap<>();
                
                option.put("id", optionId);
                option.put("label", getOptionLabel(question, optionId));
                
                questionStats.get(optionId).entrySet();
                List<Object> choices = new ArrayList<>();
                for (Entry<String, Object> choice : questionStats.get(optionId).entrySet())
                {
                    Map<String, Object> choiceMap = new HashMap<>();
                    
                    String choiceId = choice.getKey();
                    choiceMap.put("value", choiceId);
                    choiceMap.put("label", getChoiceLabel(question, choiceId));
                    choiceMap.put("count", choice.getValue());
                    
                    choices.add(choiceMap);
                }
                option.put("choices", choices);
                
                options.add(option);
            }
            questionMap.put("options", options);
            
            result.add(questionMap);
        }
        
        return result;
    }
    
    /**
     * Get an option label, depending on the question type.
     * @param question the question.
     * @param optionId the option ID.
     * @return the question label, can be the empty string.
     */
    protected String getOptionLabel(SurveyQuestion question, String optionId)
    {
        String label = "";
        
        switch (question.getType())
        {
            case FREE_TEXT:
            case MULTILINE_FREE_TEXT:
            case SINGLE_CHOICE:
            case MULTIPLE_CHOICE:
                break;
            case SINGLE_MATRIX:
            case MULTIPLE_MATRIX:
                label = question.getOptions().get(optionId);
                break;
            default:
                break;
        }
        
        return label;
    }
    
    /**
     * Get an option label, depending on the question type.
     * @param question the question.
     * @param choiceId the choice id.
     * @return the option label, can be the empty string.
     */
    protected String getChoiceLabel(SurveyQuestion question, String choiceId)
    {
        String label = "";
        
        switch (question.getType())
        {
            case FREE_TEXT:
            case MULTILINE_FREE_TEXT:
                break;
            case SINGLE_CHOICE:
            case MULTIPLE_CHOICE:
                if (question.getOptions().containsKey(choiceId))
                {
                    label = question.getOptions().get(choiceId);
                }
                else if (question.hasOtherOption())
                {
                    label = _i18nUtils.translate(new I18nizableText("plugin.survey", "PLUGINS_SURVEY_STATISTICS_OTHER_OPTION"));
                }
                break;
            case SINGLE_MATRIX:
            case MULTIPLE_MATRIX:
                label = question.getColumns().get(choiceId);
                break;
            default:
                break;
        }
        
        return label;
    }
    
    /**
     * Get the user-input value as a Map from the database value, which is a single serialized string.
     * @param question the question.
     * @param value the value from the database.
     * @return the value as a Map.
     */
    protected Map<String, Set<String>> getValueMap(SurveyQuestion question, String value)
    {
        Map<String, Set<String>> values = new HashMap<>();
        
        if (value != null)
        {
            switch (question.getType())
            {
                case SINGLE_MATRIX:
                case MULTIPLE_MATRIX:
                    String[] entries = StringUtils.split(value, ';');
                    for (String entry : entries)
                    {
                        String[] keyValue = StringUtils.split(entry, ':');
                        if (keyValue.length == 2 && StringUtils.isNotEmpty(keyValue[0]))
                        {
                            Set<String> valueSet = new HashSet<>(Arrays.asList(StringUtils.split(keyValue[1], ',')));
                            
                            values.put(keyValue[0], valueSet);
                        }
                    }
                    break;
                case SINGLE_CHOICE:
                case MULTIPLE_CHOICE:
                    Set<String> valueSet = new HashSet<>(Arrays.asList(StringUtils.split(value, ',')));
                    values.put("values", valueSet);
                    break;
                case FREE_TEXT:
                case MULTILINE_FREE_TEXT:
                default:
                    values.put("values", Collections.singleton(value));
                    break;
            }
        }
        
        return values;
    }
    
    /**
     * Determines if the user has already answered to the survey
     * @param surveyId The survey id
     * @param user the user
     * @return <code>true</code> if the user has already answered
     */
    protected boolean hasAlreadyAnswered (String surveyId, UserIdentity user)
    {
        if (user != null && StringUtils.isNotBlank(user.getLogin()) && StringUtils.isNotBlank(user.getPopulationId()))
        {
            SurveySession userSession = _surveyAnswerDao.getSession(surveyId, user);
            
            if (userSession != null)
            {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Get the email subject
     * @return The subject
     */
    protected String getMailSubject ()
    {
        return _i18nUtils.translate(new I18nizableText("plugin.survey", "PLUGINS_SURVEY_SEND_MAIL_SUBJECT"));
    }
    
    /**
     * Get the email body
     * @param surveyId The survey id
     * @param message The message
     * @param siteName The site name
     * @return The text body
     */
    protected String getMailBody (String surveyId, String message, String siteName)
    {
        Site site = _siteManager.getSite(siteName);
        String surveyURI = getSurveyUri(surveyId, siteName);
        
        String replacedMessage = StringUtils.replace(message, "[link]", surveyURI);
        replacedMessage = StringUtils.replace(replacedMessage, "[site]", site.getTitle());
        
        return replacedMessage;
    }
    
    /**
     * Get the survey page uri
     * @param surveyId The survey id
     * @param siteName The site name
     * @return The survey absolute uri
     */
    protected String getSurveyUri (String surveyId, String siteName)
    {
        Site site = _siteManager.getSite(siteName);
        Survey survey = _resolver.resolveById(surveyId);
        
        Page page = null;
        String xpathQuery = "//element(" + siteName + ", ametys:site)/ametys-internal:sitemaps/" + survey.getLanguage()
                        + "//element(*, ametys:zoneItem)[@ametys-internal:service = 'org.ametys.survey.service.Display' and ametys:service_parameters/@ametys:surveyId = '" + surveyId + "']";
        
        AmetysObjectIterable<ZoneItem> zoneItems = _resolver.query(xpathQuery);
        Iterator<ZoneItem> it = zoneItems.iterator();
        if (it.hasNext())
        {
            page = (Page) it.next().getZone().getSitemapElement();
        }
        
        if (page != null)
        {
            return site.getUrl() + "/" + page.getSitemap().getName() + "/" + page.getPathInSitemap() + ".html";
        }
        
        return "";
    }
}
