/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.observation;

import javax.jcr.Node;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.survey.dao.SurveyDAO;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.web.cache.AbstractSiteCacheObserver;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.site.Site;

/**
 * Abstract {@link Observer} for observing survey changes in order to invalidate cache on front-office.
 */
public abstract class AbstractSurveyCacheObserver extends AbstractSiteCacheObserver
{
    /** The survey DAO */
    protected SurveyDAO _surveyDAO;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _surveyDAO = (SurveyDAO) manager.lookup(SurveyDAO.ROLE);
    }

    @Override
    protected void _internalObserve(Event event, Site site, Session liveSession) throws Exception
    {
        Survey survey = getSurveyFromEvent(event);
        if (survey != null)
        {
            for (ModifiableZoneItem zoneItem : _surveyDAO.getSurveyZoneItems(survey.getSiteName(), survey.getLanguage(), survey.getId()))
            {
                SitemapElement sitemapElement = zoneItem.getZone().getSitemapElement();
                if (sitemapElement instanceof JCRAmetysObject
                    && sitemapElement instanceof ModifiablePage page)
                {
                    JCRAmetysObject jcrPage = (JCRAmetysObject) page;
                    Node pageNode = jcrPage.getNode();

                    if (liveSession.itemExists(pageNode.getPath()))
                    {
                        if (getLogger().isInfoEnabled())
                        {
                            getLogger().info("Survey re-initialized: " + event + ", invalidating cache");
                        }

                        _cachePolicy.invalidateCacheOnPageModification(page);
                    }
                }
            }
        }
    }

    @Override
    protected Site _getSite(Event event)
    {
        Survey survey = getSurveyFromEvent(event);
        if (survey != null)
        {
            return survey.getSite();
        }

        return null;
    }

    /**
     * Get the survey from the Event
     * @param event the event
     * @return the survey
     */
    protected abstract Survey getSurveyFromEvent(Event event);
}
