/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.repository;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;

import org.ametys.cms.indexing.solr.SolrAclCacheUninfluentialObject;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ChainedAmetysObjectIterable;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.survey.SurveyDateUtils;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.site.Site;

/**
 * {@link AmetysObject} representing a survey
 */
@SolrAclCacheUninfluentialObject
public class Survey extends AbstractSurveyElement<SurveyFactory> implements SiteAwareAmetysObject
{
    /** Constants for title metadata. */
    private static final String __PROPERTY_TITLE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":title";
    /** Constants for header metadata. */
    private static final String __PROPERTY_LABEL = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":label";
    /** Constants for description metadata. */
    private static final String __PROPERTY_DESC = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":description";
    /** Constants for description metadata. */
    private static final String __PROPERTY_ENDING_MSG = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":endingMessage";
    /** Constants for private metadata. */
    private static final String __PROPERTY_VALIDATED = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":validated";
    /** Constants for private metadata. */
    private static final String __PROPERTY_VALIDATION_DATE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":validationDate";
    /** Constants for start date metadata. */
    private static final String __PROPERTY_START_DATE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":startDate";
    /** Constants for end date metadata. */
    private static final String __PROPERTY_END_DATE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":endDate";
    /** Constants for private metadata. */
    private static final String __PROPERTY_REDIRECTION = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":redirection";
    /** Constants for reinit date metadata. */
    private static final String __PROPERTY_REINIT_DATE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":reinitDate";
    
    /**
     * Creates a {@link Survey}.
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link SurveyFactory} which creates the AmetysObject.
     */
    public Survey(Node node, String parentPath, SurveyFactory factory)
    {
        super(node, parentPath, factory);
    }

    /**
     * Retrieves the title.
     * @return the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getTitle() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_TITLE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get title property", e);
        }
    }
    
    /**
     * Set the title.
     * @param title the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setTitle(String title) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_TITLE, title);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set title property", e);
        }
    }
    
    /**
     * Retrieves the survey label.
     * @return the the survey label.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getLabel() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_LABEL).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get label property", e);
        }
    }
    
    /**
     * Set the survey label.
     * @param label the survey label.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setLabel(String label) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_LABEL, label);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set label property", e);
        }
    }
    
    /**
     * Retrieves the description.
     * @return the description.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getDescription() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_DESC).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get description property", e);
        }
    }
    
    /**
     * Set the description.
     * @param description the description.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setDescription(String description) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_DESC, description);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set description property", e);
        }
    }
    
    /**
     * Retrieves the ending message.
     * @return the ending message.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getEndingMessage() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_ENDING_MSG).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get ending message property", e);
        }
    }
    
    /**
     * Set the ending message.
     * @param message the ending message.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setEndingMessage(String message) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_ENDING_MSG, message);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set ending message property", e);
        }
    }
    
    /**
     * Determines if the survey is validated.
     * @return true if the survey is validated.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean isValidated() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_VALIDATED).getBoolean();
        }
        catch (PathNotFoundException e)
        {
            return false;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get validated property", e);
        }
    }
    
    /**
     * Valid or invalid survey
     * @param validated true to validate the survey
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setValidated(boolean validated) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_VALIDATED, validated);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to validate survey", e);
        }
    }
    
    /**
     * Set the date of validation
     * @param date The date of validation
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setValidationDate(Date date) throws AmetysRepositoryException
    {
        try
        {
            if (date != null)
            {
                GregorianCalendar calendar = new GregorianCalendar();
                calendar.setTime(date);
                
                getNode().setProperty(__PROPERTY_VALIDATION_DATE, calendar);
            }
            else if (getNode().hasProperty(__PROPERTY_VALIDATION_DATE))
            {
                getNode().getProperty(__PROPERTY_VALIDATION_DATE).remove();
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set date of validation", e);
        }
    }
    
    /**
     * Get the date of validation
     * @return the date of validation
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Date getValidationDate () throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_VALIDATION_DATE).getDate().getTime();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get validation date property", e);
        }
    }

    /**
     * Set the start date
     * @param date The start date
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setStartDate(Date date) throws AmetysRepositoryException
    {
        try
        {
            if (date != null)
            {
                GregorianCalendar calendar = new GregorianCalendar();
                calendar.setTime(date);
                
                getNode().setProperty(__PROPERTY_START_DATE, calendar);
            }
            else if (getNode().hasProperty(__PROPERTY_START_DATE))
            {
                getNode().getProperty(__PROPERTY_START_DATE).remove();
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set start date", e);
        }
    }
    
    /**
     * Get the start date
     * @return the start date
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Date getStartDate () throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_START_DATE).getDate().getTime();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get start date property", e);
        }
    }
    
    /**
     * Set the end date
     * @param date The end date
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setEndDate(Date date) throws AmetysRepositoryException
    {
        try
        {
            if (date != null)
            {
                GregorianCalendar calendar = new GregorianCalendar();
                calendar.setTime(date);
                
                getNode().setProperty(__PROPERTY_END_DATE, calendar);
            }
            else if (getNode().hasProperty(__PROPERTY_END_DATE))
            {
                getNode().getProperty(__PROPERTY_END_DATE).remove();
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set end date", e);
        }
    }

    /**
     * Get the end date
     * @return the end date
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Date getEndDate () throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_END_DATE).getDate().getTime();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get end date property", e);
        }
    }
    
    /**
     * Retrieves the redirection.
     * @return the page id of redirection or null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getRedirection() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_REDIRECTION).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get redirection property", e);
        }
    }
    
    /**
     * Set the redirection.
     * @param pageId the page id. Can be null to delete redirection
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setRedirection(String pageId) throws AmetysRepositoryException
    {
        try
        {
            if (pageId == null)
            {
                if (getNode().hasProperty(__PROPERTY_REDIRECTION))
                {
                    getNode().getProperty(__PROPERTY_REDIRECTION).remove();
                }
            }
            else
            {
                getNode().setProperty(__PROPERTY_REDIRECTION, pageId);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set redirection property", e);
        }
    }
    
    /**
     * Re-initialize the survey. Just set the date of the last re-initialization of the survey.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void reinit() throws AmetysRepositoryException
    {
        try
        {
            ZonedDateTime zdt = ZonedDateTime.now(ZoneId.of("UTC"));
            getNode().setProperty(__PROPERTY_REINIT_DATE, SurveyDateUtils.asCalendar(zdt)); 
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set re-initialization date", e);
        }
    }

    /**
     * Get the date of the last re-initialization of the survey
     * @return the re-initialization date
     * @throws AmetysRepositoryException if an error occurs.
     */
    public ZonedDateTime getReinitDate () throws AmetysRepositoryException
    {
        try
        {
            Calendar calendar = getNode().getProperty(__PROPERTY_REINIT_DATE).getDate();
            return SurveyDateUtils.asZonedDateTime(calendar);
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get re-initialization date property", e);
        }
    }
    
    /**
     * Get the survey pages.
     * @return the survey pages.
     * @throws AmetysRepositoryException if an error occurs when retrieving the pages of the survey
     */
    public List<SurveyPage> getPages() throws AmetysRepositoryException
    {
        return getChildren().stream()
                .filter(child -> child instanceof SurveyPage)
                .map(child -> (SurveyPage) child)
                .collect(Collectors.toList());
    }
    
    /**
     * Get a question by its name.
     * @param name the question name.
     * @return the question.
     * @throws AmetysRepositoryException if an error occurs when retrieving a question of a survey
     */
    public SurveyQuestion getQuestion(String name) throws AmetysRepositoryException
    {
        for (SurveyPage page : getPages())
        {
            if (page.hasChild(name))
            {
                return page.getQuestion(name);
            }
        }
        return null;
    }
    
    /**
     * Get the survey questions.
     * @return the survey questions.
     * @throws AmetysRepositoryException if an error occurs when retrieving all the questions of a survey
     */
    public AmetysObjectIterable<SurveyQuestion> getQuestions() throws AmetysRepositoryException
    {
        List<AmetysObjectIterable<SurveyQuestion>> questions = new ArrayList<>();
        
        for (SurveyPage page : getPages())
        {
            questions.add(page.getQuestions());
        }
        
        return new ChainedAmetysObjectIterable<>(questions);
    }
    
    @Override
    public Site getSite() throws AmetysRepositoryException
    {
        return getParent().getParent().getParent().getParent().getParent();
    }
    
    @Override
    public String getSiteName() throws AmetysRepositoryException
    {
        return getSite().getName();
    }
    
    /**
     * Get the survey language.
     * @return the survey language.
     */
    public String getLanguage()
    {
        return getParent().getName();
    }
    
    /**
     * Returns a unique question name in the survey
     * @param originalName The original name
     * @return a unique question name
     */
    public String findUniqueQuestionName (String originalName)
    {
        String name = originalName;
        int index = 2;
        while (_hasQuestionName(name))
        {
            name = originalName + "-" + (index++);
        }
        return name;
    }
    
    private boolean _hasQuestionName (String name)
    {
        for (SurveyPage page : getPages())
        {
            if (page.hasQuestion(name))
            {
                return true;
            }
        }
        return false;
    }
    
    @Override
    public Survey copyTo(ModifiableTraversableAmetysObject parent, String name) throws AmetysRepositoryException
    {
        Survey survey = parent.createChild(name, "ametys:survey");
        
        survey.setTitle(getTitle());
        survey.setLabel(getLabel());
        
        String description = getDescription();
        if (description != null)
        {
            survey.setDescription(description);
        }
        
        String endingMessage = getEndingMessage();
        if (endingMessage != null)
        {
            survey.setEndingMessage(endingMessage);
        }
        
        copyPictureTo(survey);
        
        survey.setValidated(false);
        
        for (SurveyPage surveyPage : getPages())
        {
            surveyPage.copyTo(survey, surveyPage.getName());
        }
        
        // TODO Copy ACL ?
        
        return survey;
    }
    
    @Override
    public Survey copyTo(ModifiableTraversableAmetysObject parent, String name, List<String> restrictTo) throws AmetysRepositoryException
    {
        return copyTo(parent, name);
    }
}
