/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.translationflagging;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.LocaleUtils;

import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

import jakarta.mail.MessagingException;

/**
 * When a content is saved, this workflow function looks if the pages it belongs to are translated in other languages.
 * If this is the case, an alert e-mail is sent to all the persons who are responsible for modifying the translated pages,
 * to inform them that a new version is available.
 */
public class TranslationAlertFunction extends AbstractContentWorkflowComponent implements EnhancedFunction, Initializable, PluginAware
{
    
    /** The e-mail subject i18n key. */
    public static final String I18N_KEY_SUBJECT = "PLUGINS_TRANSLATIONFLAGGING_ALERT_EMAIL_SUBJECT";
    
    /** The e-mail body's title i18n key. */
    public static final String I18N_KEY_BODY_TITLE = "PLUGINS_TRANSLATIONFLAGGING_ALERT_EMAIL_BODY_TITLE";
    
    /** The e-mail body i18n key. */
    public static final String I18N_KEY_BODY = "PLUGINS_TRANSLATIONFLAGGING_ALERT_EMAIL_BODY";
    
    /** The users manager. */
    protected UserManager _userManager;
    
    /** The rights manager. */
    protected RightManager _rightManager;
    
    /** The i18n utils. */
    protected I18nUtils _i18nUtils;
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The plugin name. */
    protected String _pluginName;
    
    /** The server base URL. */
    protected String _baseUrl;
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public void initialize() throws Exception
    {
        _baseUrl = StringUtils.removeEndIgnoreCase(Config.getInstance().getValue("cms.url"), "index.html");
        if (!_baseUrl.endsWith("/"))
        {
            _baseUrl = _baseUrl + "/";
        }
    }
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        _logger.info("Performing translation alerts workflow function.");
        
        // Retrieve current content.
        WorkflowAwareContent content = getContent(transientVars);
        
        if (content instanceof WebContent && !_contentHelper.isMultilingual(content))
        {
            WebContent webContent = (WebContent) content;
            Site site = webContent.getSite();
            
            // The content has to be a web content to be referenced by pages.
            boolean enabled = site.getValue("translationflagging-enable-alerts", false, false);
            if (enabled)
            {
                sendAlerts((WebContent) content);
            }
        }
    }
    
    /**
     * Send the alerts to tell users the translated content was modified.
     * @param content the modified content.
     */
    protected void sendAlerts(WebContent content)
    {
        // Process all the pages which reference the content.
        for (Page page : content.getReferencingPages())
        {
            Site site = content.getSite();
            // Get the master language for this site.
            String masterLanguage = site.getValue("master-language");
            
            // Process the page only if it's in the master language, or there is no master language.
            if (StringUtils.isEmpty(masterLanguage) || page.getSitemapName().equals(masterLanguage))
            {
                // Get the translated versions of the page.
                Collection<Page> translatedPages = getTranslations(page).values();
                
                for (Page translatedPage : translatedPages)
                {
                    // Get the users to sent the alert to.
                    HashSet<UserIdentity> users = getUsersToNotify(translatedPage);
                    
                    // Build and send the alert.
                    sendAlert(page, content, translatedPage, users);
                }
            }
        }
    }
    
    /**
     * Build and send an alert e-mail to inform of a translation to a list of users.
     * @param page the modified page.
     * @param content the content which was modified.
     * @param translatedPage the translated page.
     * @param users the users to send the e-mail to.
     */
    protected void sendAlert(Page page, WebContent content, Page translatedPage, Set<UserIdentity> users)
    {
        List<String> params = new ArrayList<>();
        
        Site site = page.getSite();
        String mailFrom = site.getValue("site-mail-from");
        
        // Get a human-readable version of the languages.
        String pageLang = _i18nUtils.translate(new I18nizableText("plugin.web", "I18NKEY_LANGUAGE_" + page.getSitemapName().toUpperCase()));
        String translatedLang = _i18nUtils.translate(new I18nizableText("plugin.web", "I18NKEY_LANGUAGE_" + translatedPage.getSitemapName().toUpperCase()));
        
        // Build a list of the parameters.
        params.add(page.getSite().getTitle());
        params.add(content.getTitle(LocaleUtils.toLocale(page.getSitemapName())));
        params.add(page.getTitle());
        params.add(pageLang.toLowerCase());
        params.add(translatedPage.getTitle());
        params.add(translatedLang.toLowerCase());
        params.add(getPageUrl(page));
        params.add(getPageUrl(translatedPage));
        
        String catalogue = "plugin." + _pluginName;
        
        // Get the e-mail subject and body.
        I18nizableText i18nSubject = new I18nizableText(catalogue, I18N_KEY_SUBJECT, params);
        I18nizableText i18nBody = new I18nizableText(catalogue, I18N_KEY_BODY, params);
        
        try
        {
            String subject = _i18nUtils.translate(i18nSubject);
            
            String htmlBody = StandardMailBodyHelper.newHTMLBody()
                .withTitle(new I18nizableText(catalogue, I18N_KEY_BODY_TITLE, params))
                .withMessage(i18nBody)
                .withLink(getPageUrl(translatedPage), new I18nizableText(catalogue, "PLUGINS_TRANSLATIONFLAGGING_ALERT_EMAIL_BODY_TRANSLATED_PAGE_LINK"))
                .build();
            
            // Send the e-mails.
            sendMails(subject, htmlBody, users, mailFrom);
        }
        catch (IOException e)
        {
            _logger.error("Unable to build HTML body for email alert on translation", e);
        }
        
        
    }
    
    /**
     * Send a translation alert e-mail to the specified users.
     * @param subject the e-mail subject.
     * @param htmlBody the e-mail body.
     * @param users the users to send the e-mail to.
     * @param from the e-mail will be sent with this "from" header.
     */
    protected void sendMails(String subject, String htmlBody, Set<UserIdentity> users, String from)
    {
        List<String> recipients = users.stream()
                                       .map(_userManager::getUser)
                                       .filter(Objects::nonNull)
                                       .map(User::getEmail)
                                       .filter(StringUtils::isNotEmpty)
                                       .collect(Collectors.toList());
        
        try
        {
            SendMailHelper.newMail()
                          .withSubject(subject)
                          .withHTMLBody(htmlBody)
                          .withSender(from)
                          .withRecipients(recipients)
                          .sendMail();
        }
        catch (MessagingException | IOException e)
        {
            if (_logger.isWarnEnabled())
            {
                _logger.warn("Could not send a translation alert e-mail to " + recipients, e);
            }
        }
    }
    
    /**
     * Get the users to notify about the page translation.
     * @param translatedPage the translated version of the page.
     * @return the logins of the users to notify.
     */
    protected HashSet<UserIdentity> getUsersToNotify(Page translatedPage)
    {
        HashSet<UserIdentity> users = new HashSet<>();
        
        // Get the users which have the right to modify the page AND to receive the notification.
        Set<UserIdentity> editors = _rightManager.getAllowedUsers("Workflow_Rights_Edition_Online", translatedPage).resolveAllowedUsers(Config.getInstance().getValue("runtime.mail.massive.sending"));
        Set<UserIdentity> usersToNotify = _rightManager.getAllowedUsers("TranslationFlagging_Rights_Notification", translatedPage).resolveAllowedUsers(Config.getInstance().getValue("runtime.mail.massive.sending"));
        
        users.addAll(editors);
        users.retainAll(usersToNotify);
        
        return users;
    }
    
    /**
     * Get the translations of a given page.
     * @param page the page.
     * @return the translated pages as a Map of pages, indexed by sitemap name (language).
     */
    protected Map<String, Page> getTranslations(Page page)
    {
        Map<String, Page> translations = new HashMap<>();
        
        ModelLessDataHolder translationsComposite = page.getComposite(TranslationFlaggingClientSideElement.TRANSLATIONS_META);
        
        if (translationsComposite != null)
        {
            for (String lang : translationsComposite.getDataNames())
            {
                String translatedPageId = translationsComposite.getValue(lang);
                Page translatedPage = _resolver.resolveById(translatedPageId);
                
                translations.put(lang, translatedPage);
            }
        }
        else
        {
            // Ignore : the translations composite data doesn't exist, just return an empty map.
        }
        
        return translations;
    }
    
    /**
     * Get the URL of the back-office, opening on the page tool.
     * @param page the page to open on.
     * @return the page URL.
     */
    protected String getPageUrl(Page page)
    {
        StringBuilder url = new StringBuilder(_baseUrl);
        url.append(page.getSite().getName()).append("/index.html?uitool=uitool-page,id:%27").append(page.getId()).append("%27");
        return url.toString();
    }

    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.translationflagging", "PLUGINS_TRANSLATIONFLAGGING_ALERT_FUNCTION_LABEL");
    }
    
}
