/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.accesscontroller;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.collections.MapUtils;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.cms.repository.MixinTypeExpression;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.AbstractRightBasedAccessController;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.plugins.ugc.UGCConstants;
import org.ametys.plugins.ugc.page.UGCPage;
import org.ametys.plugins.ugc.page.UGCPageHandler;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.rights.PageAccessController;

/**
 * {@link AccessController} so creator of a UGC content types receive edit/delete rights on it
 *
 */
public class UGCCreatorPageAccessController extends AbstractRightBasedAccessController implements Serviceable
{
    private static final List<String> __CREATOR_RIGHTS = List.of("Front_Edition_Access_Right");
    
    /** ContentTypes Helper */
    protected ContentTypesHelper _cTypeHelper;
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The site manager */
    protected SiteManager _siteManager;
    /** the UGC page handler */
    protected UGCPageHandler _ugcPageHandler;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _cTypeHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _ugcPageHandler = (UGCPageHandler) smanager.lookup(UGCPageHandler.ROLE);
    }
    
    public boolean supports(Object object)
    {
        return object instanceof UGCPage;
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (((UGCPage) object).getContent().getCreator().equals(user))
        {
            return __CREATOR_RIGHTS.contains(rightId) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
        }
        
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    /**
     * If creator, access to a list of rights
     */
    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Map<String, AccessResult> permissionByRight = new HashMap<>();
        
        if (((UGCPage) object).getContent().getCreator().equals(user))
        {
            for (String rightId : __CREATOR_RIGHTS)
            {
                permissionByRight.put(rightId, AccessResult.USER_ALLOWED);
            }
        }
        
        return permissionByRight;
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    /**
     * If right requested is in the list, the creator is added the list of USER_ALLOWED
     */
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        Map<UserIdentity, AccessResult> permissionByUser = new HashMap<>();
        
        if (__CREATOR_RIGHTS.contains(rightId))
        {
            permissionByUser.put(((UGCPage) object).getContent().getCreator(), AccessResult.USER_ALLOWED);
        }
        return permissionByUser;
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    @Override
    protected AccessExplanation _getAccessExplanation(AccessResult result, Object object, UserIdentity user, Set<GroupIdentity> groups, String rightId)
    {
        switch (result)
        {
            case USER_ALLOWED:
            case UNKNOWN:
                return new AccessExplanation(
                        getId(),
                        result,
                        new I18nizableText("plugin.ugc", "PLUGINS_UGC_CREATOR_ACCESS_CONTROLLER_" + result.name() + "_EXPLANATION",
                                Map.of("title", new I18nizableText(((UGCPage) object).getTitle()))
                                )
                        );
            default:
                return AccessController.getDefaultAccessExplanation(getId(), result);
        }
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Page page)
        {
            return new I18nizableText(PageAccessController.getPageObjectLabel(page));
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }

    public I18nizableText getObjectCategory(Object object)
    {
        return PageAccessController.PAGE_CONTEXT_CATEGORY;
    }

    @Override
    protected Iterable< ? extends Object> getHandledObjects(UserIdentity identity, Set<GroupIdentity> groups, Set<Object> workspacesContexts)
    {
        
        String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        Site site = _siteManager.getSite(siteName);
        
        if (site != null)
        {
            List<Page> pages = new ArrayList<>();
            
            Expression typeExpr = new MixinTypeExpression(Operator.EQ, UGCConstants.UGC_MIXIN_TYPE);
            Expression userExpression = new UserExpression(DefaultContent.METADATA_CREATOR, Operator.EQ, identity);
            Expression siteExpression = new StringExpression(SiteAwareAmetysObject.METADATA_SITE, Operator.EQ, siteName);
            
            String query = ContentQueryHelper.getContentXPathQuery(new AndExpression(siteExpression, typeExpr, userExpression));
            try (AmetysObjectIterable<Content> contents = _resolver.query(query))
            {
                for (Content content : contents)
                {
                    try (AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps())
                    {
                        for (Sitemap sitemap: sitemaps)
                        {
                            for (String type : content.getTypes())
                            {
                                _ugcPageHandler.getUgcPage(content, siteName, sitemap.getSitemapName(), type)
                                    .ifPresent(page -> pages.add(page));
                            }
                        }
                    }
                }
            }
            
        }
        return List.of();
    }
    
    @Override
    protected Collection<String> getHandledRights()
    {
        return __CREATOR_RIGHTS;
    }
}
