/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.clientsideelement;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.ugc.observation.ObservationConstants;
import org.ametys.plugins.ugc.page.UGCPageHandler;
import org.ametys.plugins.ugc.page.VirtualUGCPageFactory;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.clientsideelement.AbstractPageClientSideElement;
import org.ametys.web.repository.page.LockablePage;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;

/**
 * Client side element for a controller which set/remove the root page of a ugc.
 */
public class SetUGCRootClientSideElement extends AbstractPageClientSideElement
{
    /** Observer manager. */
    protected ObservationManager _observationManager;
    /** The extension point for content types */
    protected ContentTypeExtensionPoint _contentTypeEP;
    /** The UGC page handler */
    protected UGCPageHandler _ugcPageHandler;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _observationManager = (ObservationManager) smanager.lookup(ObservationManager.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _ugcPageHandler = (UGCPageHandler) smanager.lookup(UGCPageHandler.ROLE);
        
    }
    
    /**
     * Gets the status of the given page
     * @param pageId The page id
     * @return the status of the given page
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStatus(String pageId)
    {
        Map<String, Object> result = new HashMap<>();
        
        Map<String, Object> parameters = this._script.getParameters();
        
        Page page = _resolver.resolveById(pageId);
        
        if (!hasRight(getRights(Map.of())))
        {
            throw new IllegalStateException("User " + _currentUserProvider.getUser() + " try to access privileges feature without sufficient rights");
        }
        
        if (page instanceof JCRAmetysObject)
        {
            if (_isUGCRootPage((JCRAmetysObject) page))
            {
                List<String> i18nParameters = new ArrayList<>();
                i18nParameters.add(page.getTitle());
    
                I18nizableText ed = (I18nizableText) parameters.get("ugc-page-description");
                I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
                result.put("ugc-page-title", msg);
                
                ed = (I18nizableText) parameters.get("remove-ugc-page-description");
                msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
                result.put("remove-ugc-page-title", msg);
                
                String contentTypeId = page.getValue(UGCPageHandler.CONTENT_TYPE_DATA_NAME, StringUtils.EMPTY);
                if (StringUtils.isNotEmpty(contentTypeId))
                {
                    I18nizableText contentTypeText = _contentTypeEP.hasExtension(contentTypeId) ? _contentTypeEP.getExtension(contentTypeId).getLabel() : new I18nizableText(contentTypeId);
                    
                    Map<String, I18nizableTextParameter> contentTypeI18nParameters = new HashMap<>();
                    contentTypeI18nParameters.put("0", contentTypeText);
                    
                    ed = (I18nizableText) parameters.get("contenttype-ugc-page-description");
                    msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), contentTypeI18nParameters);
                    result.put("contenttype-ugc-page-description", msg);
                }
                
                result.put("ugc-page-id", new I18nizableText(page.getId()));
            }
            else if (!_ugcPageHandler.getUGCRootPages(page.getSiteName(), page.getSitemapName()).isEmpty())
            {
                I18nizableText ed = (I18nizableText) parameters.get("ugc-page-already-exist");
                I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey());
                result.put("ugc-page-already-exist", msg);
            }
            else
            {
                List<String> i18nParameters = new ArrayList<>();
                i18nParameters.add(page.getTitle());
    
                I18nizableText ed = (I18nizableText) parameters.get("add-ugc-page-description");
                I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
                
                result.put("add-ugc-page-id", new I18nizableText(page.getId()));
                result.put("add-ugc-page-title", msg);
            }
        }
        else
        {
            List<String> noJcrI18nParameters = new ArrayList<>();
            noJcrI18nParameters.add(page.getTitle());

            I18nizableText ed = (I18nizableText) parameters.get("no-jcr-page-description");
            I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), noJcrI18nParameters);
            
            result.put("no-jcr-page-id", new I18nizableText(page.getId()));
            result.put("no-jcr-page-title", msg);
        }
        
        return result;
    }
    
    /**
     * Sets the given page as the root of a ugc
     * @param pageId The id of the page
     * @param contentTypeId The id of the content type
     * @param attributePath The classification attribute path
     * @param classificationPageVisible the visibility of transitional pages
     * @return A result map
     * @throws RepositoryException if a repository error occurred
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> setUGCRoot(String pageId, String contentTypeId, String attributePath, boolean classificationPageVisible) throws RepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        
        Page page = _resolver.resolveById(pageId);
        
        if (!hasRight(page))
        {
            throw new IllegalStateException("User " + _currentUserProvider.getUser() + " try to access privileges feature without sufficient rights");
        }
        
        if (page instanceof LockablePage lockablePage && lockablePage.isLocked())
        {
            throw new IllegalStateException("Cannot set the locked page '/" + page.getSitemapName() + "/" + page.getPathInSitemap() + "' as UGC root");
        }
        
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
        if (contentType == null)
        {
            result.put("error", "wrong-content-type");
            return result;
        }
        else if (StringUtils.isNotBlank(attributePath))
        {
            if (!contentType.hasModelItem(attributePath))
            {
                result.put("error", "wrong-metadata");
                return result;
            }
        }
        
        _ugcPageHandler.setUGCRoot(page, contentTypeId, attributePath, classificationPageVisible);
        
        return result;
    }

    
    
    /**
     * Remove the ugc root status to the given page
     * @param pageId The id of the page
     * @return A result map
     * @throws RepositoryException if a repository error occured
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> removeUGCRoot(String pageId) throws RepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        
        Page page = _resolver.resolveById(pageId);
        
        if (page instanceof JCRAmetysObject)
        {
            if (!hasRight(page))
            {
                throw new IllegalStateException("User '" + _currentUserProvider.getUser() + "' tried to access a privileged feature without convenient right");
            }
            
            if (page instanceof LockablePage lockablePage && lockablePage.isLocked())
            {
                throw new IllegalStateException("Cannot unset the locked page '/" + page.getSitemapName() + "/" + page.getPathInSitemap() + "' as UGC root");
            }

            if (!_isUGCRootPage((JCRAmetysObject) page))
            {
                result.put("error", "no-root");
                return result;
            }
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
            
            // Unindex pages for all workspaces before the properties were removed
            _observationManager.notify(new Event(ObservationConstants.EVENT_UGC_ROOT_DELETING, _currentUserProvider.getUser(), eventParams));
            
            _removeUGCRootProperty(page);
            
            _observationManager.notify(new Event(org.ametys.web.ObservationConstants.EVENT_PAGE_UPDATED, _currentUserProvider.getUser(), eventParams));
            
            // After live synchronization
            _observationManager.notify(new Event(ObservationConstants.EVENT_UGC_ROOT_DELETED, _currentUserProvider.getUser(), eventParams));
        }
        else
        {
            result.put("error", "no-root");
        }
        return result;
    }
    
    /**
     * Gets information about ugc root status on the given.
     * @param pageId The id of the page
     * @return information about ugc root status on the given.
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getRootPageInfo(String pageId)
    {
        Map<String, Object> result = new HashMap<>();
        
        Page page = _resolver.resolveById(pageId);
        
        if (!hasRight(page))
        {
            throw new IllegalStateException("User " + _currentUserProvider.getUser() + " try to access privileges feature without sufficient rights");
        }
        
        Set<Page> currentUGCPages = _ugcPageHandler.getUGCRootPages(page.getSiteName(), page.getSitemapName());
        
        if (currentUGCPages.contains(page))
        {
            result.put("isRoot", true);
            result.put("contentType", page.getValue(UGCPageHandler.CONTENT_TYPE_DATA_NAME));
            result.put("metadata", page.getValue(UGCPageHandler.CLASSIFICATION_ATTRIBUTE_DATA_NAME));
            result.put("is-visible", page.getValue(UGCPageHandler.CLASSIFICATION_PAGE_VISIBLE_DATA_NAME, false));
        }
        else
        {
            result.put("isRoot", false);
        }
        
        return result;
    }
    
    private void _removeUGCRootProperty(Page page) throws RepositoryException
    {
        if (page instanceof JCRAmetysObject)
        {
            JCRAmetysObject jcrPage = (JCRAmetysObject) page;
            Node node = jcrPage.getNode();
            
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                List<Value> values = new ArrayList<>(Arrays.asList(node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues()));
                int index = values.stream()
                        .map(LambdaUtils.wrap(Value::getString))
                        .collect(Collectors.toList())
                        .indexOf(VirtualUGCPageFactory.class.getName());
                if (index != -1)
                {
                    values.remove(index);
                    node.setProperty(AmetysObjectResolver.VIRTUAL_PROPERTY, values.toArray(new Value[values.size()]));
                }

                // Remove the ugc root property
                if (page instanceof ModifiablePage)
                {
                    ModifiablePage modifiablePage = (ModifiablePage) page;
                    modifiablePage.removeValue(UGCPageHandler.CONTENT_TYPE_DATA_NAME);
                    modifiablePage.removeValue(UGCPageHandler.CLASSIFICATION_ATTRIBUTE_DATA_NAME);
                    modifiablePage.removeValue(UGCPageHandler.CLASSIFICATION_PAGE_VISIBLE_DATA_NAME);
                }
                
                jcrPage.saveChanges();
            }
        }
    }
    
    private boolean _isUGCRootPage (JCRAmetysObject jcrPage)
    {
        try
        {
            Node node = jcrPage.getNode();
            
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                List<Value> values = Arrays.asList(node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues());
                
                return values.stream()
                        .map(LambdaUtils.wrap(Value::getString))
                        .anyMatch(v -> VirtualUGCPageFactory.class.getName().equals(v));
            }
            else
            {
                return false;
            }
        }
        catch (RepositoryException e)
        {
            return false;
        }
    }
}
