/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.ugc.observation;

import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.environment.Context;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.VirtualFactoryExpression;
import org.ametys.plugins.ugc.page.UGCPage;
import org.ametys.plugins.ugc.page.UGCPageHandler;
import org.ametys.plugins.ugc.page.VirtualUGCPageFactory;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.cache.pageelement.PageElementCache;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;
import org.ametys.web.repository.page.Zone;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Abstract {@link Observer} for observing validation of UGC content.
 */
public abstract class AbstractContentObserver extends AbstractLogEnabled implements Observer, Serviceable, Contextualizable
{
    /** The context. */
    protected org.apache.avalon.framework.context.Context _context;
    /** Cocoon context. */
    protected Context _cocoonContext;
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** The content type helper */
    protected ContentTypesHelper _contentTypeHelper;
    /** The page element cache */
    protected PageElementCache _zoneItemCache;
    /** The UGC page handler */
    protected UGCPageHandler _ugcPageHandler;

    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (Context) context.get(org.apache.cocoon.Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _zoneItemCache = (PageElementCache) manager.lookup(PageElementCache.ROLE + "/zoneItem");
        _ugcPageHandler = (UGCPageHandler) manager.lookup(UGCPageHandler.ROLE);
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars)
    {
        try
        {
            Content content = _getTarget(event);
            AmetysObjectIterable<Page> rootPages = _getUGCRootPages();
            if (!rootPages.iterator().hasNext())
            {
                getLogger().debug("There's no UGC root page, nothing to invalidate");
                return;
            }
            
            for (Page rootPage : rootPages)
            {
                if (_isUGCContent(content, rootPage))
                {
                    _internalObserve(event, rootPage, content);
                }
            }
        }
        catch (Exception e)
        {
            getLogger().error("Unable to observe event: " + event, e);
        }
    }
    
    /**
     * Do the actual work.
     * @param event the observation event.
     * @param rootPage the page holding the UGC content pages
     * @param content the UGC content.
     */
    protected abstract void _internalObserve(Event event, Page rootPage, Content content);
    
    /**
     * Get the UGC root pages
     * @return the UGC root pages
     */
    protected AmetysObjectIterable<Page> _getUGCRootPages ()
    {
        Expression expression = new VirtualFactoryExpression(VirtualUGCPageFactory.class.getName());
        String query = PageQueryHelper.getPageXPathQuery(null, null, null, expression, null);
        
        return _resolver.query(query);
    }
    
    /**
     * Retrieve the target of the observer
     * @param event The event
     * @return The target
     * @throws Exception if failed to get content
     */
    protected Content _getTarget(Event event) throws Exception
    {
        return (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
    }
    
    /**
     * Return true if the content is a UGC content
     * @param content the content
     * @param rootPage the root page
     * @return true if the content is a UGC content
     */
    protected boolean _isUGCContent(Content content, Page rootPage)
    {
        String contentTypeId = _ugcPageHandler.getContentTypeId(rootPage);
        return content != null && _contentTypeHelper.isInstanceOf(content, contentTypeId);
    }
    
    /**
     * Remove zone item cache
     * @param rootPage the root page
     * @param content the content 
     * @param workspace the workspace
     */
    protected void _removeZoneItemCache(Page rootPage, Content content, String workspace)
    {
        if (_isUGCContent(content, rootPage))
        {
            Optional<UGCPage> ugcPage = _ugcPageHandler.getUgcPage(rootPage, content);
            if (ugcPage.isPresent())
            {
                AmetysObjectIterable< ? extends Zone> zones = ugcPage.get().getZones();
                for (Zone zone : zones)
                {
                    for (ZoneItem zoneItem : zone.getZoneItems())
                    {
                        _zoneItemCache.removeItem(workspace, rootPage.getSiteName(), "CONTENT", zoneItem.getId());
                    }
                }
            }
        }
    }
}
