/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.page;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Collectors;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;
import org.ametys.web.repository.page.Page;

/**
 * The type of classification for {@link UGCPage UGCPages}
 */
public interface ClassificationType
{
    /**
     * Gets the transitional page information for the given UG Content
     * @param ugcContent The UG Content
     * @return the transitional page information
     */
    TransitionalPageInformation getTransitionalPage(Content ugcContent);
    
    /**
     * Gets all transitional page information
     * @return all the transitional page information
     */
    Collection<TransitionalPageInformation> allTransitionalPages();
    
    /**
     * UGC pages are classified by a linked Content on their UG Content (for instance entries of a reference table)
     */
    static class TypeContent implements ClassificationType
    {
        private UGCPageHandler _ugcPageHandler;
        private Page _rootPage;
        private String _attributeContentType;
        
        TypeContent(UGCPageHandler ugcPageHandler, Page rootPage, String attributeContentType)
        {
            _ugcPageHandler = ugcPageHandler;
            _rootPage = rootPage;
            _attributeContentType = attributeContentType;
        }
        
        @Override
        public TransitionalPageInformation getTransitionalPage(Content ugcContent)
        {
            String classificationAttributePath = _ugcPageHandler.getClassificationAttribute(_rootPage);
            ContentValue classificationContentValue = ugcContent.getDataHolder().getValue(classificationAttributePath);
            return new TransitionalPageInformation.TypeContent(classificationContentValue.getContent(), _rootPage);
        }
        
        @Override
        public Collection<TransitionalPageInformation> allTransitionalPages()
        {
            AmetysObjectIterable<Content> allContents = _getAllContents(_rootPage, _attributeContentType);
            return allContents.stream()
                    .map(content -> new TransitionalPageInformation.TypeContent(content, _rootPage))
                    .collect(Collectors.toList());
        }
        
        /**
         * Get all content with the content type id
         * @param rootPage the root page
         * @param contentTypeId the content type id
         * @return the list of contents
         */
        protected AmetysObjectIterable<Content> _getAllContents(Page rootPage, String contentTypeId)
        {
            String xPathQuery = null;
            ContentTypeExpression contentTypeExp = new ContentTypeExpression(Operator.EQ, contentTypeId);
            
            ContentType contentType = _ugcPageHandler._cTypeEP.getExtension(contentTypeId);
            if (contentType.isReferenceTable())
            {
                if (contentType.isMultilingual())
                {
                    xPathQuery = QueryHelper.getXPathQuery(null, "ametys:content", contentTypeExp, null);
                }
                else
                {
                    String lang = rootPage.getSitemapName();
                    Expression finalExpr = new AndExpression(contentTypeExp, new LanguageExpression(Operator.EQ, lang));

                    xPathQuery = QueryHelper.getXPathQuery(null, "ametys:content", finalExpr, null);
                }
            }
            else
            {
                SortCriteria sort = new SortCriteria();
                sort.addCriterion(Content.ATTRIBUTE_TITLE, true, true);

                StringExpression siteEpx = new StringExpression("site", Operator.EQ, rootPage.getSiteName());
                String lang = rootPage.getSitemapName();
                Expression finalExpr = new AndExpression(contentTypeExp, siteEpx, new LanguageExpression(Operator.EQ, lang));
                
                xPathQuery = QueryHelper.getXPathQuery(null, "ametys:content", finalExpr, sort);
            }
            
            return _ugcPageHandler._resolver.query(xPathQuery);
        }
    }
    
    /**
     * UGC pages are classified by entries of an enumerator
     */
    static class TypeEnum implements ClassificationType
    {
        private UGCPageHandler _ugcPageHandler;
        private Page _rootPage;
        private Enumerator<String> _enumerator;
        
        TypeEnum(UGCPageHandler ugcPageHandler, Page rootPage, Enumerator<String> enumerator)
        {
            _ugcPageHandler = ugcPageHandler;
            _rootPage = rootPage;
            _enumerator = enumerator;
        }
        
        @Override
        public TransitionalPageInformation getTransitionalPage(Content ugcContent)
        {
            String classificationAttributePath = _ugcPageHandler.getClassificationAttribute(_rootPage);
            String key = ugcContent.getDataHolder().getValue(classificationAttributePath);
            
            try
            {
                I18nizableText i18nEntry = _enumerator.getEntry(key);
                return new TransitionalPageInformation.TypeEnum(_ugcPageHandler, key, i18nEntry);
            }
            catch (Exception e)
            {
                _ugcPageHandler.getLogger().error("An error occurred. Can't get enumerator entry for metadata path " + classificationAttributePath + " and key " + key, e);
                throw new IllegalArgumentException("", e);
            }
        }
        
        @Override
        public Collection<TransitionalPageInformation> allTransitionalPages()
        {
            Map<String, I18nizableText> typedEntries;
            try
            {
                typedEntries = _enumerator.getEntries();
            }
            catch (Exception e)
            {
                String classificationAttributePath = _ugcPageHandler.getClassificationAttribute(_rootPage);
                _ugcPageHandler.getLogger().error("An error occurred. Can't get enumerator entries for metadata path " + classificationAttributePath, e);
                return Collections.EMPTY_SET;
            }
            
            return typedEntries
                    .entrySet()
                    .stream()
                    .map(e -> new TransitionalPageInformation.TypeEnum(_ugcPageHandler, e.getKey(), e.getValue()))
                    .collect(Collectors.toList());
        }
    }
    
    /**
     * There is no transitional page
     */
    static class None implements ClassificationType
    {
        None()
        {  }
        
        @Override
        public TransitionalPageInformation getTransitionalPage(Content ugcContent)
        {
            throw new IllegalStateException("There is no transitional page");
        }
        
        @Override
        public Collection<TransitionalPageInformation> allTransitionalPages()
        {
            return Collections.emptySet();
        }
    }
}
