/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.page;

import java.util.Map;

import org.apache.commons.lang3.LocaleUtils;

import org.ametys.cms.repository.Content;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.Page;

import com.google.common.collect.ImmutableMap;

/**
 * The information about a {@link UGCTransitionalPage}
 */
public interface TransitionalPageInformation
{
    /**
     * Gets the unique key of the transitional page
     * @return the unique key of the transitional page
     */
    String getKey();
    
    /**
     * Gets the value of the classification attribute for the transitional page
     * @return the value of the classification attribute for the transitional page
     */
    String getAttributeValue();
    
    /**
     * Gets the title of the transitional page
     * @return the title of the transitional page
     */
    String getTitle();
    
    /**
     * Gets the information as a map
     * @return the information as a map
     */
    default Map<String, String> getInfo()
    {
        return ImmutableMap.of(
                UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_METADATA_VALUE, getAttributeValue(),
                UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_TITLE, getTitle());
    }
    
    /**
     * UGC pages are classified by a linked Content on their UG Content (for instance entries of a reference table)
     */
    static class TypeContent implements TransitionalPageInformation
    {
        private Content _content;
        private Page _rootPage;
        
        TypeContent(Content content, Page rootPage)
        {
            _content = content;
            _rootPage = rootPage;
        }
        
        @Override
        public String getKey()
        {
            return _content.getName();
        }
        
        @Override
        public String getAttributeValue()
        {
            return _content.getId();
        }
        
        @Override
        public String getTitle()
        {
            return _content.getTitle(LocaleUtils.toLocale(_rootPage.getSitemapName()));
        }
    }
    
    /**
     * UGC pages are classified by entries of an enumerator
     */
    static class TypeEnum implements TransitionalPageInformation
    {
        private String _key;
        private I18nizableText _text;
        private UGCPageHandler _ugcPageHandler;
        
        TypeEnum(UGCPageHandler ugcPageHandler, String key, I18nizableText text)
        {
            _ugcPageHandler = ugcPageHandler;
            _key = key;
            _text = text;
        }
        
        @Override
        public String getKey()
        {
            return _key;
        }
        
        @Override
        public String getAttributeValue()
        {
            return _key;
        }
        
        @Override
        public String getTitle()
        {
            return _ugcPageHandler._i18nUtils.translate(_text);
        }
    }
}
