/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.page;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.LocaleUtils;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.MemoryRepositoryData;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.AbstractConfigurableVirtualPage;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;

/**
 * Page representing an UGC page.
 */
public class UGCPage extends AbstractConfigurableVirtualPage<UGCPageFactory>
{
    private String _title;
    private String _path;
    private Content _ugcContent;
    
    /**
     * Constructor.
     * @param root the root page.
     * @param syncContent the synchronized content
     * @param path the path
     * @param configuration The abstract virtual page's configuration
     * @param scheme The scheme
     * @param factory The UGC page factory
     */
    public UGCPage(Page root, VirtualPageConfiguration configuration, String scheme, UGCPageFactory factory, Content syncContent, String path)
    {
        super(root, configuration, scheme, factory);
        
        _path = path;
        _ugcContent = syncContent;
        _title = _ugcContent.getTitle(LocaleUtils.toLocale(root.getSitemapName()));
    }
    
    /**
     * Returns the associated UGC {@link Content}.
     * @return the associated UGC {@link Content}.
     */
    @SuppressWarnings("unchecked")
    @Override
    public Content getContent()
    {
        return _ugcContent;
    }
    
    public int getDepth() throws AmetysRepositoryException
    {
        return _root.getDepth() + (_path.equals("_root") ? 1 : 2);
    }

    @Override
    public Set<String> getReferers() throws AmetysRepositoryException
    {
        return null;
    }

    public String getTitle() throws AmetysRepositoryException
    {
        return _title;
    }
    
    public String getLongTitle() throws AmetysRepositoryException
    {
        return _title;
    }

    public AmetysObjectIterable<? extends Page> getChildrenPages() throws AmetysRepositoryException
    {
        List<Page> children = new ArrayList<>();
        return new CollectionIterable<>(children);
    }

    public String getPathInSitemap() throws AmetysRepositoryException
    {
        if (_path.equals("_root"))
        {
            return _root.getPathInSitemap() + "/" + getName();
        }
        else
        {
            return _root.getPathInSitemap() + "/" + _path + "/" + getName();
        }
    }

    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (path.isEmpty())
        {
            throw new AmetysRepositoryException("path must be non empty");
        }
        
        return null;
    }

    @SuppressWarnings("unchecked")
    public AmetysObjectIterable<? extends AmetysObject> getChildren() throws AmetysRepositoryException
    {
        return getChildrenPages();
    }

    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        return false;
    }
    
    public String getId() throws AmetysRepositoryException
    {
        return _factory.getUgcPageHandler().computePageId(_path, _root, _ugcContent);
    }

    public String getName() throws AmetysRepositoryException
    {
        return _ugcContent.getName();
    }
    
    @SuppressWarnings("unchecked")
    public Page getParent() throws AmetysRepositoryException
    {
        if (!_path.equals("_root"))
        {
            String name = _path;
            Map<String, Map<String, String>> transitionalPageName = _factory.getUgcPageHandler().getTransitionalPage(_root);
            Map<String, String> attributes = transitionalPageName.get(name);
            String title = attributes.get(UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_TITLE);
            String metadataValue = attributes.get(UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_METADATA_VALUE);
            
            return _factory.getTransitionalPageFactory().createUGCTransitionalPage(_root, title, metadataValue, name);
        }
        else
        {
            return _root;
        }
    }

    public String getParentPath() throws AmetysRepositoryException
    {
        if (!_path.equals("_root"))
        {
            return _root.getPath() + "/" + _path;
        }
        else
        {
            return _root.getPath();
        }
    }

    public ModelLessDataHolder getDataHolder()
    {
        RepositoryData repositoryData = new MemoryRepositoryData(getName());
        return new DefaultModelLessDataHolder(_factory.getPageDataTypeEP(), repositoryData);
    }

    public AmetysObjectIterable< ? extends Page> getChildrenPages(boolean includeInvisiblePage) throws AmetysRepositoryException
    {
        List<Page> children = new ArrayList<>();
        return new CollectionIterable<>(children);
    }

    public Page getChildPageAt(int index) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        throw new UnknownAmetysObjectException("There is no child for ugc page");
    }
}
