/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.page;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.value.StringValue;
import org.slf4j.Logger;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.ugc.observation.ObservationConstants;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.query.expression.VirtualFactoryExpression;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebConstants;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;
import org.ametys.web.repository.page.jcr.DefaultPage;

/**
 * Component providing methods to retrieve ugc virtual pages, such as the ugc root,
 * transitional page and ugc content page.
 */
public class UGCPageHandler extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** The attribute to get the name of transitional page */
    public static final String ATTRIBUTE_TRANSITIONAL_PAGE_METADATA_VALUE = "metadata_value";
    
    /** The attribute to get the title of transitional page */
    public static final String ATTRIBUTE_TRANSITIONAL_PAGE_TITLE = "title";
    
    /** The avalon role. */
    public static final String ROLE = UGCPageHandler.class.getName();
    
    /** The data name for the content type of the ugc */
    public static final String CONTENT_TYPE_DATA_NAME = "ugc-root-contenttype";
    
    /** The data name for the classification attribute of the ugc */
    public static final String CLASSIFICATION_ATTRIBUTE_DATA_NAME = "ugc-root-classification-metadata";
    
    /** The data name for the visibility of transitional page of the ugc */
    public static final String CLASSIFICATION_PAGE_VISIBLE_DATA_NAME = "ugc-root-classification-page-visible";
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    /** The content types helper */
    protected ContentTypesHelper _cTypeHelper;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;

    /** The avalon context */
    protected Context _context;

    /** Observer manager. */
    protected ObservationManager _observationManager;
    
    /** Current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _cTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    protected Logger getLogger()
    {
        return super.getLogger();
    }
    
    /**
     * Gets the path of the classification attribute
     * @param rootPage The ugc root page
     * @return the path of the classification attribute
     */
    public String getClassificationAttribute(Page rootPage)
    {
        return rootPage.getValue(CLASSIFICATION_ATTRIBUTE_DATA_NAME);
    }
    
    /**
     * Gets the content type id
     * @param rootPage The ugc root page
     * @return the content type id
     */
    public String getContentTypeId(Page rootPage)
    {
        return rootPage.getValue(CONTENT_TYPE_DATA_NAME);
    }
    
    /**
     * <code>true</code> if the classification pages are visible
     * @param rootPage The ugc root page
     * @return <code>true</code> if the classification pages are visible
     */
    public boolean isClassificationPagesVisible(Page rootPage)
    {
        return rootPage.getValue(CLASSIFICATION_PAGE_VISIBLE_DATA_NAME, false);
    }
    
    /**
     * True if the page a UGC root page for the given content type
     * @param page the page
     * @param contentTypeId The id of content type. Cannot be null.
     * @return true if the page is a UGC root page for the given content type
     */
    public boolean isUGCRootPage(DefaultPage page, String contentTypeId)
    {
        try
        {
            Node node = page.getNode();
            
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                List<Value> values = Arrays.asList(node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues());
                
                boolean isUGCRootPage = values.stream()
                        .map(LambdaUtils.wrap(Value::getString))
                        .anyMatch(v -> VirtualUGCPageFactory.class.getName().equals(v));
                
                if (isUGCRootPage)
                {
                    return contentTypeId.equals(page.getValue(CONTENT_TYPE_DATA_NAME));
                }
            }
        }
        catch (RepositoryException e)
        {
            getLogger().warn("Unable to determine if page '" + page.getId() + "' is a UGC root page", e);
        }
        
        return false;
    }
    
    /**
     * Gets the ugc root pages from the given content type id.
     * @param siteName the site name
     * @param sitemapName the sitemap name
     * @param contentTypeId The content type id
     * @return the ugc root page.
     * @throws AmetysRepositoryException  if an error occured.
     */
    public Page getUGCRootPage(String siteName, String sitemapName, String contentTypeId) throws AmetysRepositoryException
    {
        Expression expression = new VirtualFactoryExpression(VirtualUGCPageFactory.class.getName());
        Expression contentTypeExp = new StringExpression(CONTENT_TYPE_DATA_NAME, Operator.EQ, contentTypeId);
        
        AndExpression andExp = new AndExpression(expression, contentTypeExp);
        
        String query = PageQueryHelper.getPageXPathQuery(siteName, sitemapName, null, andExp, null);
        
        AmetysObjectIterable<Page> pages = _resolver.query(query);
        
        return pages.iterator().hasNext() ? pages.iterator().next() : null;
    }
    
    /**
     * Get the ugc root pages
     * @param siteName the current site.
     * @param sitemapName the sitemap name.
     * @return the ugc root pages
     * @throws AmetysRepositoryException if an error occured.
     */
    public Set<Page> getUGCRootPages(String siteName, String sitemapName) throws AmetysRepositoryException
    {
        Expression expression = new VirtualFactoryExpression(VirtualUGCPageFactory.class.getName());
        
        String query = PageQueryHelper.getPageXPathQuery(siteName, sitemapName, null, expression, null);
        
        AmetysObjectIterable<Page> pages = _resolver.query(query);
        
        return pages.stream().collect(Collectors.toSet());
    }
    
    /**
     * Set the ugc root page
     * @param page the page to set as root
     * @param contentTypeId the type of content root
     * @param attributePath path to classification attribute
     * @param classificationPageVisible true to show classification
     * @throws RepositoryException if a repository error occurred
     */
    public void setUGCRoot(Page page, String contentTypeId, String attributePath, boolean classificationPageVisible) throws RepositoryException
    {
        Page currentUGCPage = getUGCRootPage(page.getSiteName(), page.getSitemapName(), contentTypeId);
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
        
        if (currentUGCPage != null && currentUGCPage.getId().equals(page.getId()))
        {
            // Unindex pages for all workspaces before the properties changed 
            _observationManager.notify(new Event(ObservationConstants.EVENT_UGC_ROOT_UPDATING, _currentUserProvider.getUser(), eventParams));
            
            _updateUGCRootProperty(page, contentTypeId, attributePath, classificationPageVisible);
        }
        else
        {
            _addUGCRootProperty(page, contentTypeId, attributePath, classificationPageVisible);
        }
        
        // Live synchronization
        _observationManager.notify(new Event(org.ametys.web.ObservationConstants.EVENT_PAGE_UPDATED, _currentUserProvider.getUser(), eventParams));
        
        // Indexation
        _observationManager.notify(new Event(ObservationConstants.EVENT_UGC_ROOT_UPDATED, _currentUserProvider.getUser(), eventParams));
    }
    
    private void _addUGCRootProperty(Page page, String contentType, String metadata, boolean classificationPageVisible) throws RepositoryException
    {
        if (page instanceof JCRAmetysObject)
        {
            JCRAmetysObject jcrPage = (JCRAmetysObject) page;
            Node node = jcrPage.getNode();
            
            List<Value> values = new ArrayList<>();
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                values.addAll(Arrays.asList(node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues()));
            }
            
            StringValue virtualUGCPageFactoryClassName = new StringValue(VirtualUGCPageFactory.class.getName());
            if (!values.contains(virtualUGCPageFactoryClassName))
            {
                values.add(virtualUGCPageFactoryClassName);
            }
            
            node.setProperty(AmetysObjectResolver.VIRTUAL_PROPERTY, values.toArray(new Value[values.size()]));

            // Set the ugc root property
            if (page instanceof ModifiablePage)
            {
                ((ModifiablePage) page).setValue(UGCPageHandler.CONTENT_TYPE_DATA_NAME, contentType);
                ((ModifiablePage) page).setValue(UGCPageHandler.CLASSIFICATION_ATTRIBUTE_DATA_NAME, metadata);
                ((ModifiablePage) page).setValue(UGCPageHandler.CLASSIFICATION_PAGE_VISIBLE_DATA_NAME, classificationPageVisible);
            }
            
            jcrPage.saveChanges();
        }
    }
    
    private void _updateUGCRootProperty(Page page, String contentType, String metadata, boolean classificationPageVisible)
    {
        if (page instanceof ModifiablePage)
        {
            ModifiablePage modifiablePage = (ModifiablePage) page;
            
            // Set the ugc root property
            modifiablePage.setValue(UGCPageHandler.CONTENT_TYPE_DATA_NAME, contentType);
            modifiablePage.setValue(UGCPageHandler.CLASSIFICATION_ATTRIBUTE_DATA_NAME, metadata);
            modifiablePage.setValue(UGCPageHandler.CLASSIFICATION_PAGE_VISIBLE_DATA_NAME, classificationPageVisible);
            
            modifiablePage.saveChanges();
        }
    }
    
    /**
     * Get UGC contents from rootPage
     * @param rootPage the root page
     * @return the list of UGC contents
     */
    public AmetysObjectIterable<Content> getContentsForRootPage(Page rootPage)
    {
        String lang = rootPage.getSitemapName();
        String contentType = getContentTypeId(rootPage);
        
        ContentTypeExpression contentTypeExp = new ContentTypeExpression(Operator.EQ, contentType);
        
        StringExpression siteExpr = new StringExpression(SiteAwareAmetysObject.METADATA_SITE, Operator.EQ, rootPage.getSiteName());
        Expression noSiteExpr = new NotExpression(new MetadataExpression(SiteAwareAmetysObject.METADATA_SITE));
        Expression fullSiteExpr = new OrExpression(siteExpr, noSiteExpr);
        
        Expression finalExpr = new AndExpression(contentTypeExp, new LanguageExpression(Operator.EQ, lang), fullSiteExpr);
        
        SortCriteria sort = new SortCriteria();
        sort.addCriterion(Content.ATTRIBUTE_TITLE, true, true);
        
        String xPathQuery = QueryHelper.getXPathQuery(null, "ametys:content", finalExpr, sort);
        
        return _resolver.query(xPathQuery);
    }
    
    /**
     * Determines if content is part of UGC root page
     * @param rootPage The root page
     * @param content the content
     * @return true if content is part of UGC root page
     */
    public boolean hasContentForRootPage(Page rootPage, Content content)
    {
        String contentType = getContentTypeId(rootPage);
        String siteName = rootPage.getSiteName();
        String classificationMetadata = getClassificationAttribute(rootPage);
        
        return _cTypeHelper.isInstanceOf(content, contentType) // match content type
                && (!(content instanceof SiteAwareAmetysObject) || siteName.equals(((SiteAwareAmetysObject) content).getSiteName())) // match site
                && StringUtils.isBlank(classificationMetadata); // no classification attribute
    }
    
    /**
     * Get the map of transitional page (name : (id, title))
     * @param rootPage the root page
     * @return The map of transitional page
     */
    public Map<String, Map<String, String>> getTransitionalPage(Page rootPage)
    {
        return _getClassificationType(rootPage)
                .allTransitionalPages()
                .stream()
                .sorted(Comparator.comparing(TransitionalPageInformation::getTitle))
                .collect(LambdaUtils.Collectors.toLinkedHashMap(
                        TransitionalPageInformation::getKey, 
                        TransitionalPageInformation::getInfo));
    }
    
    private ClassificationType _getClassificationType(Page rootPage)
    {
        String classificationAttributePath = getClassificationAttribute(rootPage);
        if (StringUtils.isBlank(classificationAttributePath))
        {
            // No classification attribute defined, so no transitional page
            return new ClassificationType.None();
        }
        String contentTypeId = getContentTypeId(rootPage);
        ContentType contentType = _cTypeEP.getExtension(contentTypeId);
        
        if (contentType == null)
        {
            getLogger().warn("Can not classify UGC content of type '" + contentTypeId + "' on root page " + rootPage.getId());
        }
        else if (contentType.hasModelItem(classificationAttributePath))
        {
            ModelItem modelItem = contentType.getModelItem(classificationAttributePath);
            if (modelItem instanceof ContentAttributeDefinition)
            {
                String attributeContentType = ((ContentAttributeDefinition) modelItem).getContentTypeId();
                return new ClassificationType.TypeContent(this, rootPage, attributeContentType);
            }
            else if (modelItem instanceof ElementDefinition<?>)
            {
                @SuppressWarnings("unchecked")
                Enumerator<String> enumerator = ((ElementDefinition<String>) modelItem).getEnumerator();
                if (enumerator != null)
                {
                    return new ClassificationType.TypeEnum(this, rootPage, enumerator);
                }
            }
        }
        
        return new ClassificationType.None();
    }
    
    /**
     * Get contents under transitional page
     * @param rootPage the root page
     * @param metadataValue the metadata value (linked to the transitional page)
     * @return list of contents under transitional page
     */
    public AmetysObjectIterable<Content> getContentsForTransitionalPage(Page rootPage, String metadataValue)
    {
        String classificationMetadata = getClassificationAttribute(rootPage);
        
        String lang = rootPage.getSitemapName();
        String contentType = getContentTypeId(rootPage);
        
        ContentTypeExpression contentTypeExp = new ContentTypeExpression(Operator.EQ, contentType);
        StringExpression metadataExpression = new StringExpression(classificationMetadata, Operator.EQ, metadataValue);
        
        StringExpression siteExpr = new StringExpression(SiteAwareAmetysObject.METADATA_SITE, Operator.EQ, rootPage.getSiteName());
        Expression noSiteExpr = new NotExpression(new MetadataExpression(SiteAwareAmetysObject.METADATA_SITE));
        Expression fullSiteExpr = new OrExpression(siteExpr, noSiteExpr);
        
        Expression finalExpr = new AndExpression(contentTypeExp, metadataExpression, new LanguageExpression(Operator.EQ, lang), fullSiteExpr);
        
        SortCriteria sort = new SortCriteria();
        sort.addCriterion(Content.ATTRIBUTE_TITLE, true, true);
        
        String xPathQuery = QueryHelper.getXPathQuery(null, "ametys:content", finalExpr, sort);
        
        return _resolver.query(xPathQuery);
    }
    
    /**
     * Determines if given content is part of a transitional page
     * @param rootPage the root page
     * @param metadataValue the metadata value (linked to the transitional page). Cannot be null.
     * @param content the content
     * @return true if content is part of the transitional page
     */
    public boolean hasContentForTransitionalPage(Page rootPage, String metadataValue, Content content)
    {
        String contentType = getContentTypeId(rootPage);
        String siteName = rootPage.getSiteName();
        String classificationMetadata = getClassificationAttribute(rootPage);
        
        return _cTypeHelper.isInstanceOf(content, contentType) // match content type
                && (!(content instanceof SiteAwareAmetysObject) || siteName.equals(((SiteAwareAmetysObject) content).getSiteName())) // match site
                && StringUtils.isNotEmpty(classificationMetadata) && metadataValue.equals(((ContentValue) content.getValue(classificationMetadata)).getContentId()); // match classification attribute
                
    }
    
    /**
     * Computes a page id
     * @param path The path
     * @param root The root page
     * @param ugcContent The UGC content
     * @return The id
     */
    public String computePageId(String path, Page root, Content ugcContent)
    {
        // E.g: ugccontent://path?rootId=...&contentId=...
        return "ugccontent://" + path + "?rootId=" + root.getId() + "&contentId=" + ugcContent.getId();
    }
    
    /**
     * Gets the UGC page related to the given UG Content id
     * @param contentId the id of UG Content
     * @param siteName The site name. Can be nul to get site from content or current site.
     * @return the UGC page or null if not found
     */
    public UGCPage getUgcPage(String contentId, String siteName)
    {
        if (contentId == null)
        {
            return null;
        }
        
        Content content;
        try
        {
            content = _resolver.resolveById(contentId);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
        
        Request request = ContextHelper.getRequest(_context);
        String site = StringUtils.isNotBlank(siteName) ? siteName : WebHelper.getSiteName(request, content);
        
        String sitemap = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITEMAP_NAME);
        
        for (String type : content.getTypes())
        {
            Optional<UGCPage> ugcPage = getUgcPage(content, site, sitemap, type);
            if (ugcPage.isPresent())
            {
                return ugcPage.get();
            }
        }
        
        return null;
    }
    
    /**
     * Gets the UGC page related to the given UG Content for given site, sitemap and type
     * @param ugcContent the UG Content
     * @param site the site name
     * @param sitemap the sitemap name
     * @param contentType the content type id
     * @return the UGC page
     */
    public Optional<UGCPage> getUgcPage(Content ugcContent, String site, String sitemap, String contentType)
    {
        String language = Optional.of(ugcContent)
                .map(Content::getLanguage)
                .orElse(sitemap);
        Page ugcRootPage = getUGCRootPage(site, language, contentType);
        
        return Optional.ofNullable(ugcRootPage)
                .flatMap(root -> getUgcPage(root, ugcContent));
    }
    
    /**
     * Gets the UGC page related to the given UG Content for given UGC root
     * @param ugcRootPage the UGC root page
     * @param ugcContent the UG Content
     * @return the UGC page
     */
    public Optional<UGCPage> getUgcPage(Page ugcRootPage, Content ugcContent)
    {
        String path = _getPath(ugcRootPage, ugcContent);
        return Optional.ofNullable(path)
                .map(p -> computePageId(p, ugcRootPage, ugcContent))
                .map(this::_silentResolve);
    }
    
    private String _getPath(Page ugcRootPage, Content ugcContent)
    {
        try
        {
            ClassificationType transtionalPageType = _getClassificationType(ugcRootPage);
            if (transtionalPageType instanceof ClassificationType.None)
            {
                return "_root";
            }
            else
            {
                TransitionalPageInformation transitionalPageInfo = transtionalPageType.getTransitionalPage(ugcContent);
                return transitionalPageInfo.getKey();
            }
        }
        catch (Exception e)
        {
            getLogger().error("Cannot get path for root {} and content {}", ugcRootPage, ugcContent, e);
            return null;
        }
    }
    
    private UGCPage _silentResolve(String id)
    {
        try
        {
            return _resolver.resolveById(id);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
}
