/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.page;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.virtual.VirtualAmetysObjectFactory;
import org.ametys.web.repository.page.Page;

/**
 * {@link AmetysObjectFactory} for handling "virtual" ugc page
 */
public class VirtualUGCPageFactory extends AbstractUGCPageFactory implements VirtualAmetysObjectFactory<Page>
{
    @Override
    public Page getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // No page object linked to this factory
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        // No page object linked to this factory
        throw new UnsupportedOperationException();
    }

    @Override
    public String getScheme()
    {
        return "ugcroot";
    }

    @Override
    public AmetysObjectIterable<Page> getChildren(JCRAmetysObject parent)
    {
        if (!(parent instanceof Page))
        {
            throw new IllegalArgumentException("The holder of the ugc virtual pages should be a page.");
        }
        
        List<Page> children = new ArrayList<>();
        Page rootPage = (Page) parent;
        
        String classificationMetadata = _ugcPageHandler.getClassificationAttribute(rootPage);
        if (StringUtils.isNotBlank(classificationMetadata))
        {
            Map<String, Map<String, String>> transitionalPageName = _ugcPageHandler.getTransitionalPage(rootPage);
            for (String name : transitionalPageName.keySet())
            {
                Map<String, String> attributes = transitionalPageName.get(name);
                String metadataValue = attributes.get(UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_METADATA_VALUE);
                AmetysObjectIterable<Content> contentsForTransitionalPage = _ugcPageHandler.getContentsForTransitionalPage(rootPage, metadataValue);
                if (contentsForTransitionalPage.getSize() != 0)
                {
                    String title = attributes.get(UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_TITLE);
                    children.add(getTransitionalPageFactory().createUGCTransitionalPage(rootPage, title, metadataValue, name));
                }
            }
        }
        else
        {
            AmetysObjectIterable<Content> contents = _ugcPageHandler.getContentsForRootPage(rootPage);
            for (Content content : contents)
            {
                children.add(getUGCPageFactory().createUGCPage(rootPage, content, "_root"));
            }
        }
        
        return new CollectionIterable<>(children);
    }

    @Override
    public Page getChild(JCRAmetysObject parent, String childName)
    {
        if (!(parent instanceof Page))
        {
            throw new IllegalArgumentException("The holder of the ugc virtual pages should be a page.");
        }
        
        Page rootPage = (Page) parent;
        
        String classificationMetadata = _ugcPageHandler.getClassificationAttribute(rootPage);
        if (StringUtils.isNotBlank(classificationMetadata))
        {
            Map<String, Map<String, String>> transitionalPageName = _ugcPageHandler.getTransitionalPage(rootPage);
            if (transitionalPageName.containsKey(childName))
            {
                Map<String, String> attributes = transitionalPageName.get(childName);
                String metadataValue = attributes.get(UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_METADATA_VALUE);
                
                AmetysObjectIterable<Content> contentsForTransitionalPage = _ugcPageHandler.getContentsForTransitionalPage(rootPage, metadataValue);
                if (contentsForTransitionalPage.getSize() != 0)
                {
                    String title = attributes.get(UGCPageHandler.ATTRIBUTE_TRANSITIONAL_PAGE_TITLE);
                    return getTransitionalPageFactory().createUGCTransitionalPage(rootPage, title, metadataValue, childName);
                }
                else
                {
                    throw new UnknownAmetysObjectException("No transitional ugc page named " + childName);
                }
            }
            else
            {
                throw new UnknownAmetysObjectException("No transitional ugc page named " + childName);
            }
        }
        else
        {
            AmetysObjectIterable<Content> contents = _ugcPageHandler.getContentsForRootPage(rootPage);
            List<Content> contentFilters = contents.stream().filter(c -> c.getName().equals(childName)).collect(Collectors.toList());
            
            if (!contentFilters.isEmpty())
            {
                Content content = contentFilters.get(0);
                return getUGCPageFactory().createUGCPage(rootPage, content, "_root");
            }
            else
            {
                throw new UnknownAmetysObjectException("No ugc page named " + childName);
            }
        }
    }

    @Override
    public boolean hasChild(JCRAmetysObject parent, String childName)
    {
        if (!(parent instanceof Page))
        {
            throw new IllegalArgumentException("The holder of the ugc virtual pages should be a page.");
        }

        Page rootPage = (Page) parent;
        
        String classificationMetadata = _ugcPageHandler.getClassificationAttribute(rootPage);
        if (StringUtils.isNotBlank(classificationMetadata))
        {
            AmetysObjectIterable<Content> contentsForTransitionalPage = _ugcPageHandler.getContentsForTransitionalPage(rootPage, childName);
            if (contentsForTransitionalPage.getSize() != 0)
            {
                Map<String, Map<String, String>> transitionalPageName = _ugcPageHandler.getTransitionalPage(rootPage);
                return transitionalPageName.containsKey(childName);
            }
            
            return false;
        }
        else
        {
            AmetysObjectIterable<Content> contents = _ugcPageHandler.getContentsForRootPage(rootPage);
            List<Content> contentFilters = contents.stream().filter(c -> c.getName().equals(childName)).collect(Collectors.toList());
            
            return !contentFilters.isEmpty();
        }
    }
}
