/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.cachepolicy;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.web.cache.pageelement.AbstractSimplePageElementCachePolicy;
import org.ametys.web.cache.pageelement.PageElementCachePolicy;

/**
 * {@link PageElementCachePolicy} for the insert user service.
 */
public class InsertUserServiceCachePolicy extends AbstractSimplePageElementCachePolicy implements Serviceable
{

    /** The insert user service ID. */
    public static final String SERVICE_INSERT_USER_ID = "org.ametys.plugins.userdirectory.service.InsertUser";
    
    private ContentTypesHelper _contentTypesHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }
    
    @Override
    public Set<String> getPageElementTypes()
    {
        return Collections.singleton("SERVICE:" + SERVICE_INSERT_USER_ID);
    }
    
    @Override
    protected boolean _supports (Event event, String workspace)
    {
        boolean support = super._supports(event, workspace);
        
        if (support)
        {
            Object object = event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            if (object != null && object instanceof Content content)
            {
                return _contentTypesHelper.isInstanceOf(content, UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE);
            }
        }
        
        return support;
    }
    
    @Override
    protected List<String> _getRemovingCacheEventIds(String workspace)
    {
        if ("default".equals(workspace))
        {
            return Arrays.asList(ObservationConstants.EVENT_CONTENT_MODIFIED, ObservationConstants.EVENT_CONTENT_DELETING);
        }
        else if ("live".equals(workspace))
        {
            return Arrays.asList(ObservationConstants.EVENT_CONTENT_VALIDATED,
                                 ObservationConstants.EVENT_CONTENT_DELETING,
                                 ObservationConstants.EVENT_CONTENT_UNTAG_LIVE);
        }
        
        return Collections.emptyList();
    }
}
