/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.userdataprovider;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.FilenameUtils;

import org.ametys.cms.data.File;
import org.ametys.core.user.User.UserImage;
import org.ametys.core.user.UserImageAccessor;
import org.ametys.core.util.ImageHelper;
import org.ametys.plugins.core.ui.user.AbstractImageReader;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The user image accessor that retrieves the image of a ud user
 */
public class UserDirectoryImageAccessor extends AbstractLogEnabled implements UserImageAccessor
{
    /** The type of image returned by this type of accessor */
    public static final String USER_DIRECTORY_TYPE = "UserDirectoryUpload";
    private File _file;
    
    /**
     * Constructor of User directory image accessor with a file
     * @param file The file
     */
    public UserDirectoryImageAccessor(File file)
    {
        _file = file;
    }

    public UserImage getImage(int size, int maxSize)
    {
        try
        {
            InputStream is = _getCroppedImageInputStream(size);
            return new UserImage(is, _file.getName(), null, _file.getLastModificationDate().toInstant().toEpochMilli(), USER_DIRECTORY_TYPE); // no length because image is cropped
        }
        catch (IOException e)
        {
            getLogger().error(String.format("Unable to provide the uploaded cropped image for file '%s'.", _file.getName()), e);
        }
        
        return null;
    }
    
    private InputStream _getCroppedImageInputStream(int size) throws IOException
    {
        try (InputStream is = _file.getInputStream())
        {
            // Retrieve the input stream of the cropped image
            String filename =  _file.getName();
            String format = FilenameUtils.getExtension(filename);
            format = AbstractImageReader.ALLOWED_IMG_FORMATS.contains(format) ? format : "png";
            
            try (ByteArrayOutputStream buffer = new ByteArrayOutputStream())
            {
                ImageHelper.generateThumbnail(is, buffer, format, 0, 0, 0, 0, size, size);
                return new ByteArrayInputStream(buffer.toByteArray());
            }
        }
    }
}
