/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.synchronize;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Triple;
import org.slf4j.Logger;

import org.ametys.cms.content.references.OutgoingReferencesExtractor;
import org.ametys.cms.data.ContentSynchronizationResult;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDataProvider;
import org.ametys.plugins.contentio.synchronize.impl.SQLSynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.workflow.EditSynchronizedContentFunction;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.group.Repeater;
import org.ametys.plugins.repository.data.holder.group.RepeaterEntry;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.plugins.repository.data.holder.values.SynchronizationResult;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.userdirectory.DeleteOrgUnitComponent;
import org.ametys.plugins.userdirectory.OrganisationChartPageHandler;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

/**
 * Synchronizable collection for UD Orgunits
 */
public class SQLSynchronizableUDOrgunitCollection extends SQLSynchronizableContentsCollection
{
    /** The internal data name for orgUnit remote sql id */
    public static final String ORGUNIT_REMOTE_ID_INTERNAL_DATA = "orgunit-remote-id";
    
    /** The result map key for number of deleted contents */
    public static final String RESULT_NB_SYNCHRONIZED_ORGUNITS_RELATIONS = "nbSynchronizedOrgUnitsRelations";

    private static final String __PARAM_SQL_TABLE_USER = "tableNameUser";
    private static final String __PARAM_SQL_ORGUNIT_JOIN_COLUMN_NAME = "orgUnitJoinColumnName";
    private static final String __PARAM_LOGIN_USER_ATTRIBUTE_NAME = "loginUser";
    private static final String __PARAM_SQL_LOGIN_USER_COLUMN_NAME = "loginColumnName";
    private static final String __PARAM_SQL_ROLE_USER_COLUMN_NAME = "roleColumnName";
    private static final String __PARAM_SQL_ORGUNIT_REMOTE_ID_COLUMN_NAME = "orgunitRemoteIdColumnName";
    private static final String __PARAM_SQL_USER_SORT_COLUMN_NAME = "sortColumnName";
    private static final String __PARAM_SQL_USER_SORT_PREVAIL_NAME = "sortPrevail";
    
    /** The map which link orgunit with this parent */
    protected Map<String, String> _orgUnitParents;
    
    /** The map which link users (userId and role) to their orgunits */
    protected Map<String, Map<String, String>> _usersByOrgUnitId;
    
    /** The map which link orgunit with sql remote ids */
    protected Map<String, String> _orgUnitRemoteIds;
    
    
    /** Number of updated contents for parent-child org unit relation */
    protected Integer _nbSynchronizedOrgUnit;
    
    /** The sql user search DAO */
    protected SQLUserSearchDAO _sqlUserDAO;
    
    /** The organisation chart page handler */
    protected OrganisationChartPageHandler _orgChartPageHandler;
    
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    
    /** The OutgoingReferences extractor */
    protected OutgoingReferencesExtractor _outgoingReferencesExtractor;
    
    /** The delete orgUnit component */
    protected DeleteOrgUnitComponent _deleteOrgUnitComponent;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _sqlUserDAO = (SQLUserSearchDAO) smanager.lookup(SQLUserSearchDAO.ROLE);
        _orgChartPageHandler = (OrganisationChartPageHandler) smanager.lookup(OrganisationChartPageHandler.ROLE);
        _userProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _outgoingReferencesExtractor = (OutgoingReferencesExtractor) smanager.lookup(OutgoingReferencesExtractor.ROLE);
        _deleteOrgUnitComponent = (DeleteOrgUnitComponent) smanager.lookup(DeleteOrgUnitComponent.ROLE);
    }
    
    public boolean handleRightAssignmentContext()
    {
        return false;
    }
    
    /**
     * Get the name of user SQL table
     * @return The user SQL table name
     */
    public String getUserTableName()
    {
        return (String) getParameterValues().get(__PARAM_SQL_TABLE_USER);
    }
    
    /**
     * Get the name of the orgunit join column name of the user table
     * @return The name of the orgunit join column name
     */
    public String getOrgunitJoinColumnNameForUser()
    {
        return (String) getParameterValues().get(__PARAM_SQL_ORGUNIT_JOIN_COLUMN_NAME);
    }
    
    /**
     * Get the login user attribute name
     * @return The login user attribute name
     */
    public String getLoginUserAttributeName()
    {
        return (String) getParameterValues().get(__PARAM_LOGIN_USER_ATTRIBUTE_NAME);
    }
    
    /**
     * Get the login user column name
     * @return The login user column name
     */
    public String getLoginUserColumnName()
    {
        return (String) getParameterValues().get(__PARAM_SQL_LOGIN_USER_COLUMN_NAME);
    }
    
    /**
     * Get the role user column name
     * @return The role user column name
     */
    public String getRoleUserColumnName()
    {
        return (String) getParameterValues().get(__PARAM_SQL_ROLE_USER_COLUMN_NAME);
    }
    
    /**
     * Get the user sort column name
     * @return The user sort column name
     */
    public String getUserSortColumnName()
    {
        return (String) getParameterValues().get(__PARAM_SQL_USER_SORT_COLUMN_NAME);
    }
    
    /**
     * Get the orgunit remote id column name
     * @return The orgunit remote id column name
     */
    public String getOrgUnitRemoteIdColumnName()
    {
        return (String) getParameterValues().get(__PARAM_SQL_ORGUNIT_REMOTE_ID_COLUMN_NAME);
    }
    
    /**
     * True if the SQL sort for users prevail
     * @return true if the SQL sort for users prevail
     */
    public boolean isUserSortPrevail()
    {
        return Boolean.valueOf((String) getParameterValues().get(__PARAM_SQL_USER_SORT_PREVAIL_NAME));
    }
    
    @Override
    protected Map<String, Object> _getSearchParameters(Map<String, Object> parameters, int offset, int limit, List<Object> sort, List<String> columns)
    {
        // Add the sql column name for the orgunit id.
        // It's for setting the ametys-internal:orgunit-remote-id and retrieve easily the orgUnit content with this Id
        String orgUnitIdColumnName = getOrgUnitRemoteIdColumnName();
        if (!columns.contains(orgUnitIdColumnName))
        {
            columns.add(orgUnitIdColumnName);
        }
        
        return super._getSearchParameters(parameters, offset, limit, sort, columns);
    }
    
    @Override
    protected Map<String, Map<String, Object>> internalSearch(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger)
    {
        Map<String, Map<String, Object>> internalSearch = super.internalSearch(searchParameters, offset, limit, sort, logger);

        // Fill _orgUnitRemoteIds and _orgUnitParents maps with search results
        String orgUnitRemoteIdColumnName = getOrgUnitRemoteIdColumnName();
        for (String orgUnitIdValue : internalSearch.keySet())
        {
            Map<String, Object> orgUnitValues = internalSearch.get(orgUnitIdValue);
            _orgUnitRemoteIds.put(orgUnitIdValue, orgUnitValues.get(orgUnitRemoteIdColumnName).toString());
            
            Map<String, List<String>> mapping = getMapping();
            if (mapping.containsKey(OrganisationChartPageHandler.PARENT_ORGUNIT_ATTRIBUTE_NAME))
            {
                // We take the first because it's no sense to defined two sql columns to define orgunit parents
                String parentColumn = mapping.get(OrganisationChartPageHandler.PARENT_ORGUNIT_ATTRIBUTE_NAME).get(0);
                if (orgUnitValues.containsKey(parentColumn))
                {
                    String parentId = Optional.of(orgUnitValues)
                            .map(v -> v.get(parentColumn))
                            .map(Object::toString)
                            .orElse(null);
                    _orgUnitParents.put(orgUnitIdValue, parentId);
                }
            }
        }
        
        return internalSearch;
    }
    
    @Override
    protected void _logSynchronizationResult(Logger logger)
    {
        super._logSynchronizationResult(logger);
        logger.info("{} contents have been modified to update the parent-child relations.", _nbSynchronizedOrgUnit);
    }
    
    @Override
    protected boolean _hasSomethingChanged()
    {
        return super._hasSomethingChanged() || _nbSynchronizedOrgUnit > 0;
    }
    
    @Override
    protected List<ModifiableContent> _internalPopulate(Logger logger, ContainerProgressionTracker progressionTracker)
    {
        _orgUnitParents = new HashMap<>();
        _orgUnitRemoteIds = new HashMap<>();
        _usersByOrgUnitId = _getUsersByOrgUnit(logger);
        _nbSynchronizedOrgUnit = 0;
        
        List<ModifiableContent> contents = super._internalPopulate(logger, progressionTracker);
        
        // All orgunits are imported, now we can set relation with parent orgunit
        _setContentsRelationWithParentOrgunit(contents, logger);
        
        return contents;
    }
    
    /**
     * Retrieves the user attributes for all org units
     * @param logger The logger
     * @return the org units' users
     */
    protected Map<String, Map<String, String>> _getUsersByOrgUnit(Logger logger)
    {
        Map<String, Map<String, String>> orgUnitsUsers = new HashMap<>();

        Map<String, List<String>> mapping = getMapping();
        String idField = getIdField();
        List<String> orgUnitKeys = mapping.get(idField);
        if (orgUnitKeys != null && orgUnitKeys.size() > 0)
        {
            String orgUnitKey = orgUnitKeys.get(0);
            Map<String, Object> userParameters = _getSearchUserParameters(orgUnitKey, logger);
            List<Map<String, Object>> searchUserList = _sqlUserDAO.searchUser(userParameters, getDataSourceId());

            String loginColumnName = getLoginUserColumnName();
            String roleColumnName = getRoleUserColumnName();

            List<String> userColumns = new ArrayList<>();
            userColumns.add(loginColumnName);
            userColumns.add(orgUnitKey);
            if (StringUtils.isNotBlank(roleColumnName))
            {
                userColumns.add(roleColumnName);
            }

            for (Map<String, Object> userMap : searchUserList)
            {
                Map<String, Object> normalizedUserMap = _getNormalizedSearchResult(userColumns, userMap);
                Optional<Triple<String, String, String>> orgUnitUser = _getUserByOrgUnit(orgUnitKey, loginColumnName, roleColumnName, normalizedUserMap, logger);
                if (orgUnitUser.isPresent())
                {
                    Triple<String, String, String> triple = orgUnitUser.get();
                    String orgUnitId = triple.getLeft();
                    Map<String, String> orgUnitUsers = orgUnitsUsers.computeIfAbsent(orgUnitId, __ -> new LinkedHashMap<>());
                    orgUnitUsers.put(triple.getMiddle(), triple.getRight());
                }
            }
        }

        return orgUnitsUsers;
    }
    
    /**
     * Retrieves a {@link Triple} containing the orgunit id, and the user's login and role for the given normalized user
     * @param orgUnitKey the orgUnit key
     * @param loginColumnName the login column name
     * @param roleColumnName the role column name
     * @param normalizedUser the normalized user
     * @param logger the logger
     * @return the user info as a {@link Triple}
     */
    protected Optional<Triple<String, String, String>> _getUserByOrgUnit(String orgUnitKey, String loginColumnName, String roleColumnName, Map<String, Object> normalizedUser, Logger logger)
    {
        String loginValue = (normalizedUser.get(loginColumnName) == null) ? null : String.valueOf(normalizedUser.get(loginColumnName));
        String orgUnitIdValue = normalizedUser.get(orgUnitKey).toString();
        
        if (StringUtils.isNotBlank(loginValue))
        {
            String roleValue = null;
            if (StringUtils.isNotBlank(roleColumnName))
            {
                roleValue = (normalizedUser.get(roleColumnName) == null) ? null :  String.valueOf(normalizedUser.get(roleColumnName));
            }
            
            return Optional.of(Triple.of(orgUnitIdValue, loginValue, roleValue));
        }
        else
        {
            logger.warn("Can't add user to orgunit '" + orgUnitIdValue + "' because the login value is blank ...");
            return Optional.empty();
        }
    }
    
    @Override
    protected Optional<ModifiableContent> _importOrSynchronizeContent(String idValue, String lang, Map<String, List<Object>> remoteValues, boolean forceImport, Logger logger)
    {
        // Do not set relation with parent orgunit now, while they might miss some that are not yet created
        remoteValues.remove(OrganisationChartPageHandler.PARENT_ORGUNIT_ATTRIBUTE_NAME);
        
        return super._importOrSynchronizeContent(idValue, lang, remoteValues, forceImport, logger);
    }
    
    @Override
    public ContentSynchronizationResult additionalCommonOperations(ModifiableContent content, Map<String, Object> additionalParameters, Logger logger)
    {
        ContentSynchronizationResult result = super.additionalCommonOperations(content, additionalParameters, logger);
        
        try
        {
            SynchronizationResult additionalResult = new SynchronizationResult();
            String orgUnitIdValue = content.getValue(getIdField());
            ModifiableModelLessDataHolder internalDataHolder = content.getInternalDataHolder();
            
            String oldValue = internalDataHolder.getValueOfType(ORGUNIT_REMOTE_ID_INTERNAL_DATA, ModelItemTypeConstants.STRING_TYPE_ID);
            String value = _orgUnitRemoteIds.get(orgUnitIdValue);
            
            if (!value.equals(oldValue))
            {
                internalDataHolder.setValue(ORGUNIT_REMOTE_ID_INTERNAL_DATA, value);
                additionalResult.setHasChanged(true);
            }
            
            result.aggregateResult(additionalResult);
        }
        catch (Exception e)
        {
            _nbError++;
            logger.error("An error occurred while importing or synchronizing orgunit '{}' and setting its remote id.", content, e);
        }
        
        return result;
    }
    
    @Override
    protected Map<String, Object> getAdditionalAttributeValues(String idValue, Content content, Map<String, Object> additionalParameters, boolean create, Logger logger)
    {
        Map<String, Object> additionalRemoteValues = super.getAdditionalAttributeValues(idValue, content, additionalParameters, create, logger);
        
        List< ? extends RepeaterEntry> oldOrgUnitUsers = Optional.ofNullable(content.getRepeater("users"))
            .map(Repeater::getEntries)
            .orElseGet(() -> List.of());
        List<Map<String, Object>> newOrgUnitUsers = _getOrgUnitUsers(idValue, content.getLanguage());
        if (!oldOrgUnitUsers.isEmpty() || !newOrgUnitUsers.isEmpty())
        {
            // Add values only if there are old orgunit users or if there are new orgunit users
            additionalRemoteValues.put(OrganisationChartPageHandler.ORGUNIT_USERS_ATTRIBUTE_NAME, newOrgUnitUsers);
        }
        
        return additionalRemoteValues;
    }
    
    /**
     * Retrieves the users of the given orgunit
     * @param orgUnitId the orgunit identifier
     * @param lang the language of the orgunit
     * @return the users linked to the given orgunit
     */
    protected List<Map<String, Object>> _getOrgUnitUsers(String orgUnitId, String lang)
    {
        if (_usersByOrgUnitId.containsKey(orgUnitId))
        {
            List<Map<String, Object>> users = new ArrayList<>();
            for (Map.Entry<String, String> user : _usersByOrgUnitId.get(orgUnitId).entrySet())
            {
                Map<String, Object> orgUnitUser = new HashMap<>();
                
                String loginValue = user.getKey();
                Content userContent = _getUserContent(loginValue, lang);
                
                if (userContent != null)
                {
                    String roleValue = user.getValue();
                    orgUnitUser.put(OrganisationChartPageHandler.ORGUNIT_USER_ROLE_ATTRIBUTE_NAME, roleValue);
                    orgUnitUser.put(OrganisationChartPageHandler.ORGUNIT_USER_ATTRIBUTE_NAME, userContent);
                    
                    users.add(orgUnitUser);
                }
            }
            
            return users;
        }
        else
        {
            return List.of();
        }
    }
    
    /**
     * Set all orgunit parents relation for each synchronized content
     * @param orgUnitContents the synchronized content
     * @param logger the logger
     */
    protected void _setContentsRelationWithParentOrgunit(List<ModifiableContent> orgUnitContents, Logger logger)
    {
        for (ModifiableContent orgUnitContent : orgUnitContents)
        {
            try
            {
                if (orgUnitContent instanceof WorkflowAwareContent)
                {
                    I18nizableText commentText = new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_WORKFLOW_ACTION_EDIT_ORGUNIT_REFERENCE_MSG");
                    String comment = _i18nUtils.translate(commentText, orgUnitContent.getLanguage());

                    View view = View.of(orgUnitContent.getModel(), OrganisationChartPageHandler.PARENT_ORGUNIT_ATTRIBUTE_NAME);

                    Map<String, Object> values = new HashMap<>();
                    String orgUnitIdValue = orgUnitContent.getValue(getIdField());
                    String newParentSynchroId = _orgUnitParents.get(orgUnitIdValue);
                    Content newParentContent = newParentSynchroId != null ? _getOrgUnitContentFromRemoteId(newParentSynchroId, orgUnitContent.getLanguage(), logger) : null;
                    values.put(OrganisationChartPageHandler.PARENT_ORGUNIT_ATTRIBUTE_NAME, newParentContent);
                    
                    SynchronizationContext synchronizationContext = SynchronizationContext.newInstance()
                                                                                          .withStatus(ExternalizableDataStatus.EXTERNAL)
                                                                                          .withExternalizableDataContextEntry(SynchronizableContentsCollectionDataProvider.SCC_ID_CONTEXT_KEY, getId())
                                                                                          .withIncompatibleValuesIgnored(true);
                    
                    if (orgUnitContent.hasDifferences(view, values, synchronizationContext))
                    {
                        Map<String, Object> inputs = new HashMap<>();
                        if (StringUtils.isNotEmpty(comment))
                        {
                            inputs.put("comment", comment);
                        }
    
                        Map<String, Object> parameters = new HashMap<>();
    
                        parameters.put(EditContentFunction.VIEW, view);
                        parameters.put(EditContentFunction.VALUES_KEY, values);
                        parameters.put(EditContentFunction.QUIT, true);
                        inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, parameters);
    
                        inputs.put(EditSynchronizedContentFunction.SYNCHRO_INVERT_EDIT_ACTION_ID_KEY, getSynchronizeActionId());
    
                        Map<String, Object> actionResult = _contentWorkflowHelper.doAction((WorkflowAwareContent) orgUnitContent, getSynchronizeActionId(), inputs);
                        if ((boolean) actionResult.getOrDefault(AbstractContentWorkflowComponent.HAS_CHANGED_KEY, false))
                        {
                            _nbSynchronizedOrgUnit++;
                        }
                    }
                }
            }
            catch (WorkflowException | InvalidActionException e)
            {
                logger.error("An error occurred while updating parent-child relations of org unit {} ", orgUnitContent, e);
            }
        }
    }
    
    /**
     * Get user content from login value
     * @param loginValue the login value
     * @param lang the language
     * @return the user content
     */
    protected Content _getUserContent(String loginValue, String lang)
    {
        String loginAttribute = getLoginUserAttributeName();
        Set<String> contentTypes = _contentTypeEP.getSubTypes(UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE);
        
        Expression ctypeExpression = new ContentTypeExpression(Operator.EQ, contentTypes.toArray(new String[contentTypes.size()]));
        LanguageExpression languageExpression = new LanguageExpression(Operator.EQ, lang);
        StringExpression loginExp = new StringExpression(loginAttribute, Operator.EQ, loginValue);
        
        Expression userExp = new AndExpression(loginExp, ctypeExpression, languageExpression);
        String xPathQuery = ContentQueryHelper.getContentXPathQuery(userExp);
        
        AmetysObjectIterable<Content> contentQuery = _resolver.query(xPathQuery);
        AmetysObjectIterator<Content> contentIterator = contentQuery.iterator();
        if (contentIterator.hasNext())
        {
            return contentIterator.next();
        }
        
        return null;
    }
    
    /**
     * Get orgunit content from the remote Id
     * @param remoteId the remote Id
     * @param lang the language
     * @param logger the logger
     * @return the orgunit content
     */
    protected Content _getOrgUnitContentFromRemoteId(String remoteId, String lang, Logger logger)
    {
        Expression ctypeExpression = new ContentTypeExpression(Operator.EQ, getContentType());
        StringExpression remoteIdOrgunitExpression = new StringExpression(ORGUNIT_REMOTE_ID_INTERNAL_DATA, Operator.EQ, remoteId, ExpressionContext.newInstance().withInternal(true));
        LanguageExpression languageExpression = new LanguageExpression(Operator.EQ, lang);
        
        Expression collectionExpression = _sccHelper.getCollectionExpression(getId());
        
        Expression userExp = new AndExpression(remoteIdOrgunitExpression, ctypeExpression, languageExpression, collectionExpression);
        String xPathQuery = ContentQueryHelper.getContentXPathQuery(userExp);
        
        AmetysObjectIterable<Content> contentQuery = _resolver.query(xPathQuery);
        AmetysObjectIterator<Content> contentIterator = contentQuery.iterator();
        if (contentIterator.hasNext())
        {
            return contentIterator.next();
        }
        
        return null;
    }
    
    /**
     * Get the parameters map for user mybatis search
     * @param orgUnitColumnKey the column name of the orgunit key
     * @param logger the logger
     * @return the parameter map
     */
    protected Map<String, Object> _getSearchUserParameters(String orgUnitColumnKey, Logger logger)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("loginColumnName", getLoginUserColumnName());
        params.put("tableUser", getUserTableName());
        params.put("tableOrgUnit", getTableName());
        params.put("joinColumnName", getOrgunitJoinColumnNameForUser());
        params.put("orgUnitColumnKey", orgUnitColumnKey);
        params.put("orgUnitIdColumnName", getOrgUnitRemoteIdColumnName());
        
        String roleUserColumnName = getRoleUserColumnName();
        if (StringUtils.isNotBlank(roleUserColumnName))
        {
            params.put("roleColumnName", roleUserColumnName);
        }
        
        String sortColumnName = getUserSortColumnName();
        if (StringUtils.isNotBlank(sortColumnName))
        {
            params.put("sortColumnName", sortColumnName);
        }

        return params;
    }
    
    @Override
    protected int _deleteContents(List<Content> contentsToRemove, Logger logger)
    {
        return _deleteOrgUnitComponent.deleteContentsWithLog(contentsToRemove, Map.of(DeleteOrgUnitComponent.SCC_ID_PARAMETERS_KEY, getId()), Map.of(), logger);
    }
    
    @Override
    public Map<String, Integer> getSynchronizationResult()
    {
        Map<String, Integer> result = super.getSynchronizationResult();

        result.put(RESULT_NB_SYNCHRONIZED_ORGUNITS_RELATIONS, _nbSynchronizedOrgUnit);

        return result;
    }
}
