/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.synchronize;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.directory.UserDirectory;
import org.ametys.core.user.directory.UserDirectoryFactory;
import org.ametys.core.user.directory.UserDirectoryModel;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.core.impl.user.directory.JdbcUserDirectory;
import org.ametys.plugins.core.impl.user.directory.LdapUserDirectory;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for User Population Synchronizable Contents Collections.
 */
public class UserPopulationSynchronizableContentsCollectionHelper extends AbstractLogEnabled implements Component, Serviceable
{
    private UserPopulationDAO _userPopulationDAO;
    private UserDirectoryFactory _userDirectoryFactory;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _userPopulationDAO = (UserPopulationDAO) manager.lookup(UserPopulationDAO.ROLE);
        _userDirectoryFactory = (UserDirectoryFactory) manager.lookup(UserDirectoryFactory.ROLE);
    }
    
    /**
     * Gets the supported user directories (i.e. user directories based on a datasource) of the population in a json map
     * @param populationId The id of the user population
     * @return the supported user directories (i.e. user directories based on a datasource) of the population in a json map
     */
    @Callable (rights = "Runtime_Rights_Admin_Access", context = "/admin") // used by 'edition.populationMapping' widget
    public List<Map<String, Object>> getSupportedUserDirectories(String populationId)
    {
        List<Map<String, Object>> result = new ArrayList<>();
        
        UserPopulation userPopulation = _userPopulationDAO.getUserPopulation(populationId);
        List<UserDirectory> userDirectories = userPopulation.getUserDirectories();
        for (String udId : _getDatasourceBasedUserDirectories(userDirectories))
        {
            UserDirectory userDirectory = userPopulation.getUserDirectory(udId);
            String udModelId = userDirectory.getUserDirectoryModelId();
            UserDirectoryModel udModel = _userDirectoryFactory.getExtension(udModelId);
            Map<String, Object> udMap = new HashMap<>();
            
            udMap.put("id", udId);
            udMap.put("modelLabel", udModel.getLabel());
            
            String label = userDirectory.getLabel();
            if (StringUtils.isNotBlank(label))
            {
                udMap.put("label", label);
            }
            
            if (userDirectory instanceof JdbcUserDirectory)
            {
                udMap.put("type", "SQL");
            }
            else if (userDirectories instanceof LdapUserDirectory)
            {
                udMap.put("type", "LDAP");
            }
            
            result.add(udMap);
        }
        
        return result;
    }
    
    private List<String> _getDatasourceBasedUserDirectories(List<UserDirectory> userDirectories)
    {
        List<String> ids = new ArrayList<>();
        for (UserDirectory userDirectory : userDirectories)
        {
            if (userDirectory instanceof JdbcUserDirectory || userDirectory instanceof LdapUserDirectory)
            {
                ids.add(userDirectory.getId());
            }
        }
        
        return ids;
    }
}
