/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webanalytics.matomo.tracking;

import java.util.List;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.matomo.java.tracking.MatomoRequest;
import org.matomo.java.tracking.MatomoRequests;

import org.ametys.plugins.webanalytics.matomo.MatomoDataHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.site.Site;

/**
 * Abstract class for Matomo tracking provider. It provide only {#MatomoRequests.event}
 */
public abstract class AbstractMatomoEventTrackingProvider extends AbstractLogEnabled implements MatomoTrackingProvider, Configurable
{
    /** The events category */
    protected String _eventsCategory;
    
    /** The events category */
    protected String _eventsAction;
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _eventsCategory = configuration.getChild("category").getValue();
        _eventsAction = configuration.getChild("action").getValue();
    }
    
    public List<MatomoRequest> getRequests(Site site)
    {
        String matomoSiteId = site.getValue(MatomoDataHelper.MATOMO_SITE_ID_SITE_CONFIG);

        return getEvents(site).stream()
                .map(e -> _getRequest(site, matomoSiteId, e))
                .toList();
    }
    
    private MatomoRequest _getRequest(Site site, String matomoSiteId, MatomoEvent event)
    {
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("Sending request to Matomo: idsite={}&e_c={}&e_a={}&e_n={}&e_v={}", matomoSiteId, getEventsCategory(), getEventsAction(), event.name(), event.count());
        }

        return MatomoRequests.event(getEventsCategory(), getEventsAction(), event.name(), event.count() > 0 ? event.count() : null)
                    .siteId(Integer.valueOf(matomoSiteId))
                    .userId("--AmetysServer--") // Use a invented user to regroup all tracking server events
                    .actionUrl(site.getUrl() + "/tracking-url") // Dummy URL to have a valid request
                    .build();
    }
    
    /**
     * The events category
     * @return the events category
     */
    protected String getEventsCategory()
    {
        return _eventsCategory;
    }
    
    /**
     * The events action
     * @return the events action
     */
    protected String getEventsAction()
    {
        return _eventsAction;
    }
    
    /**
     * Get the list of event to send to matomo
     * @param site the site
     * @return the list of matomo events
     */
    abstract protected List<MatomoEvent> getEvents(Site site);
    
    /**
     * Record for a matomo event
     * @param name the name of the event
     * @param count the count value of the event
     */
    public record MatomoEvent(String name, double count) { /* empty */ }
}

