/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webcontentio.archive;

import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.xml.sax.ContentHandler;

import org.ametys.plugins.contentio.archive.DefaultPluginArchiver;
import org.ametys.plugins.contentio.archive.ImportReport;
import org.ametys.plugins.contentio.archive.Merger;
import org.ametys.plugins.contentio.archive.ResourcesArchiverHelper;
import org.ametys.plugins.contentio.archive.SystemViewHandler;
import org.ametys.plugins.contentio.archive.ZipEntryHelper;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * {@link SitePluginArchiver} handling the "web-explorer" part, holding all shared resources.
 */
public class WebExplorerArchiver extends DefaultPluginArchiver
{
    private static final String __WEB_EXPLORER_NODE_NAME = "web-explorer";
    private static final String __SHARED_RESOURCES_NODE_NAME = "shared-resources";
    
    private AmetysObjectResolver _resolver;
    private ResourcesArchiverHelper _resourcesArchiverHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _resourcesArchiverHelper = (ResourcesArchiverHelper) manager.lookup(ResourcesArchiverHelper.ROLE);
    }
    
    @Override
    public void export(String pluginName, Node node, ZipOutputStream zos, String prefix) throws IOException
    {
        super.export(pluginName, node, zos, prefix);
        
        try
        {
            if (node.hasNode(__SHARED_RESOURCES_NODE_NAME))
            {
                Node sharedResourcesNode = node.getNode(__SHARED_RESOURCES_NODE_NAME);
                NodeIterator it = sharedResourcesNode.getNodes();
                while (it.hasNext())
                {
                    Node resources = it.nextNode();
                    ResourceCollection rootResources = _resolver.resolve(resources, false);
                    _resourcesArchiverHelper.exportCollection(rootResources, zos, prefix + "/shared-resources/" + resources.getName() + "/");
                }
            }
        }
        catch (RepositoryException e)
        {
            throw new RuntimeException("Unable to archive plugin " + pluginName, e);
        }
    }
    
    @Override
    protected ContentHandler getSystemViewHandler(ContentHandler initialHandler)
    {
        return new SystemViewHandler(initialHandler, name -> List.of("ametys:contents", "shared-resources").contains(name), __ -> false);
    }
    
    @Override
    public ImportReport partialImport(String pluginName, Node allPluginsNode, Path zipPath, String zipPluginEntryPath, Merger merger) throws IOException
    {
        // ALWAYS remove web-explorer node, even if merger.deleteBeforePartialImport() is false because it MUST be re-created with the good UUID with the super method (JCR import)
        _removeWebExplorerNode(allPluginsNode);
        ImportReport importPluginReport = super.partialImport(pluginName, allPluginsNode, zipPath, zipPluginEntryPath, merger);
        
        Node webExplorerNode = _getWebExplorerNode(allPluginsNode);
        
        ImportReport importResourceReport = new ImportReport();
        String zipSharedResourcesEntryPath = zipPluginEntryPath + "/" + "shared-resources/";
        if (ZipEntryHelper.zipEntryFolderExists(zipPath, zipSharedResourcesEntryPath))
        {
            importResourceReport = _importSharedResources(webExplorerNode, zipPath, zipSharedResourcesEntryPath, merger);
        }
        
        return ImportReport.union(importPluginReport, importResourceReport);
    }
    
    private void _removeWebExplorerNode(Node allPluginsNode) throws IOException
    {
        try
        {
            if (allPluginsNode.hasNode(__WEB_EXPLORER_NODE_NAME))
            {
                allPluginsNode.getNode(__WEB_EXPLORER_NODE_NAME).remove();
                allPluginsNode.getSession().save();
            }
        }
        catch (RepositoryException e)
        {
            throw new IOException(e);
        }
    }
    
    private Node _getWebExplorerNode(Node allPluginsNode) throws IOException
    {
        try
        {
            Node webExplorerNode = allPluginsNode.getNode(__WEB_EXPLORER_NODE_NAME);
            return webExplorerNode;
        }
        catch (RepositoryException e)
        {
            throw new IOException("Unexpected state. The '" + __WEB_EXPLORER_NODE_NAME + "' node was not imported.", e);
        }
    }
    
    private ImportReport _importSharedResources(Node webExplorerNode, Path zipPath, String zipSharedResourcesEntryPath, Merger merger) throws IOException
    {
        List<ImportReport> reports = new ArrayList<>();
        
        Node sharedResourcesNode;
        try
        {
            sharedResourcesNode = webExplorerNode.hasNode(__SHARED_RESOURCES_NODE_NAME)
                    ? webExplorerNode.getNode(__SHARED_RESOURCES_NODE_NAME)
                    : webExplorerNode.addNode(__SHARED_RESOURCES_NODE_NAME, "ametys:unstructured");
        }
        catch (RepositoryException e)
        {
            throw new IOException(e);
        }
        
        DirectoryStream<Path> roots = ZipEntryHelper.children(
                zipPath, 
                Optional.of(zipSharedResourcesEntryPath), 
                Files::isDirectory);
        try (roots)
        {
            for (Path rootPath : roots)
            {
                Node parentOfRootResources = sharedResourcesNode;
                ImportReport importCollectionReport = _resourcesArchiverHelper.importCollection(rootPath.toString() + "/", parentOfRootResources, zipPath, merger);
                reports.add(importCollectionReport);
            }
        }
        
        return ImportReport.union(reports);
    }
}
