/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
const EMPTY_LONG = java.lang.Long.MIN_VALUE + 1;
const EMPTY_DOUBLE = java.lang.Double.NEGATIVE_INFINITY;
const EMPTY_CALENDAR_TIME_IN_MILLIS = -1855872000000;

const NAMESPACE_PREFIX = org.ametys.plugins.repository.RepositoryConstants.NAMESPACE_PREFIX;
const NAMESPACE_PREFIX_INTERNAL = org.ametys.plugins.repository.RepositoryConstants.NAMESPACE_PREFIX_INTERNAL;

const COMPOSITE_NODETYPE = org.ametys.plugins.repository.RepositoryConstants.COMPOSITE_NODETYPE;
const LONG_REPOSITORY_DATA_TYPE = org.ametys.plugins.repository.data.repositorydata.RepositoryData.LONG_REPOSITORY_DATA_TYPE;
const DOUBLE_REPOSITORY_DATA_TYPE = org.ametys.plugins.repository.data.repositorydata.RepositoryData.DOUBLE_REPOSITORY_DATA_TYPE;
const CALENDAR_REPOSITORY_DATA_TYPE = org.ametys.plugins.repository.data.repositorydata.RepositoryData.CALENDAR_REPOSITORY_DATA_TYPE;

const EMPTY_METADATA_SUFFIX = org.ametys.plugins.repository.data.type.RepositoryModelItemType.EMPTY_METADATA_SUFFIX;

// Projects and project members are model less data holders. Use the Reposiory data to look for empty data
// And also use the RepositoryData API to set the values as empty by setting the boolean internal property 
function __migrateAllEmptyDataOfRepositoryData(repositoryData)
{
    try
    {
        let hasChanged = false;
        repositoryData.getDataNames().forEach(
            dataName => {
                let dataType = repositoryData.getType(dataName);
                if (dataType == LONG_REPOSITORY_DATA_TYPE)
                {
                    hasChanged = __migrateEmptyDataOfRepositoryData(repositoryData, dataName, "long", repositoryData.getLong(dataName), EMPTY_LONG) || hasChanged;
                }
                else if (dataType == DOUBLE_REPOSITORY_DATA_TYPE)
                {
                    hasChanged = __migrateEmptyDataOfRepositoryData(repositoryData, dataName, "double", repositoryData.getDouble(dataName), EMPTY_DOUBLE) || hasChanged;
                }
                else if (dataType == CALENDAR_REPOSITORY_DATA_TYPE)
                {
                    hasChanged = __migrateEmptyDataOfRepositoryData(repositoryData, dataName, "datetime", repositoryData.getDate(dataName).getTimeInMillis(), EMPTY_CALENDAR_TIME_IN_MILLIS) || hasChanged;
                }
                else if (dataType == COMPOSITE_NODETYPE)
                {
                    let groupData = repositoryData.getRepositoryData(dataName);
                    hasChanged = __migrateAllEmptyDataOfRepositoryData(groupData) || hasChanged;
                }
            }
        );
        return hasChanged;
    }
    catch (e)
    {
        logger.error("Unable to migrate empty data of " + repositoryData, e);
        throw e;
    }
}

function __migrateEmptyDataOfRepositoryData(repositoryData, dataName, typeId, value, emptyValue)
{
    if (value == emptyValue)
    {
        repositoryData.removeValue(dataName);
        repositoryData.setValue(dataName + EMPTY_METADATA_SUFFIX, true, NAMESPACE_PREFIX_INTERNAL);
        if (logger.isDebugEnabled())
        {
            logger.debug("The empty " + typeId + " data named '" + dataName + "' on node '" + repositoryData + "' has been migrated");
        }
        return true;
    }
    else
    {
        return false;
    }
}

// PROJECTS
logger.info("Start migration of empty values for projects");
__migrationAllEmptyDataOfProjects(Repository.session);
__migrationAllEmptyDataOfProjects(Repository.liveSession);

function __migrationAllEmptyDataOfProjects(session)
{
    let sessionName = session.getWorkspace().getName();
    let projects = Repository.query("//element(*, ametys:project)", true, session);
    let totalProjects = projects.getSize();
    let handledProjects = 0;
    projects.forEach(
        project => {
            if(__migrateAllEmptyDataOfRepositoryData(project.getRepositoryData()))
            {
                logger.info("The empty data of project " + project + " have been migrated");
                project.saveChanges();
            }
            
            handledProjects++;
            if (handledProjects % 10 == 0)
            {
                logger.info("[" + handledProjects + "/" + totalProjects + "] handled projects in workspace " + sessionName);
            }
        }
    );
    logger.info(handledProjects + " handled projects in workspace " + sessionName);
}

// PROJECT MEMBERS
logger.info("Start migration of empty values for project members");
__migrationAllEmptyDataOfProjectMembers(Repository.session);
__migrationAllEmptyDataOfProjectMembers(Repository.liveSession);

function __migrationAllEmptyDataOfProjectMembers(session)
{
    let sessionName = session.getWorkspace().getName();
    let projectMembers = Repository.query("//element(*, ametys:project-member)", true, session);
    let totalProjectMembers = projectMembers.getSize();
    let handledProjectMembers = 0;
    projectMembers.forEach(
        projectMember => {
            if(__migrateAllEmptyDataOfRepositoryData(projectMember.getRepositoryData()))
            {
                logger.info("The empty data of project member " + projectMember + " have been migrated");
                projectMember.saveChanges();
            }
            
            handledProjectMembers++;
            if (handledProjectMembers % 10 == 0)
            {
                logger.info("[" + handledProjectMembers + "/" + totalProjectMembers + "] handled project members in workspace " + sessionName);
            }
        }
    );
    logger.info(handledProjectMembers + " handled project members in workspace " + sessionName);
}

logger.info("Migration of empty values in CMS is successfully terminated")
