/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


/**
 * Class to handle link insertion in inline editor
 * @private
 */
Ext.define('Ametys.plugins.workspaces.editor.Links', {
	override: 'Ametys.plugins.cms.editor.Links',

    //------------------------------------------------------//
    //                      WORKSPACE LINK                  //
    //------------------------------------------------------//
    
    /**
     * Initialize function to add a link handler for project resources
     * @param controller {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    initializeProjectResourceLinks: function(controller)
    {
        this.addLinkHandler('project-resource', Ext.create('Ametys.plugins.workspaces.editor.ProjectResourceLinkHandler', {}));
    },
    
    /**
     * Insert a link to a project resource.
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    insertProjectResourceLink: function(controller)
    {
        var editor = controller.getCurrentField().getEditor();
        this._insertProjectResourceLink(editor);
    },
    
    /**
     * Insert a link to a project resource.
     * @param {tinymce.Editor} editor The tinymce active editor
     * @private
     */
    _insertProjectResourceLink: function(editor)
    {
        var currentId = null;
        var node = editor.dom.getParent(editor.selection.getNode(), 'a');
        if (node != null && (node.getAttribute('data-ametys-type') == 'project-resource'))
        {
            currentId = node.getAttribute('data-ametys-href');
        }

        this._doInsertProjectResourceLink(currentId);
    },    

    /**
     * Open dialog to select the project resource.
     * @param {String} currentId The resource id to select when opening the dialog box
     * @private
     */
    _doInsertProjectResourceLink: function(currentId)
    {
    	Ametys.plugins.workspaces.project.helper.ChooseProjectResource.open({
            title: "{{i18n PLUGINS_WORKSPACES_EDITOR_LINK_RESSOURCE_LABEL}}",
            value: currentId,
            callback: Ext.bind(this._insertProjectResourceLinkCb, this)
        });
    },
    
    /**
     * Callback function called when user has selected the resource to link. Inserts the link to the current cursor position
     * @param {String} id Id of the resource
	 * @param {String} filename File name of the resource
	 * @param {String} filesize File length of the resource
	 * @param {String} viewHref The URL to view the resource file
	 * @param {String} downloadHref The URL to download the resource file
	 * @param {Object} actionResult The result of the upload action. Can be null
	 * @param {String} [type='explorer'] The type of the resource to insert.
     * @private
     */
    _insertProjectResourceLinkCb: function(id, filename, filesize, viewHref, downloadHref, actionResult, type)
    {
    	type = type || 'project-resource';
		
		if (id)
		{
		    // FIXME "tinyMCE.activeEditor" a better method is to use the field.getEditor()
			tinyMCE.activeEditor.execCommand('mceBeginUndoLevel');

			var node = tinyMCE.activeEditor.dom.getParent(tinyMCE.activeEditor.selection.getNode(), 'a');
			if (node == null && tinyMCE.activeEditor.selection.isCollapsed())
			{
				var text = "{{i18n PLUGINS_WORKSPACES_EDITOR_LINK_DOWNLOAD}} «" + filename + "» (" + Ext.util.Format.fileSize(filesize) + ")"; 
				tinyMCE.activeEditor.execCommand('mceInsertContent', false, "<a href='#'>" + text + "{$caret}</a>");

				node = tinyMCE.activeEditor.dom.getParent(tinyMCE.activeEditor.selection.getNode(), 'a');
				tinyMCE.activeEditor.selection.select(node);
			}

			tinyMCE.activeEditor.execCommand('mceInsertLink', false, { "data-ametys-href": id, href: downloadHref, "class": "download", "data-ametys-type": type, "_mce_ribbon_select" : "1" });
			tinyMCE.activeEditor.selection.collapse();

			tinyMCE.activeEditor.execCommand('mceEndUndoLevel');
		}

		// Delayed to wait for the dialog box to hide.
		window.setTimeout(function() {tinyMCE.activeEditor.focus();}, 100);
    }
});
