/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import i18n from 'i18n';
import AmetysFront from 'AmetysFront';
import { useProjectKeywordsStore } from '@/stores/projectKeywords';
import { useProjectFiltersStore } from '@/stores/projectFilters';

/**
 * Pinia store for projects
 */
export const useProjectsStore = defineStore('projects', {
    state: () => ({
        myProjects: [], // List of projects of the user
        pubProjects: [], // List of public projects the user can subscribe
        menuProjects: [], // List of projects to display in the menu
        categories: [], // List of categories of projects
        type: 'getMyProjects', // Type of projects to display, either 'getMyProjects' or 'getPubProjects'
        isLoaded: false, // Whether the projects are loaded
        canCreate: false, // Whether the user can create projects
        canCreatePrivateProject: false, // Whether the user can create private projects
        canCreatePublicProjectWithModeration: false, // Whether the user can create public projects with moderation
        canCreatePublicProject: false, // Whether the user can create public projects without moderation
        users: [] // List of users of the workspace
    }),
    getters: {
        
        /**
         * Get the projects filtered by the current filters
         * @returns {Array} the filtered projects
         */
        getProjects: (state) => {
            const projectFiltersStore = useProjectFiltersStore();
            const rootCategories = state.$rootCategories || {};
            const filters = { filterByName: projectFiltersStore.filterByName, filterByCategory: projectFiltersStore.filterByCategory };
            if (state.type == 'getMyProjects')
            {
                return state.myProjects.filter(project => {
                    return AmetysFront.Utils.deemphasize(project.title.toLowerCase()).indexOf(AmetysFront.Utils.deemphasize(filters.filterByName.toLowerCase())) !== -1
                        && (project.category ? rootCategories[project.category.name].name.indexOf(filters.filterByCategory) !== -1 : true)
                });
            }
            else
            {
                return state.pubProjects.filter(project => {
                    return AmetysFront.Utils.deemphasize(project.title.toLowerCase()).indexOf(AmetysFront.Utils.deemphasize(filters.filterByName.toLowerCase())) !== -1
                        && (project.category ? rootCategories[project.category.name].name.indexOf(filters.filterByCategory) !== -1 : true)
                }).sort(function(x, y){
                    if (x.title < y.title) return -1;
                    if (x.title > y.title) return 1;
                    return 0;
                });
            }
        }
    },
    actions: {

        /**
         * Set the type of projects to display
         * @param {String} type the type of projects to display, either 'getMyProjects' or 'getPubProjects'
         */
        setType(type)
        {
            this.type = type;
        },
        
        /**
         * Load projects of the user and the public projects he can subscribe.
         * @param {String} the zoneItemId of the catalog service, used to get allowed populations
         */
        async loadProjects(zoneitemId) {
            try
            {
                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.project.ProjectsCatalogueManager',
                    methodName: 'getUserAndPublicProjects',
                    parameters: [zoneitemId]
                });
                this.pubProjects = data.availablePublicProjects;
                this.myProjects = data.userProjects;
                this.isLoaded = true;
                this.setCanCreate(data);
                this.setCategories();
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_CATALOGUE_LOADPROJECT_FAIL,
                    text : i18n.PLUGINS_WORKSPACES_CATALOGUE_LOADPROJECT_FAIL_TEXT,
                    details: e
                });
            }
        },
        
        /**
         * 
         * Load the users of the workspace
         * @param {Object} options the options to load users
         * @param {String} options.sitename the site name
         * @param {Number} options.count the number of results to return
         * @param {Number} options.paginateOffset the offset to paginate results
         * @param {String} options.search the search string to filter users
         * @param {String} options.zoneitemId the zoneItemId of the catalog service, used to get allowed populations
         */
        async loadMembers(options)
        {
            try
            {
                let {users} = await callMethod({
                    role: 'org.ametys.plugins.workspaces.user.UserComponent',
                    methodName: 'getUsers',
                    parameters: [options.sitename, options.count, options.paginateOffset, options.search, options.zoneitemId]
                })
    
                users = users.map(user => {
                    return {
                        ...user,
                        text: 'asd'
                    }  
                })
    
                this.users = users;
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_CATALOGUE_LOADUSERS_FAIL,
                    text : i18n.PLUGINS_WORKSPACES_CATALOGUE_LOADUSERS_FAIL_TEXT,
                    details: e
                });
            }
        },

        /**
         * Creates a new project or edit an existing one
         * @param {Object} payload the payload to create or edit a project
         * @param {String} payload.zoneitemId the zoneItemId of the catalog service, used to get allowed populations
         * @param {String} payload.id the id of the project to edit, if not provided a new project will be created
         * @param {String} payload.title the title of the project
         * @param {String} payload.description the description of the project
         * @param {String} payload.illustration the illustration of the project
         * @param {Object} payload.category the category of the project
         * @param {String} payload.category.name the name of the category
         * @param {String} payload.category.color the color of the category
         * @param {Array} payload.keywords the keywords of the project
         * @param {Number} payload.visibility the visibility of the project, 0 for private, 1 for public with moderation, 2 for public without moderation          * @param {String} payload.defaultProfile the default profile of the project
         * @param {Array} payload.managers the managers of the project
         * @param {Array} payload.modules the modules of the project
         * @param {String} payload.language the language of the project, only used when creating a new project
         * @param {Boolean} payload.redirected if true, the user will be redirected to the project page after creation
         * 
         */
        async createOrEditProject(payload)
        {
            const projectKeywordsStore = useProjectKeywordsStore();
            AmetysFront.Event.fire('loaderStart', {
                text: payload.id ? i18n.PLUGINS_WORKSPACES_CATALOGUE_EDITPROJECT_RUNNING
                                : i18n.PLUGINS_WORKSPACES_CATALOGUE_NEWPROJECT_RUNNING
            });

            var parameters;

            if (payload.id) // edit
            {
                parameters = [
                    payload.zoneitemId,
                    payload.id,
                    payload.title,
                    payload.description,
                    payload.illustration,
                    payload.category,
                    payload.keywords,
                    payload.visibility,
                    payload.defaultProfile,
                    payload.managers,
                    payload.modules
                ];
            }
            else // create
            {
                parameters = [
                    payload.zoneitemId,
                    payload.title,
                    payload.description,
                    payload.illustration,
                    payload.category,
                    payload.keywords,
                    payload.visibility,
                    payload.defaultProfile,
                    payload.language,
                    payload.managers,
                    payload.modules
               ];
            }

            try
            {
                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.project.ProjectsCatalogueManager',
                    methodName: payload.id ? 'editProject': 'createProject',
                    parameters: parameters
                })
                if (result.success)
                {
                    if (payload.id)
                    {
                        let index = this.myProjects.findIndex(p => p.id == payload.id);
                       

                        // TODO verifier si on peut pas faire mieux
                        var project = this.myProjects[index];

                        for (let c in result.project)
                        {
                            if (result.project.hasOwnProperty(c))
                            {
                                project[c] = result.project[c];
                            }
                        }
                       
                        this.setCategories();
                        projectKeywordsStore.addKeywords(result.keywords);
                    }
                    else
                    {
                        if (result.project.managers.map(m => m.login + '#' + m.populationId).indexOf(window.ametysUser.login + '#' + window.ametysUser.populationId) == -1)
                        {
                            // I'm not part of the new project, public or not ?
                            if (result.project.visibility != 1)
                            {
                                this.pubProjects.push(result.project);
                            }
                            else
                            {
                                // A new project, I cannot see..
                            }
                        }
                        else
                        {
                            // My new project => adding, redirecting
                            this.myProjects.push(result.project);
                            window.location.href = result.project.url;
                            result.project.redirected = true;
                        }
                       
                        this.setCategories();
 
                        projectKeywordsStore.addKeywords(result.keywords);
                    }

                    AmetysFront.Event.fire('loaderEnd', {
                        text: payload.id ? i18n.PLUGINS_WORKSPACES_CATALOGUE_EDITPROJECT_DONE
                                         : (payload.redirected ? i18n.PLUGINS_WORKSPACES_CATALOGUE_NEWPROJECT_DONEREDIRECTION : i18n.PLUGINS_WORKSPACES_CATALOGUE_NEWPROJECT_DONE)
                    });
                }
                else  {
                    AmetysFront.Event.fire('loaderFail', {
                        title: (payload.id ? i18n.PLUGINS_WORKSPACES_CATALOGUE_EDITPROJECT_FAIL
                                             : i18n.PLUGINS_WORKSPACES_CATALOGUE_NEWPROJECT_FAIL ),
                        text : (payload.id ? i18n.PLUGINS_WORKSPACES_CATALOGUE_EDITPROJECT_FAIL_TEXT
                                             : i18n.PLUGINS_WORKSPACES_CATALOGUE_NEWPROJECT_FAIL_TEXT ),
                        details: null
                    });
                }
            }
            catch(e)
            {
                console.error(e);
                AmetysFront.Event.fire('loaderFail', {
                    title: (payload.id ? i18n.PLUGINS_WORKSPACES_CATALOGUE_EDITPROJECT_FAIL
                                         : i18n.PLUGINS_WORKSPACES_CATALOGUE_NEWPROJECT_FAIL ),
                    text : (payload.id ? i18n.PLUGINS_WORKSPACES_CATALOGUE_EDITPROJECT_FAIL_TEXT
                                         : i18n.PLUGINS_WORKSPACES_CATALOGUE_NEWPROJECT_FAIL_TEXT ),
                    details: null
                });
            }
        },
        /**
         * Delete a project
         * @param {String} id the id of the project to delete
         * @param {String} zoneItemId the zoneItemId of the catalog service, used to get allowed populations
         */
        async deleteProject(id, zoneItemId)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_CATALOGUE_DELETEPROJECT_RUNNING
            });

            try
            {
                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.project.ProjectsCatalogueManager',
                    methodName: 'deleteProject',
                    parameters: [zoneItemId, id]
                });

                if (result.success)
                {
                    this.myProjects = this.myProjects.filter(p => p.id != id);

                    AmetysFront.Event.fire('loaderEnd', {
                        text: i18n.PLUGINS_WORKSPACES_CATALOGUE_DELETEPROJECT_DONE
                    });
                }
                else
                {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_CATALOGUE_DELETEPROJECT_FAIL,
                        text: i18n.PLUGINS_WORKSPACES_CATALOGUE_DELETEPROJECT_FAIL_TEXT,
                        details: null
                    });
                }
            }
            catch(e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_CATALOGUE_DELETEPROJECT_FAIL,
                    text: i18n.PLUGINS_WORKSPACES_CATALOGUE_DELETEPROJECT_FAIL_TEXT,
                    details: null
                 });
            }
        },

        /**
         * Leave a project
         * @param {String} name the name of the project to leave
         * @param {String} id the id of the project to leave
         * @param {String} zoneItemId the zoneItemId of the catalog service, used to check rights
         */
        async leaveProject(name, id, zoneitemId) {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_CATALOGUE_LEAVE_PROJECT_RUNNING
            });

            try
            {
                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.members.ProjectMemberManager',
                    methodName: 'leaveProject',
                    parameters: [name]
                });

                if (result.success)
                {
                    let removedProject = this.myProjects.find(p => p.id == id);

                    this.myProjects = this.myProjects.filter(p => p.id != id);

                    // If the project is public, either with or without moderation, we should load it in public project with new values
                    if (removedProject.visibility != 1)
                    {
                        removedProject = await callMethod({
                            role: 'org.ametys.plugins.workspaces.project.ProjectsCatalogueManager',
                            methodName: 'getProjectByName',
                            parameters: [name, zoneitemId]
                        });
                        this.pubProjects.push(removedProject);
                    }
                    AmetysFront.Event.fire('loaderEnd', {
                        text: i18n.PLUGINS_WORKSPACES_CATALOGUE_LEAVE_PROJECT_DONE
                    });
                }
                else
                {
                    if (result["message"] == "unknown-project")
                    {
                        AmetysFront.Event.fire('loaderFail', {
                            title: i18n.PLUGINS_WORKSPACES_CATALOGUE_LEAVE_PROJECT_FAIL,
                            text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_PROJECT_ERROR_TEXT,
                            details: null
                        });
                    }
                    else if (result["message"] == "unknown-user")
                    {
                        AmetysFront.Event.fire('loaderFail', {
                            title: i18n.PLUGINS_WORKSPACES_CATALOGUE_LEAVE_PROJECT_FAIL,
                            text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_USER_ERROR_TEXT,
                            details: null
                        });
                    }
                    else if (result["message"] == "only-manager")
                    {
                        AmetysFront.Event.fire('loaderFail', {
                            title: i18n.PLUGINS_WORKSPACES_CATALOGUE_LEAVE_PROJECT_FAIL,
                            text : i18n.PLUGINS_WORKSPACES_LEAVE_PROJECT_ONLY_MANAGER_ERROR,
                            details: null
                        });
                    }
                    else if (result["message"] == "unknown-member")
                    {
                        AmetysFront.Event.fire('loaderFail', {
                            title: i18n.PLUGINS_WORKSPACES_CATALOGUE_LEAVE_PROJECT_FAIL,
                            text : i18n.PLUGINS_WORKSPACES_LEAVE_PROJECT_UNKNOWN_MEMBER_ERROR,
                            details: null
                        });
                    }
                    else
                    {
                      AmetysFront.Event.fire('loaderFail', {
                          title: i18n.PLUGINS_WORKSPACES_CATALOGUE_LEAVE_PROJECT_FAIL,
                          text: i18n.PLUGINS_WORKSPACES_CATALOGUE_LEAVE_PROJECT_FAIL_TEXT,
                          details: null
                      });
                    }
                }
            }
            catch(e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_CATALOGUE_LEAVE_PROJECT_FAIL,
                    text: i18n.PLUGINS_WORKSPACES_CATALOGUE_LEAVE_PROJECT_FAIL_TEXT,
                    details: null
                });
            }
        },

        /**
         * Change favorite 
         * @param {String} id the id of the project
         * @param {String} name the name of the project
         * @param {Boolean} favorite true to set as favorite, false to remove from favorites
         * 
         */
        async setFavorite(id, name, favorite)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETFAVORITEPROJECT_RUNNING
            });

            try
            {
                let success = await callMethod({
                   role: 'org.ametys.plugins.workspaces.project.favorites.FavoritesHelper',
                   methodName: 'setFavoriteProject',
                   parameters: [name, favorite]
                });

                if (success)
                {
                    let index = this.myProjects.findIndex(p => p.id == id);
                    this.myProjects[index].favorite = favorite;
                    
                    AmetysFront.Event.fire('loaderEnd', {
                      text: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETFAVORITEPROJECT_DONE
                    });
                }
                else
                {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETFAVORITEPROJECT_FAIL,
                        text: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETFAVORITEPROJECT_FAIL_TEXT,
                        details: null
                    });
                }
            }
            catch(e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETFAVORITEPROJECT_FAIL,
                    text: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETFAVORITEPROJECT_FAIL_TEXT,
                   details: null
                });
            }
        },

        /**
         * Change notification
         * @param {String} name the name of the project
         * @param {String} id the id of the project
         * @param {Boolean} notification true to set notification, false to remove notification
         */
        async setNotification(name, id, notification)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETNOTIFICATIONPROJECT_RUNNING
            });

            try
            {
                let success = await callMethod({
                    role: 'org.ametys.plugins.workspaces.project.notification.preferences.NotificationPreferencesHelper',
                    methodName: 'setPauseProject',
                    parameters: [name, !notification]
                });

                if (success)
                {
                   let index = this.myProjects.findIndex(p => p.id == id);
                   this.myProjects[index].notification = notification;
                   AmetysFront.Event.fire('loaderEnd', {
                       text: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETNOTIFICATIONPROJECT_DONE
                   });
                }
                else
                {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETNOTIFICATIONPROJECT_FAIL,
                        text: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETNOTIFICATIONPROJECT_FAIL_TEXT,
                        details: null
                    });
                }
            }
            catch(e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETNOTIFICATIONPROJECT_FAIL,
                    text: i18n.PLUGINS_WORKSPACES_CATALOGUE_SETNOTIFICATIONPROJECT_FAIL_TEXT,
                    details: null
                });
            }
        },
        
        // Setters

        /**
         * Set the canCreate properties from the data received from the server
         * @param {Object} data the data received from the server
         * @param {Boolean} data.canCreate whether the user can create projects
         * @param {Boolean} data.canCreatePrivateProject whether the user can create private projects
         * @param {Boolean} data.canCreatePublicProjectWithModeration whether the user can create public projects with moderation
         * @param {Boolean} data.canCreatePublicProject whether the user can create public projects
         */
        setCanCreate(data) {
            this.canCreate = data.canCreate;
            this.canCreatePrivateProject = data.canCreatePrivateProject;
            this.canCreatePublicProjectWithModeration = data.canCreatePublicProjectWithModeration;
            this.canCreatePublicProject = data.canCreatePublicProject;
        },
        
        /**
         * Set the list of used categories
         */
        setCategories() {
            
            function pushIfNotExist(categories, category)
            {
                if (category)
                {
                    for (let i = 0; i < categories.length; i++)
                    {
                        if (categories[i].name == category.name)
                        {
                            return;
                        }
                    }
                    categories.push(category);
                }
            }

            var rootCategories = this.$rootCategories;
            function _getRoot(category)
            {
                return category ? { color: "#000", ...rootCategories[category.name]} : null;
            }

            let categories = [];
            this.myProjects.forEach(p => pushIfNotExist(categories, _getRoot(p.category)));
            this.pubProjects.forEach(p => pushIfNotExist(categories, _getRoot(p.category)));
            categories.sort(function(c1,c2) {
                if (c1.title == c2.title) return 0;
                else if (c1.title < c2.title) return -1;
                else return 1;
            });

            this.categories = categories;
        },
        
    }
});
