/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront'
import i18n from 'i18n'
import { callMethod } from '@common/helper/ServerCommHelper'
import { useFilesStore } from '@/stores/files'
import { useFiltersStore } from '@/stores/filters'
import { useMoveFoldersAndFilesStore } from '@/stores/moveFoldersAndFiles'
import { useSelectedItemsStore } from '@/stores/selectedItems'
import { usePreviewItemStore } from '@/stores/previewItem'

/**
 * Recursive function to get a folder by its id from the folder tree.
 * @param {Object} folder the folder object to search in.
 * @param {string} id the id of the folder to find.
 */
function _getFolder(folder, id)
{
    if (folder.id == id)
    {
        return folder
    }
    if (folder.children)
    {
        for (let child of folder.children)
        {
            let childF = _getFolder(child, id)
            if (childF != null)
            {
                return childF
            }
        }
    }
    return null
}

function _recursivelyRemoveFolder(folder, idToRemove)
{
    if (folder.children)
    {
        let index = folder.children.findIndex(folder => folder.id == idToRemove);
        if (index != -1)
        {
            folder.children.splice(index, 1);
        }
        else
        {
            for (let f of folder.children)
            {
                _recursivelyRemoveFolder(f, idToRemove);
            }
        }
    }
}
/**
 * Pinia store for managing folders.
 */
export const useFoldersStore = defineStore('folders', {
    state: () => ({
        rootFolder: {}, // Root folder
        folders: [], // Folders of the current folder
        isFilter: false, // Is filter active
        currentFolder: {}, // Current opened folder
        treeOpenedFolders: [], // Opened folders in the tree
        treeActiveFolder: [] // Active folder in the tree
    }),
    getters: {
        
        /**
         * Get path for current opened folder
         */
        getPath: state =>
        {
            if (state.currentFolder == null || state.currentFolder == undefined || !state.currentFolder.hasOwnProperty('id'))
            {
                return false
            } 
            let paths = []
            let obj = [state.rootFolder]
            let pathKey = 0

            function collectPath(currentFolder)
            {
                if (!currentFolder.path?.length)
                {
                    return paths
                }
                obj.forEach(child =>
                {
                    if (child.id === currentFolder.path[pathKey])
                    {
                        paths.push(child)
                        pathKey++
                        if (pathKey < currentFolder.path.length)
                        {
                            obj = child.children
                            return obj != undefined ? collectPath(currentFolder) : []
                        }
                    }
                })
                return paths
            }
            return collectPath(state.currentFolder)
        },

        /**
         * Get folders of the current opened folder
         * @returns {Array}
         */
        getFolders()
        {
            const filtersStore = useFiltersStore()
            if (filtersStore.hideFolders)
            {
                return []
            }
            else
            {
                return this.folders.sort((f1, f2) =>
                    f1.name.toLowerCase() < f2.name.toLowerCase() ? -1 :
                    (f1.name.toLowerCase() > f2.name.toLowerCase() ? 1 : 0)
                )
            }
        }
    },
    actions: {
        /**
         * Reload the current folder (folders and files)
         */
        async reload()
        {
            const filesStore = useFilesStore()
            try
            {
                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'getFoldersAndFiles',
                    parameters: [this.currentFolder.id]
                })

                this.currentFolder.children = data.children
                this.folders = data.children
                filesStore.setFiles(data.files)
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Load root folder
         */
        async loadRootFolder()
        {
            const filtersStore = useFiltersStore();
            const filesStore = useFilesStore();
            const selectedItemsStore = useSelectedItemsStore();
            const previewItemStore = usePreviewItemStore();
            
            filtersStore.loading = true;

            try
            {
                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'getFoldersAndFiles',
                    parameters: [null]
                })

                data.root = true

                this.rootFolder = data
                this.currentFolder = data
                this.folders = data.children

                filesStore.setFiles(data.files)
                filtersStore.loading = false
                filtersStore.resetFilters(false)

                selectedItemsStore.unselectAll()
                
                previewItemStore.setItem({item: data, notOpen: true})

                if (!data.root)
                {
                    AmetysFront.Event.fire('hideMobileHeader')
                }
                else
                {
                    AmetysFront.Event.fire('showMobileHeader', true)
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Expand folder in tree
         * @param {Object} folderObj the folder object
         */
        async expandFolder(folderObj)
        {
            try
            {
                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'getFoldersAndFiles',
                    parameters: [folderObj.id]
                })

                folderObj.children = data.children.length
                    ? data.children.sort((f1, f2) =>
                        f1.name.toLowerCase() < f2.name.toLowerCase() ? -1 :
                        (f1.name.toLowerCase() > f2.name.toLowerCase() ? 1 : 0)
                    )
                    : undefined
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Open folder from breadcrumb
         * @param {Object} folderObj the folder object
         */
        async breadOpenFolder(folderObj)
        {
            const filesStore = useFilesStore()
            const filtersStore = useFiltersStore()
            const previewItemStore = usePreviewItemStore()
            
            try
            {
                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'getFoldersAndFiles',
                    parameters: [folderObj.id]
                })

                filtersStore.loading = true

                this.currentFolder = folderObj
                this.folders = folderObj.children ? folderObj.children : []
                this.addTreeOpenedFolder(folderObj)
                this.setTreeActiveFolder(folderObj)
                filesStore.setFiles(data.files)

                filtersStore.loading = false
                filtersStore.resetFilters(false)

                previewItemStore.setItem({item: folderObj, notOpen: true})
                
                if (folderObj.root)
                {
                    AmetysFront.Event.fire('showMobileHeader', true)
                }
                else
                {
                    AmetysFront.Event.fire('hideMobileHeader')
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Open the parent folder of the given file
         * @param {string} fileId the file id
         */
        async openFileParentFolder(fileId)
        {
            const filesStore = useFilesStore();
            const filtersStore = useFiltersStore();
            try
            {
                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'getFileParentInfo',
                    parameters: [fileId]
                })

                if (!data.error)
                {
                    filtersStore.loading = true

                    this.currentFolder = data
                    this.folders = data.children ? data.children : []
                    this.addTreeOpenedFolder(data)
                    this.setTreeActiveFolder(data)
                    filesStore.setFiles(data.files)

                    filtersStore.loading = false
                    filtersStore.resetFilters(false)

                    if (data.root)
                    {
                        AmetysFront.Event.fire('showMobileHeader', true)
                    }
                    else
                    {
                        AmetysFront.Event.fire('hideMobileHeader')
                    }
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Open folder
         * @param {Object} folderObj the folder object
         */
        async openFolder(folderObj)
        {
            const filtersStore = useFiltersStore()
            const filesStore = useFilesStore()
            const selectedItemsStore = useSelectedItemsStore()
            const previewItemStore = usePreviewItemStore()
            
            filtersStore.loading = true
            try
            {
                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'getFoldersAndFiles',
                    parameters: [folderObj.id]
                })

                if (data.hasOwnProperty('children'))
                {
                    folderObj.children = data.children
                }
                else
                {
                    delete folderObj.children
                }

                this.currentFolder = folderObj
                this.folders = folderObj.children ? folderObj.children : []
                this.addTreeOpenedFolder(folderObj)
                this.setTreeActiveFolder(folderObj)
                filesStore.setFiles(data.files)

                filtersStore.loading = false
                filtersStore.resetFilters(false)
                
                selectedItemsStore.unselectAll()
                previewItemStore.setItem({item: folderObj, notOpen: true})

                AmetysFront.Event.fire('reset-filter')
                AmetysFront.Event.fire('hideMobileHeader')
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })

                filtersStore.loading = false
                filtersStore.resetFilters(false)
                AmetysFront.Event.fire('reset-filter')
            }
        },

        /**
         * Create a new folder
         * @param {Object} payload the payload object
         * @param {Object} payload.folder he parent folder object
         * @param {string} payload.name the name of the new folder
         * @param {string} payload.description the description of the new folder
         */
        async createFolder({ folder: parent, name, description })
        {
            const moveFoldersAndFilesStore = useMoveFoldersAndFilesStore();
            let parentFolder = parent ? parent : this.currentFolder
            try
            {
                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'addFolder',
                    parameters: [parentFolder.id, name, description]
                })

                if (result.error)
                {
                    let text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR
                    if (result.message == 'locked')
                    {
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_LOCKED
                    }
                    else if (result.message == 'already-exist')
                    {
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_ALREADY_EXIST
                    }

                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_ADD,
                        text: text,
                        details: null
                    })
                }
                else
                {
                    // If the folder is created in the current folder, refresh the current folder values
                    if (parentFolder.id == this.currentFolder.id)
                    {
                        if (this.currentFolder.hasOwnProperty('children'))
                        {
                            this.currentFolder.children.push(result)
                        }
                        else
                        {
                            this.currentFolder.children = [result]
                        }
                        this.currentFolder.hasChildren = true
                        this.folders = this.currentFolder.children
                        
                        
                    }
                    
                    // Sometimes the parent folder is not the  same javascript object as the current folder, 
                    // so we need to also update the related folder from the rootFolder
                    let folderInRoot = _getFolder(this.rootFolder, parentFolder.id)
                    folderInRoot.children = this.currentFolder.children;
                    folderInRoot.hasChildren = true;
                    
                    // Opeen the folder in the tree
                    this.addTreeOpenedFolder(parentFolder)
                    
                    // Apply the changes to the moveFoldersAndFilesStore
                    moveFoldersAndFilesStore.createFolder(result);
                    moveFoldersAndFilesStore.treeOpenedFolders.push(parentFolder);
                    moveFoldersAndFilesStore.currentFolder = result;
                    AmetysFront.Event.fire('rerender-file-tree')
                    
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_ADD,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Delete a folder
         * @param {Object} payload the payload object
         * @param {string} payload.id the id of the folder to delete
         * @param {string} payload.parentId the id of the parent folder
         */
        async deleteFolder(payload)
        {
            const filesStore = useFilesStore()
            const previewItemStore = usePreviewItemStore()
            
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETING
            })

            if (this.currentFolder.children?.length === 1)
            {
                this.treeOpenedFolders = this.treeOpenedFolders.filter(f => f.id !== this.currentFolder.id)
            }

            try
            {
                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'deleteFolder',
                    parameters: [payload.id]
                })

                if (!result.error)
                {
                    this._deleteFolder(payload.id)

                    let data = await callMethod({
                        role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                        methodName: 'getFoldersAndFiles',
                        parameters: [payload.parentId]
                    })

                    if (this.currentFolder.id == payload.id)
                    {
                        filesStore.setFiles(data.files)
                        this.currentFolder = data
                        this.setTreeActiveFolder(data)
                        this.folders = data.children

                        // Parent is root, we need to update the rootFolder
                        if (data.root)
                        {
                            this.rootFolder = data;
                        }
                        else
                        {
                            // Remove the deleted folder from rootFolder
                            let folder = this.rootFolder;
                            _recursivelyRemoveFolder(folder, payload.id);
                        
                            // remove the deleted folder from the opened folders and update parent folder
                            let indexOpenFolder = this.treeOpenedFolders.findIndex(folder => folder.id == payload.parentId);
                            if (indexOpenFolder != -1)
                            {
                                this.treeOpenedFolders.splice(indexOpenFolder, 1);
                                let parentFolder = _getFolder(this.rootFolder, payload.parentId);
                                if (parentFolder && parentFolder.children.length == 0)
                                {
                                    parentFolder.hasChildren = false;
                                }
                            }
                        }
                    }
                    else if (!this.currentFolder.children.length)
                    {
                        this.currentFolder.hasChildren = false
                    }

                    previewItemStore.setItem({ item: this.currentFolder, notOpen: true})
                    
                    setTimeout(() =>
                    {
                        AmetysFront.Event.fire('loaderEnd', {
                            text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETE_FOLDER_SUCCESS
                        })
                    }, 500)
                }
                else
                {
                    let errorMsg = result.message
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_DELETE,
                        text: errorMsg == 'locked'
                            ? i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_LOCKED
                            : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                        details: null
                    })
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_DELETE,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }

        },

        /**
         * Rename a folder
         * @param {Object} payload the payload object
         * @param {Object} payload.folder the folder object to rename
         * @param {string} payload.name the new name of the folder
         * @return {boolean} true if the folder was renamed successfully, false otherwise
         */
        async renameFolder(payload)
        {
            let folderObj = payload.folder
            let success = false

            try
            {
                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'renameFolder',
                    parameters: [folderObj.id, payload.name]
                })

                if (!result.success)
                {
                    let text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR
                    if (result.message == 'locked')
                    {
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_LOCKED
                    }
                    else if (result.message == 'already-exist')
                    {
                        text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_ALREADY_EXIST
                    }

                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_RENAME,
                        text: text,
                        details: null
                    })
                }
                else
                {
                    success = true
                    folderObj.name = payload.name
                }
                return success
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_RENAME,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
            return success
        },

        /**
         * Edit a folder
         * @param {Object} payload the payload object
         * @param {Object} payload.folder the folder object to edit
         * @param {string} payload.name the new name of the folder
         * @param {string} payload.description the new description of the folder
         * @returns {boolean} true if the folder was updated successfully, false otherwise
         */
        async updateFolder(payload)
        {
            let folderObj = payload.folder
            let success = false

            try
            {
                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'editFolder',
                    parameters: [folderObj.id, payload.name, payload.description]
                })

                if (!result.success)
                {
                    let text = result.message == 'locked'
                        ? i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_LOCKED
                        : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR

                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_UPDATE,
                        text: text,
                        details: null
                    })
                }
                else
                {
                    success = true
                    if (payload.name)
                    {
                        folderObj.name = payload.name
                    }
                    if (payload.description != undefined)
                    {
                        folderObj.description = payload.description
                    }
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_UPDATE,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
            return success
        },

        /**
         * Set tree opened folders
         * @param {Array} foldersArr
         */
        setTreeOpenedFolders(foldersArr)
        {
            this.treeOpenedFolders = foldersArr
        },

        /**
         * Add a folder to tree opened folders
         * @param {Object} folderObj
         */
        addTreeOpenedFolder(folderObj)
        {
            let index = this.treeOpenedFolders.findIndex(f => f.id == folderObj.id)
            if (index == -1)
            {
                this.treeOpenedFolders.push(folderObj)
            }
            else
            {
                this.treeOpenedFolders.splice(index, 1, folderObj)
            }
        },

        /**
         * Set the active folder in the tree
         * @param {Object} folderObj
         */
        setTreeActiveFolder(folderObj)
        {
            if (folderObj)
            {
                this.treeActiveFolder = [folderObj]
            }
            else
            {
                this.treeActiveFolder = []
            }
        },
        
        /**
         * Update folders after moving them to another folder
         * @param {Array} selectedFoldersIds the ids of the folders to move
         * @param {string} toFolderId the id of the folder to move the selected folders to
         */
        updateFoldersAfterMove(selectedFoldersIds, toFolderId)
        {
            function getMovedFolders(folders, selectedFoldersIds)
            {
                let movedFolders = folders.filter((f) => selectedFoldersIds.includes(f.id));
                for (let f of folders)
                {
                    if (f.children && f.children.length)
                    {
                        for (let childF of getMovedFolders(f.children, selectedFoldersIds))
                        {
                            movedFolders.push(childF);
                        }
                    }
                }
              
                return movedFolders;
            }
          
            let toFolder = _getFolder(this.rootFolder, toFolderId);
            let path = toFolder.path;
          
          
            let oldParents = [];
          
            let movedFolders = getMovedFolders(this.rootFolder.children, selectedFoldersIds);
            for (let f of movedFolders)
            {
                f.path = path;
                oldParents.push(f.parentId);
                f.parentId = toFolderId;
            }
          
            /* Remove Moved Folders */
            function removedFolders(data, ids)
            {
                const items = data.filter((d) => !ids.includes(d.id));
                if (items.length > 0) 
                {
                    items.forEach((item) =>
                    { 
                        if (item.children != undefined) item.children = removedFolders(item.children, ids);
                        if (item.id == toFolderId)
                        {
                            if (item.children == undefined) 
                            {
                                item.children = [];
                            }
                  
                            for (let movedF of movedFolders)
                            {
                                item.children.push(movedF);
                            }
                        }
                    });
                }
                return items.length > 0 ? items : undefined;
            }
          
            this.rootFolder.children = removedFolders(this.rootFolder.children, selectedFoldersIds).sort(function(f1, f2)
            {
                return f1.name.toLowerCase() < f2.name.toLowerCase() ? -1 : (f1.name.toLowerCase() > f2.name.toLowerCase() ? 1 : 0);
            });
            
            if (this.rootFolder.id == toFolderId)
            {
                for (let movedF of movedFolders)
                {
                    this.rootFolder.children.push(movedF);
                    this.rootFolder.hasChildren = true;
                }
                this.rootFolder.children = this.rootFolder.children.sort(function(f1, f2) {
                    return f1.name.toLowerCase() < f2.name.toLowerCase() ? -1 : (f1.name.toLowerCase() > f2.name.toLowerCase() ? 1 : 0);
                });
            }
          
            if(this.currentFolder.id == toFolderId){
            
                this.openFolder(this.currentFolder);
            }
          
            let newFolders = removedFolders(this.folders, selectedFoldersIds);

            this.folders = newFolders != undefined ? newFolders : [];

            for (let parentId of oldParents)
            {
                let parentFolder = _getFolder(this.rootFolder, parentId);
                if (parentFolder)
                {
                    if (parentFolder.children == undefined) 
                    {
                        parentFolder.children = [];
                    }
                    parentFolder.hasChildren = parentFolder.children.length > 0;
                }
            }
            toFolder.hasChildren = true;
          
            if (this.treeActiveFolder)
            {
                let activeFolder = removedFolders(this.treeActiveFolder, selectedFoldersIds);
                this.treeActiveFolder = activeFolder != undefined ? newFolders : [];
            }
            let treeOpenedFolders = removedFolders(this.treeOpenedFolders, selectedFoldersIds);
            this.treeOpenedFolders = treeOpenedFolders != undefined ? treeOpenedFolders : [];
          
            let currentF = movedFolders.filter((f) => f.id == this.currentFolder.id);
            if (currentF.length > 0)
            {
                this.currentFolder = currentF[0];
            }

            AmetysFront.Event.fire('rerender-file-tree');
        },
        

        /**
         * Delete a folder from the store
         * @param {Object} payload the payload object
         * @param {string} payload.folderId the id of the folder to delete
         * @param {string} payload.parentId the id of the parent folder
         */
        _deleteFolder(folderId)
        {
            let index = this.folders.findIndex(folder => folder.id == folderId);
            if (index != -1)
            {
                let deletedFolders = this.folders.splice(index, 1);
                for (let deletedFolder of deletedFolders)
                {
                    let parentId = deletedFolder.parentId;
                    let parentFolder = _getFolder(this.rootFolder, parentId);
                    if (parentFolder.children == undefined) 
                    {
                        parentFolder.children = [];
                    }
                    if (parentFolder.children.length == 0)
                    {
                        parentFolder.hasChildren = false;
                    }
                }
            }
            let indexOpenFolder = this.treeOpenedFolders.findIndex(folder => folder.id == folderId);
            if (indexOpenFolder != -1)
            {
                this.treeOpenedFolders.splice(indexOpenFolder, 1);
            }
            
            if (this.treeActiveFolder.id == folderId)
            {
                this.treeActiveFolder = [];
            }
        },
    }
})
