/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'

/**
 * Pinia store for task filters
 */
export const useTasksFiltersStore = defineStore('tasksFilters', {
    state: () => ({
        filterByName: '', // search by name filter
        filterByType: {key: '', text: ''}, // search by type filter
        filterByMy: false // search by current user author filter
    }),
    actions:
    {
        /**
         * reset all filters
         */
        resetFilters() {
            this.filterByName = '';
            this.filterByType = {key: '', text: ''};
            this.filterByMy = false;
        },
        
        /**
         * set search by name filter
         * @param {String} name  the name to filter by
         */
        setByName(name) {
            this.filterByName = name;
        },
        
        /**
         * set search by type filter
         * @param {String} type  the type to filter by
         */
        setByType(type) {
            this.filterByType = type;
        },
        
        /**
         * set search by mine filter
         * @param {String} byMy true to only display tasks of the current user
         */
        setByMy(byMy) {
            this.filterByMy = byMy;
        },
    },
    getters:
    {
        /**
         * Check if at least one filter is set
         * @returns {Boolean} true if at least one filter is set
         */
        isFilter: state => 
        {
            return state.filterByType.key !== '' || state.filterByMy || state.filterByName.length > 0
        }
    }
})
